package ru.yandex.autotests.directapi.finance.transfermoney;

import java.math.BigDecimal;
import java.math.RoundingMode;

import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.TransferMoneyInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.equalTo;


/**
 * Author pavryabov
 * Date 27.04.14
 */
@Aqua.Test
@Features(FinanceFeatures.TRANSFER_MONEY)
public class TransferMoneyLowLimitTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String clientLogin = Logins.SUBCLIENT_RUB;
    private static final Money MIN_AMOUNT = MoneyCurrency.get(Currency.RUB).getMinTransferAmount().addVAT();

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(Logins.AGENCY_RUB);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static long activeCampaignID1;
    private static long activeCampaignID2;
    private static long activeCampaignID3;
    private static long activeCampaignID4;

    @BeforeClass
    public static void createCampaigns() {
        activeCampaignID1 = api.userSteps.addActiveCampaign(clientLogin, MIN_AMOUNT.multiply(100).floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(clientLogin, activeCampaignID1));
        activeCampaignID2 = api.userSteps.addActiveCampaign(clientLogin, MIN_AMOUNT.multiply(100).floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(clientLogin, activeCampaignID2));
        activeCampaignID3 = api.userSteps.addActiveCampaign(clientLogin, MIN_AMOUNT.multiply(100).floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(clientLogin, activeCampaignID3));
        activeCampaignID4 = api.userSteps.addActiveCampaign(clientLogin, MIN_AMOUNT.multiply(100).floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(clientLogin, activeCampaignID4));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6567")
    public void validRoundedToSumFromOneToSeveral() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignToElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        float sumFrom = 3000.49f;
        float sumTo1 = 1500.416f;
        float sumTo2 = 1500.074f;
        api.userSteps.financeSteps().transferMoney(new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(sumFrom))
                .withToCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID2)
                                .withCurrency(Currency.RUB)
                                .withSum(sumTo1),
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID3)
                                .withCurrency(Currency.RUB)
                                .withSum(sumTo2)));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignFrom,
                        Money.valueOf(sumFrom).setScale(2, RoundingMode.HALF_UP).negate()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignTo,
                        Money.valueOf(sumTo1).setScale(2, RoundingMode.HALF_UP)));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignToElse,
                        Money.valueOf(sumTo2).setScale(2, RoundingMode.HALF_UP)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6568")
    public void invalidRoundedSumFromOneToSeveral() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignToElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        float sumFrom = 2050.50f;
        float sumTo1 = 1025.416f;
        float sumTo2 = 1025.074f;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo1),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID3)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo2)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_AMOUNT_NOT_EQUAL_RUS,
                        Money.valueOf(sumFrom).setScale(RoundingMode.HALF_UP).bigDecimalValue().toString(),
                        Money.valueOf(sumTo1).setScale(RoundingMode.HALF_UP).add(
                                Money.valueOf(sumTo2).setScale(RoundingMode.HALF_UP)).bigDecimalValue().toString()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignToElse),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6569")
    public void validRoundedFromSumFromOneToSeveral() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignToElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        float sumFrom = 3000.599f;
        float sumTo1 = 1500.556f;
        float sumTo2 = 1500.044f;
        api.userSteps.financeSteps().transferMoney(new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(sumFrom))
                .withToCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID2)
                                .withCurrency(Currency.RUB)
                                .withSum(sumTo1),
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID3)
                                .withCurrency(Currency.RUB)
                                .withSum(sumTo2)));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignFrom,
                        Money.valueOf(sumFrom).setScale(2, RoundingMode.HALF_UP).negate()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignTo,
                        Money.valueOf(sumTo1).setScale(2, RoundingMode.HALF_UP)));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignToElse,
                        Money.valueOf(sumTo2).setScale(2, RoundingMode.HALF_UP)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6570")
    public void validRoundedSumAmount4digits() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignToElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        float sumFrom = 3000.599f;
        float sumTo1 = 1500.5535f;
        float sumTo2 = 1500.0455f;
        api.userSteps.financeSteps().transferMoney(new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(sumFrom))
                .withToCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID2)
                                .withCurrency(Currency.RUB)
                                .withSum(sumTo1),
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID3)
                                .withCurrency(Currency.RUB)
                                .withSum(sumTo2)));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignFrom,
                        Money.valueOf(sumFrom).setScale(RoundingMode.HALF_UP).negate()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignTo,
                        Money.valueOf(sumTo1).setScale(RoundingMode.HALF_UP)));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignToElse,
                        Money.valueOf(sumTo2).setScale(RoundingMode.HALF_UP)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6571")
    public void invalidTransferBetweenSeveralCampaigns() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignFromElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);
        CampaignGetItem campaignToElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID4);

        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(2000.556f),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(2000.044f))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID3)
                                        .withCurrency(Currency.RUB)
                                        .withSum(2000.550f),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID4)
                                        .withCurrency(Currency.RUB)
                                        .withSum(2000.049f)),
                new AxisError(353,
                        AxisErrorDetails.FUNDS_CANNOT_BE_TRANSFERRED_TO_FROM_MORE_THAN_ONE_CAMPAIGN_AT_A_TIME));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFromElse),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignToElse),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6572")
    public void validTransferFromSeveralToOne() {
        CampaignGetItem campaignFrom1 =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignFrom2 =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignFrom3 =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID4);

        float sumFrom1 = 1500.550f;
        float sumFrom2 = 1500.556f;
        float sumFrom3 = 1500.044f;
        float sumTo = 4501.149f;
        api.userSteps.financeSteps().transferMoney(new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(sumFrom1),
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID2)
                                .withCurrency(Currency.RUB)
                                .withSum(sumFrom2),
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID3)
                                .withCurrency(Currency.RUB)
                                .withSum(sumFrom3))
                .withToCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID4)
                                .withCurrency(Currency.RUB)
                                .withSum(sumTo)));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignFrom1,
                        Money.valueOf(sumFrom1).setScale(2, RoundingMode.HALF_UP).negate()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignFrom2,
                        Money.valueOf(sumFrom2).setScale(2, RoundingMode.HALF_UP).negate()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignFrom3,
                        Money.valueOf(sumFrom3).setScale(2, RoundingMode.HALF_UP).negate()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(clientLogin, campaignTo,
                        Money.valueOf(sumTo).setScale(2, RoundingMode.HALF_UP)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6573")
    public void transferFromOneToManyWithLowAmount() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignToElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        MoneyCurrency rubCurrency = MoneyCurrency.get(Currency.RUB);

        float sumFrom = 501.97f;
        float sumTo1 = 200.99f;
        float sumTo2 = 300.98f;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo1),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID3)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo2)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_LOW_AMOUNT_TO_WITH_PLACEHOLDERS,
                        activeCampaignID2,
                        rubCurrency.getMinTransferAmount().stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        rubCurrency.getAPIAbbreviation("en")));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignToElse),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6574")
    public void transferFromManyWithLowAmountToOne() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignFromElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        MoneyCurrency rubCurrency = MoneyCurrency.get(Currency.RUB);

        float sumFrom1 = 300.98f;
        float sumFrom2 = 200.99f;
        float sumTo = 501.97f;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom1),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom2))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID3)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_LOW_AMOUNT_WITH_PLACEHOLDERS,
                        rubCurrency.getMinTransferAmount().stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        rubCurrency.getAPIAbbreviation("en")));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFromElse),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6575")
    public void transferFromOneToManyWithNegativeAmount() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignToElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        float sumFrom = 200.01f;
        float sumTo1 = 500.99f;
        float sumTo2 = -300.98f;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo1),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID3)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo2)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_SUM_NEGATIVE));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignToElse),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6576")
    public void transferFromManyWithNegativeAmountToOne() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignFromElse =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID3);

        float sumFrom1 = 500.99f;
        float sumFrom2 = -300.98f;
        float sumTo = 200.01f;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom1),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom2))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID3)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_SUM_NEGATIVE));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFromElse),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6577")
    public void transferFromOneToOneLowAmount() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);

        MoneyCurrency rubCurrency = MoneyCurrency.get(Currency.RUB);

        float sumFrom = 299.01f;
        float sumTo = 299.01f;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_LOW_AMOUNT_WITH_PLACEHOLDERS,
                        rubCurrency.getMinTransferAmount().stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        rubCurrency.getAPIAbbreviation("en")));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6578")
    public void transferFromOneActiveWithLowRestBalanceToOne() {
        CampaignGetItem campaignFrom =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID1);
        CampaignGetItem campaignTo =
                api.userSteps.campaignSteps().getCampaign(clientLogin, activeCampaignID2);

        float sumFrom = Money.valueOf(campaignFrom.getFunds().getCampaignFunds().getBalance())
                .divide(BigDecimal.valueOf(1_000_000L)).subtract(9.99f).floatValue();
        float sumTo = sumFrom;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumFrom))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(sumTo)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_LOW_REMAINING_BALANCE, activeCampaignID1));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignFrom),
                equalTo(false));
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                        .campaignFundsChanged(clientLogin, campaignTo),
                equalTo(false));
    }
}
