package ru.yandex.autotests.directapi.finance.transfermoney;

import java.math.RoundingMode;

import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.TransferMoneyInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.CoreMatchers.equalTo;

/**
 * Author pavryabov
 * Date 28.04.14
 */
@Aqua.Test
@Features(FinanceFeatures.TRANSFER_MONEY)
@Tag(TagDictionary.TRUNK)
public class TransferMoneyStoppedCampaignTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static final String CLIENT_LOGIN = Logins.SUBCLIENT_RUB;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(Logins.AGENCY_RUB).clientLogin(CLIENT_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static final Money INIT_SUM = Money.valueOf(15000f, Currency.RUB).addVAT();

    private long activeCampaignID1;
    private static long activeCampaignID2;

    @BeforeClass
    public static void createCampaigns() {
        activeCampaignID2 = api.userSteps.addActiveCampaign(CLIENT_LOGIN, INIT_SUM.floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(activeCampaignID2));
    }

    @Before
    public void activeCampaign() {
        activeCampaignID1 = api.userSteps.addActiveCampaign(CLIENT_LOGIN, INIT_SUM.floatValue(), Currency.RUB);
        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(activeCampaignID1));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6626")
    public void transferFromOneStoppedWithLowRestBalanceToOne() {
        CampaignGetItem campaignFrom = api.userSteps.campaignSteps().getCampaign(activeCampaignID1);
        api.userSteps.campaignFakeSteps().makeCampaignStopped(activeCampaignID1);
        CampaignGetItem campaignTo = api.userSteps.campaignSteps().getCampaign(activeCampaignID2);
        MoneyCurrency campCurrency = MoneyCurrency.get(Currency.getFor(campaignFrom.getCurrency().value()));

        Money transferSum = Money.valueOf(campaignFrom.getFunds().getCampaignFunds().getBalance(), Currency.RUB)
                .bidShort().subtract(MoneyCurrency.get(Currency.RUB).getMinTransferAmount().getPrevious())
                .setScale(2, RoundingMode.HALF_UP);
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID1)
                                        .withCurrency(Currency.RUB)
                                        .withSum(transferSum))
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(activeCampaignID2)
                                        .withCurrency(Currency.RUB)
                                        .withSum(transferSum)),
                new AxisError(353, AxisErrorDetails.TRANSFERMONEY_INVALID_REMAINING_BALANCE_WITH_CURRENCY,
                        campCurrency.getMinTransferRest().stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        campCurrency.getApiDescription()));

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(campaignFrom),
                equalTo(false));

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChanged(campaignTo),
                equalTo(false));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6625")
    public void transferAllFromStopped() {
        CampaignGetItem campaignFrom = api.userSteps.campaignSteps().getCampaign(activeCampaignID1);
        CampaignGetItem campaignTo = api.userSteps.campaignSteps().getCampaign(activeCampaignID2);

        Money transferSum = Money.valueOf(campaignFrom.getFunds().getCampaignFunds().getBalance(), Currency.RUB)
                .bidShort().setScale(2, RoundingMode.HALF_UP);
        api.userSteps.campaignFakeSteps().makeCampaignStopped(activeCampaignID1);
        api.userSteps.financeSteps().transferMoney(new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(transferSum))
                .withToCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID2)
                                .withCurrency(Currency.RUB)
                                .withSum(transferSum)));

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(
                        campaignFrom, transferSum.negate()));

        ConditionFactories.NOTIFY_ORDER2.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(
                        campaignTo, transferSum));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6627")
    public void transferNotAllMoneyFromStoppedCampaign() {
        CampaignGetItem campaignFrom = api.userSteps.campaignSteps().getCampaign(activeCampaignID1);
        CampaignGetItem campaignTo = api.userSteps.campaignSteps().getCampaign(activeCampaignID2);

        Money transferSum = Money.valueOf(campaignFrom.getFunds().getCampaignFunds().getBalance(), Currency.RUB)
                .bidShort().subtract(MoneyCurrency.get(Currency.RUB).getMinTransferAmount())
                .setScale(2, RoundingMode.HALF_UP);
        api.userSteps.campaignFakeSteps().makeCampaignStopped(activeCampaignID1);
        api.userSteps.financeSteps().transferMoney(new TransferMoneyInfoMap(api.type())
                .withFromCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID1)
                                .withCurrency(Currency.RUB)
                                .withSum(transferSum))
                .withToCampaigns(
                        new PayCampElementMap(api.type())
                                .withCampaignID(activeCampaignID2)
                                .withCurrency(Currency.RUB)
                                .withSum(transferSum)));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(campaignFrom, transferSum.negate()));

        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps()
                .campaignFundsChangedOnFunds(campaignTo, transferSum));
    }
}
