package ru.yandex.autotests.directapi.finance.transfermoney;

import java.math.RoundingMode;
import java.util.Arrays;
import java.util.Collection;

import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.aqua.annotations.project.Feature;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.direct.utils.money.MoneyFormat;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.finance.FinanceFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.TransferMoneyInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

/**
 * Author pavryabov
 * Date 26.05.14
 * https://jira.yandex-team.ru/browse/TESTIRT-513
 */
@Aqua.Test
@Feature(FinanceFeatures.TRANSFER_MONEY)
@Stories(ApiStories.CURRENCY)
@RunWith(Parameterized.class)
public class TransferMoneySumFieldTest {
    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static String rubClient = Logins.LOGIN_FOR_RUB;
    private static String usdClient = Logins.LOGIN_FOR_USD;
    private static String bynClient = Logins.BYN_CLIENT2;

    @Parameterized.Parameter(0)
    public String client;

    @Parameterized.Parameter(1)
    public Currency clientCurrency;

    @Parameterized.Parameter(2)
    public Currency transferCurrency;

    @Parameterized.Parameters(name = "Client = {0}, TransferCurrency = {2}")
    public static Collection logins() {
        Object[][] data = new Object[][]{
                {rubClient, Currency.RUB, Currency.RUB},
                {usdClient, Currency.USD, Currency.USD},
                {bynClient, Currency.BYN, Currency.BYN}
        };
        return Arrays.asList(data);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6629")
    public void transferMoneyWithSumLessThanMinInTo() {
        api.as(client);
        long campaignIDFrom = api.userSteps.addActiveCampaign(
                3000f, clientCurrency);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps().campaignFundsChanged(campaignIDFrom));
        long campaignIDTo1 = api.userSteps.addModeratedCampaign();
        long campaignIDTo2 = api.userSteps.addModeratedCampaign();
        float sumFrom = 2000f;
        float sumTo1 = MoneyCurrency.get(clientCurrency).getMinTransferAmount().getPrevious()
                .convert(transferCurrency).setScale(RoundingMode.DOWN).floatValue();
        float sumTo2 = sumFrom - sumTo1;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDFrom)
                                        .withSum(sumFrom)
                                        .withCurrency(transferCurrency)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDTo1)
                                        .withSum(sumTo1)
                                        .withCurrency(transferCurrency),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDTo2)
                                        .withSum(sumTo2)
                                        .withCurrency(transferCurrency)
                        ),
                new AxisError(353, AxisErrorDetails.LOW_FUND_TRANSFERS_TO,
                        campaignIDTo1,
                        MoneyCurrency.get(clientCurrency).getMinTransferAmount()
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        MoneyCurrency.get(clientCurrency).getApiDescription()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("6630")
    public void transferMoneyWithSumLessThanMinInFrom() {
        api.as(client);
        long campaignIDFrom1 = api.userSteps.addActiveCampaign(3000f, clientCurrency);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps().campaignFundsChanged(campaignIDFrom1));
        long campaignIDFrom2 = api.userSteps.addActiveCampaign(3000f, clientCurrency);
        ConditionFactories.NOTIFY_ORDER2.until(api.userSteps.campaignSteps().campaignFundsChanged(campaignIDFrom2));
        long campaignIDTo = api.userSteps.addModeratedCampaign();
        float sumTo = 2000f;
        float sumFrom1 = MoneyCurrency.get(clientCurrency).getMinTransferAmount().getPrevious()
                .convert(transferCurrency).setScale(RoundingMode.DOWN).floatValue();
        float sumFrom2 = sumTo - sumFrom1;
        api.userSteps.financeSteps().shouldGetErrorOnTransferMoney(
                new TransferMoneyInfoMap(api.type())
                        .withFromCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDFrom1)
                                        .withSum(sumFrom1)
                                        .withCurrency(transferCurrency),
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDFrom2)
                                        .withSum(sumFrom2)
                                        .withCurrency(transferCurrency)
                        )
                        .withToCampaigns(
                                new PayCampElementMap(api.type())
                                        .withCampaignID(campaignIDTo)
                                        .withSum(sumTo)
                                        .withCurrency(transferCurrency)
                        ),
                new AxisError(353, AxisErrorDetails.LOW_FUND_TRANSFERS_FROM,
                        MoneyCurrency.get(clientCurrency).getMinTransferAmount()
                                .stringValue(MoneyFormat.TWO_DIGITS_POINT_SEPARATED),
                        MoneyCurrency.get(clientCurrency).getApiDescription()));
    }
}
