package ru.yandex.autotests.directapi.images.adimage.delete;

import org.junit.*;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.matchers.BeanEquals;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.common.api45.AdImage;
import ru.yandex.autotests.directapi.common.api45.AdImageAssociationResponse;
import ru.yandex.autotests.directapi.common.api45.AdImageResponse;
import ru.yandex.autotests.directapi.enums.ImageType;
import ru.yandex.autotests.directapi.images.ImageLogins;
import ru.yandex.autotests.directapi.images.ImagesFeatures;
import ru.yandex.autotests.directapi.matchers.axiserror.ApiResponse;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.images.AdImageMap;
import ru.yandex.autotests.directapi.model.images.AdImageSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.hazelcast.SemaphoreRule;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;

import java.util.Arrays;
import java.util.List;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.*;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 24.05.13
 * <p>
 * Удаление только по AdImageHashes
 * + внутренние роли
 */
@Aqua.Test
@Features(ImagesFeatures.AD_IMAGE_DELETE)
@Tag(TagDictionary.TRUNK)
public class DeleteImageByHashSuccessTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String login = ImageLogins.LOGIN_FOR_DELETE_TESTS1;
    private static String another_login = ImageLogins.LOGIN_FOR_DELETE_TESTS2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private String sameHash;
    private String[] selfHashes;

    @Before
    public void initTest() {
        String[] hashes = api.as(login).userSteps.imagesSteps().configLoginImages(login, ImageType.REGULAR, 1, 2, 4);
        api.as(another_login).userSteps.imagesSteps().configLoginImages(login, ImageType.REGULAR, 1, 3);
        sameHash = hashes[0];
        selfHashes = new String[]{hashes[1], hashes[2]};
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7425")
    public void deleteImageByHash() {
        String[] hashesToDelete = new String[]{selfHashes[0]};
        log.info("Удаление картинки: " + Arrays.toString(hashesToDelete));

        AdImageResponse response = api.as(login).userSteps.imagesSteps().adImageDelete(hashesToDelete);
        assertThat("Ошибка при удалении картинки", response, equalTo(new AdImageResponse()));

        AdImage[] images = api.userSteps.imagesSteps().getAdImages(
                new AdImageSelectionCriteriaMap(api.type()).withAdImageHashes(hashesToDelete)
        );
        List<String> imagesHashes =
                extract(images, on(AdImage.class).getAdImageHash());
        assertThat(imagesHashes, not(hasItems(hashesToDelete)));
        log.info("Картинка удалена");
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7426")
    public void deleteUnbindedImage() {
        Long campaignId = api.as(login).userSteps.campaignStepsV5().addDefaultTextCampaign(login);
        Long adGroupId = api.as(login).userSteps.adGroupsSteps().addDefaultGroup(campaignId, login);
        Long adId = api.userSteps.adsSteps().addDefaultTextAd(adGroupId, login);
        String hashToDelete = selfHashes[0];
        AdImageAssociationResponse adImageAssociationResponse =
                api.userSteps.imagesSteps().setAdImageAssociation(adId, hashToDelete);
        assumeThat("картинка привязалась", JsonUtils.toString(adImageAssociationResponse), ApiResponse.hasNoError());
        api.userSteps.imagesSteps().setAdImageAssociation(adId, null);
        log.info("Удаление картинки: " + hashToDelete);

        AdImageResponse response = api.userSteps.imagesSteps().adImageDelete(hashToDelete);
        assertThat("Ошибка при удалении картинки", response, equalTo(new AdImageResponse()));

        AdImage[] images = api.userSteps.imagesSteps().getAdImages(
                new AdImageSelectionCriteriaMap(api.type()).withAdImageHashes(hashToDelete)
        );
        List<String> imagesHashes =
                extract(images, on(AdImage.class).getAdImageHash());
        assertThat(imagesHashes, not(hasItems(hashToDelete)));
        log.info("Картинка удалена");
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7427")
    public void deleteByAdHashes() {
        String[] hashesToDelete = selfHashes;
        log.info("Удаление картинок: " + Arrays.toString(hashesToDelete));

        AdImageResponse response = api.as(login).userSteps.imagesSteps().adImageDelete(hashesToDelete);
        assertThat("Ошибка при удалении картинки", response, equalTo(new AdImageResponse()));

        AdImage[] images = api.userSteps.imagesSteps().getAdImages(
                new AdImageSelectionCriteriaMap(api.type()).withAdImageHashes(hashesToDelete)
        );
        List<String> imagesHashes =
                extract(images, on(AdImage.class).getAdImageHash());
        assertThat(imagesHashes, not(hasItems(hashesToDelete)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7428")
    public void deleteBySameAdHash() {
        String[] hashesToDelete = new String[]{sameHash};

        AdImageResponse response = api.as(login).userSteps.imagesSteps().adImageDelete(hashesToDelete);
        assertThat("Ошибка при удалении картинки", response, equalTo(new AdImageResponse()));

        AdImage[] images = api.as(another_login).userSteps.imagesSteps().getAdImages(
                new AdImageSelectionCriteriaMap(api.type()).withAdImageHashes(hashesToDelete)
        );
        List<String> imagesHashes =
                extract(images, on(AdImage.class).getAdImageHash());
        assertThat(imagesHashes, hasItems(hashesToDelete));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7429")
    public void internalRoleDeleteByOneLoginHashes() {
        String[] hashesToDelete = new String[]{sameHash};

        AdImageResponse response = api.as(Logins.SUPER_LOGIN).userSteps.imagesSteps()
                .adImageDelete(new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(hashesToDelete)
                        .withLogins(login));
        assertThat("Ошибка при удалении картинки", response, equalTo(new AdImageResponse()));

        AdImage[] images = api.as(Logins.SUPER_LOGIN).userSteps.imagesSteps().getAdImages(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(hashesToDelete)
                        .withLogins(login, another_login)
        );
        AdImage expectedImage = (AdImage) new AdImageMap(api.type())
                .withAdImageHash(sameHash)
                .withLogin(another_login)
                .getBean();
        assertThat("Неверный список картинок",
                Arrays.asList(images[0]), contains(BeanEquals.beanEquals(expectedImage)));
    }
}
