package ru.yandex.autotests.directapi.images.adimage.delete;

import org.apache.commons.lang3.StringUtils;
import org.junit.*;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45.AdImage;
import ru.yandex.autotests.directapi.common.api45.AdImageResponse;
import ru.yandex.autotests.directapi.enums.ImageType;
import ru.yandex.autotests.directapi.images.ImageLogins;
import ru.yandex.autotests.directapi.images.ImagesFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.images.AdImageMap;
import ru.yandex.autotests.directapi.model.images.AdImageRequestMap;
import ru.yandex.autotests.directapi.model.images.AdImageSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.hazelcast.SemaphoreRule;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;

import java.util.List;

import static ch.lambdaj.Lambda.*;
import static org.hamcrest.Matchers.*;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 24.05.13
 * <p>
 * Удаление только по AdImageHashes
 * + внутренние роли
 */
@Aqua.Test
@Features(ImagesFeatures.AD_IMAGE_DELETE)
@Tag(TagDictionary.TRUNK)
public class DeleteImageByHashTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String login = ImageLogins.LOGIN_FOR_DELETE_TESTS3;
    private static String another_login = ImageLogins.LOGIN_FOR_DELETE_TESTS4;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static String noRight;
    private static String[] hasRight;
    private String[] hashes;

    @Before
    public void initTest() {
        hashes = api.as(login).userSteps.imagesSteps().configLoginImages(login, ImageType.REGULAR, 1, 2, 4, 5);
        String[] hashes_another = api.as(another_login).userSteps.imagesSteps().configLoginImages(login, ImageType.REGULAR, 1, 3);
        noRight = hashes_another[1];
        hasRight = new String[]{hashes[1], hashes[2]};
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7430")
    public void deleteByNoRightHash() {
        api.as(login).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .delete(
                                new AdImageSelectionCriteriaMap<>(api.type())
                                        .withAdImageHashes(noRight)),
                new AxisError(271, AxisErrorDetails.INVALID, StringUtils.capitalize(AdImageMap.AD_IMAGE_HASH)));

        AdImage[] images = api.as(another_login).userSteps.imagesSteps().getAdImages(
                new AdImageSelectionCriteriaMap(api.type()).withAdImageHashes(noRight)
        );
        List<String> ahashes = extract(images, on(AdImage.class).getAdImageHash());
        assertThat(ahashes, contains(noRight));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7431")
    public void deleteByEmptyHashArray() {
        api.as(login).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .delete(
                                new AdImageSelectionCriteriaMap<>(api.type())
                                        .withAdImageHashes()),
                new AxisError(271, AxisErrorDetails.INVALID, StringUtils.capitalize(AdImageMap.AD_IMAGE_HASH)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7432")
    public void deleteAssignedImage() {
        Long campaignID = api.as(login).userSteps.campaignStepsV5().addDefaultTextCampaign(login);
        Long adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID, login);
        Long adID = api.userSteps.adsSteps().addDefaultTextAd(adGroupID, login);
        api.userSteps.imagesSteps().assignImageToBannerIfNot(adID, hasRight[0]);
        api.userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .delete(
                                new AdImageSelectionCriteriaMap(api.type())
                                        .withAdImageHashes(hasRight)
                        ),
                new AxisError(80, AxisErrorDetails.ASSIGNED_IMAGES_COULD_NOT_BE_DELETED));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7433")
    public void deleteNotExistingImage() {
        String deletedImage = hashes[3];
        AdImageResponse response = api.as(login).userSteps.imagesSteps().adImageDelete(deletedImage);
        assumeThat("отсутствует ошибка при удалении картинки", response, equalTo(new AdImageResponse()));

        api.as(login).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .delete(
                                new AdImageSelectionCriteriaMap(api.type())
                                        .withAdImageHashes(deletedImage)
                        ),
                new AxisError(271, AxisErrorDetails.INVALID, "AdImageHash"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7434")
    public void internalRoleDeleteBySomeLoginHashes() {
        api.as(Logins.SUPER_LOGIN).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .delete(
                                new AdImageSelectionCriteriaMap(api.type())
                                        .withAdImageHashes(hasRight[0], noRight)
                                        .withLogins(login, another_login)),
                new AxisError(241, AxisErrorDetails.THE_ARRAY_MUST_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                        "SelectionCriteria.Logins", 1));

        AdImageResponse response = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(hasRight[0], noRight)
                        .withLogins(login, another_login)
        );
        List<String> hashes = extract(response.getAdImages(), on(AdImage.class).getAdImageHash());

        assertThat(hashes, containsInAnyOrder(hasRight[0], noRight));

    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7435")
    public void internalRoleDeleteNoLogin() {
        api.as(Logins.SUPER_LOGIN).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .delete(
                                new AdImageSelectionCriteriaMap(api.type())
                                        .withAdImageHashes(hasRight)),
                new AxisError(71, AxisErrorDetails.FIELD_MUST_NOT_BE_EMPTY,
                        "SelectionCriteria.Logins"));
    }
}
