package ru.yandex.autotests.directapi.images.adimage.get;

import org.apache.commons.lang.ArrayUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.enums.ImageType;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Action;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45.AdImageResponse;
import ru.yandex.autotests.directapi.model.images.AdImageMap;
import ru.yandex.autotests.directapi.model.images.AdImageRequestMap;
import ru.yandex.autotests.directapi.model.images.AdImageSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.hazelcast.SemaphoreRule;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.ImagesSteps;
import ru.yandex.autotests.directapi.images.ImageLogins;
import ru.yandex.autotests.directapi.images.ImagesFeatures;
import ru.yandex.qatools.allure.annotations.Features;

import java.util.Arrays;
import java.util.List;

import static ch.lambdaj.Lambda.extractProperty;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * User: mariabye
 * Date: 22.05.13
 */
@Aqua.Test
@Features(ImagesFeatures.AD_IMAGE_GET)
@Tag(TagDictionary.TRUNK)
public class PagerTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String login = ImageLogins.LOGIN_GET_IMAGES_5;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);


    private static String[] clientImages;

    @BeforeClass
    public static void initTest() {
        clientImages = api.as(login).userSteps.imagesSteps().configLoginImages(login, ImageType.REGULAR, 1, 2, 3, 4, 5);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7469")
    public void getHead() {
        AdImageResponse imageResponse = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(clientImages)
                        .withOffset(0)
                        .withLimit(2)
        );

        List<String> gotHashes = extractProperty(imageResponse.getAdImages(), AdImageMap.AD_IMAGE_HASH);
        assertThat(gotHashes, equalTo(Arrays.asList(clientImages[0], clientImages[1])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7470")
    public void getBody() {
        AdImageResponse imageResponse = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(clientImages)
                        .withOffset(1)
                        .withLimit(2)
        );
        List<String> gotHashes = extractProperty(imageResponse.getAdImages(), AdImageMap.AD_IMAGE_HASH);
        assertThat(gotHashes, equalTo(Arrays.asList(clientImages[1], clientImages[2])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7479")
    public void getTail() {
        AdImageResponse imageResponse = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(clientImages)
                        .withOffset(clientImages.length - 2)
                        .withLimit(2)
        );
        List<String> gotHashes = extractProperty(imageResponse.getAdImages(), AdImageMap.AD_IMAGE_HASH);
        assertThat(gotHashes, equalTo(Arrays.asList(
                clientImages[clientImages.length - 2], clientImages[clientImages.length - 1])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7471")
    public void getOverTail() {
        AdImageResponse imageResponse = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(clientImages)
                        .withOffset(clientImages.length - 2)
                        .withLimit(clientImages.length + 10)
        );
        List<String> gotHashes = extractProperty(imageResponse.getAdImages(), AdImageMap.AD_IMAGE_HASH);
        assertThat(gotHashes, equalTo(Arrays.asList(
                clientImages[clientImages.length - 2], clientImages[clientImages.length - 1])));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7472")
    public void getNextAfterTail() {
        AdImageResponse imageResponse = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(clientImages)
                        .withOffset(clientImages.length)
                        .withLimit(10)
        );
        List<String> gotHashes = extractProperty(imageResponse.getAdImages(), AdImageMap.AD_IMAGE_HASH);
        assertThat(gotHashes, empty());
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7473")
    public void requestEmptyLimitWithOffset() {
        AdImageResponse imageResponse = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(clientImages)
                        .withOffset(1)
        );
        List<String> gotHashes = extractProperty(imageResponse.getAdImages(), AdImageMap.AD_IMAGE_HASH);
        assertThat(gotHashes, equalTo(Arrays.asList((String[]) ArrayUtils.remove(clientImages.clone(), 0))));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7474")
    public void requestNegativeLimit() {
        api.userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .withAction(Action.GET)
                        .withAdImageSelectionCriteria(
                                new AdImageSelectionCriteriaMap(api.type())
                                        .withAdImageHashes(clientImages)
                                        .withLimit(-10)
                        ),
                new AxisError(71,
                        AxisErrorDetails.FIELD_SHOULD_CONTAIN_AN_UNSIGNED_INTEGER_VALUE,
                        "SelectionCriteria.Limit")
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7475")
    public void requestOverLimit() {
        api.userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .withAction(Action.GET)
                        .withAdImageSelectionCriteria(
                                new AdImageSelectionCriteriaMap(api.type())
                                        .withAdImageHashes(clientImages)
                                        .withOffset(0)
                                        .withLimit(ImagesSteps.AD_IMAGE_GET_LIMIT_MAX_NUMBER + 1)
                        ),
                new AxisError(71,
                        AxisErrorDetails.FIELD_VALUE_SHOULD_NOT_BE_GREATER_THAN,
                        "SelectionCriteria.Limit", ImagesSteps.AD_IMAGE_GET_LIMIT_MAX_NUMBER)
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7476")
    public void requestMaxLimit() {
        AdImageResponse imageResponse = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(clientImages)
                        .withOffset(0)
                        .withLimit(ImagesSteps.AD_IMAGE_GET_LIMIT_MAX_NUMBER)
        );
        List<String> gotHashes = extractProperty(imageResponse.getAdImages(), AdImageMap.AD_IMAGE_HASH);
        assertThat(gotHashes, equalTo(Arrays.asList(clientImages)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7477")
    public void requestEmptyOffsetWithLimit() {
        AdImageResponse imageResponse = api.userSteps.imagesSteps().adImageGet(
                new AdImageSelectionCriteriaMap(api.type())
                        .withAdImageHashes(clientImages)
                        .withLimit(10)
        );
        List<String> gotHashes = extractProperty(imageResponse.getAdImages(), AdImageMap.AD_IMAGE_HASH);
        assertThat(gotHashes, equalTo(Arrays.asList(clientImages)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7478")
    public void requestNegativeOffset() {
        api.userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .withAction(Action.GET)
                        .withAdImageSelectionCriteria(
                                new AdImageSelectionCriteriaMap(api.type())
                                        .withAdImageHashes(clientImages)
                                        .withOffset(-10)
                        ),
                new AxisError(71,
                        AxisErrorDetails.FIELD_SHOULD_CONTAIN_AN_UNSIGNED_INTEGER_VALUE,
                        "SelectionCriteria.Offset")
        );
    }
}
