package ru.yandex.autotests.directapi.images.adimage.getlimits;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.images.ImagesFeatures;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.direct.utils.matchers.BeanEquals;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45.AdImageLimit;
import ru.yandex.autotests.directapi.common.api45.AdImageResponse;
import ru.yandex.autotests.directapi.matchers.arrays.ArraySizeMatcher;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.clients.ClientInfoMap;
import ru.yandex.autotests.directapi.model.images.AdImageRequestMap;
import ru.yandex.autotests.directapi.model.images.AdImageSelectionCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.hazelcast.SemaphoreRule;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.ImagesSteps;
import ru.yandex.autotests.directapi.images.ImageLogins;
import ru.yandex.qatools.allure.annotations.Features;

import java.util.Arrays;
import java.util.Collection;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * User: mariabye
 * Date: 23.05.13
 */
@Aqua.Test
@Features(ImagesFeatures.AD_IMAGE_GET_LIMITS)
@Tag(TagDictionary.TRUNK)
@RunWith(Parameterized.class)
public class GetLimitRequestByLoginTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String login;
    private static String agencyLogin = ImageLogins.AGENCY_YE_DEFAULT;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(agencyLogin);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void createSubClient() {
        User user = api.userSteps.addAgencySubClient(agencyLogin,"at-image-limit", Currency.RUB);
        login = user.getLogin();
    }

    @Parameterized.Parameter
    public String opertator;

    @Parameterized.Parameters(name = "operator = {0}")
    public static Collection roleValues() {
        Object[][] data = new Object[][]{
                {agencyLogin},
                {Logins.PLACER},
                {Logins.SUPER_READER},
                {Logins.SUPPORT},
        };
        return Arrays.asList(data);
    }
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7483")
    public void getLimitByUndefLogins() {
        api.as(opertator).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .getLimits(new AdImageSelectionCriteriaMap<>(api.type())
                                .withLogins((String[])null)),
                new AxisError(71, AxisErrorDetails.FIELD_MUST_NOT_BE_EMPTY, "SelectionCriteria.Logins")
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7484")
    public void getLimitByLogin() {
        AdImageLimit limit = api.as(opertator).userSteps.imagesSteps().getLimit(login);

        AdImageLimit defaultLimit = new AdImageLimit();
        defaultLimit.setLogin(ClientInfoMap.normilizeLogin(login));
        defaultLimit.setUtilized(0);
        defaultLimit.setCapacity(1002);
        assertThat("Неверные значения лимитов", limit, BeanEquals.beanEquals(defaultLimit));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7485")
    public void getLimitByNoDirectLogin() {
        api.as(opertator).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .getLimits(new AdImageSelectionCriteriaMap<>(api.type())
                                .withLogins(ImageLogins.NO_DIRECT_CLIENT)),
                new AxisError(251, AxisErrorDetails.NO_ONE_VALID_LOGIN)
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7487")
    public void getLimitByMaxNumberOfLogins() {
        String[] logins = new String[ImagesSteps.GET_LIMIT_LOGINS_MAX_SIZE];
        Arrays.fill(logins, login);
        AdImageLimit[] limit = api.as(opertator).userSteps.imagesSteps().getLimits(logins);

        assertThat("Неверное число записей в ответе", limit.length, equalTo(logins.length));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7488")
    public void getLimitByOverLimitLogins() {
        String[] logins = new String[ImagesSteps.GET_LIMIT_LOGINS_MAX_SIZE + 1];
        Arrays.fill(logins, login);

        api.as(opertator).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap<>(api.type())
                        .getLimits(new AdImageSelectionCriteriaMap<>(api.type())
                                .withLogins(logins)),
                new AxisError(241, AxisErrorDetails.THE_ARRAY_MUST_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                        "SelectionCriteria.Logins", ImagesSteps.GET_LIMIT_LOGINS_MAX_SIZE)
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7486")
    public void getLimitsLimitAndOffsetIgnored() {
        AdImageResponse response = api.as(ImageLogins.SUPPORT).userSteps.imagesSteps()
                .adImageGetLimits(new AdImageSelectionCriteriaMap(api.type())
                        .withLogins(ImageLogins.CLIENT_SELF, login)
                        .withLimit(1).withOffset(1));

        assertThat("Неверные значения лимитов", response.getAdImageLimits(), ArraySizeMatcher.hasSize(2));
    }
}
