package ru.yandex.autotests.directapi.images.adimage.upload;

import org.apache.commons.lang.RandomStringUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.enums.ImageType;
import ru.yandex.autotests.directapi.images.ImageLogins;
import ru.yandex.autotests.directapi.images.ImagesFeatures;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.AdImageResponse;
import ru.yandex.autotests.directapi.common.api45.AdImageUpload;
import ru.yandex.autotests.directapi.model.images.AdImageSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.images.AdImageURLMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.hazelcast.SemaphoreRule;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.ImagesSteps;
import ru.yandex.qatools.allure.annotations.Features;

import java.util.List;

import static ch.lambdaj.Lambda.collect;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.contains;
import static org.junit.Assert.assertThat;

/**
 * User: mariabye
 * Date: 23.05.13
 */
@Aqua.Test
@Features(ImagesFeatures.AD_IMAGE_UPLOAD)
public class ImageNameTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String login = ImageLogins.LOGIN_UPLOAD_1;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    @BeforeClass
    public static void initTest() {
        api.userSteps.imagesSteps().adImageDeleteForLogin(login);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7496")
    public void uploadImageWithValidName() {
        String name = RandomStringUtils.randomAlphabetic(5) + ".jpg";
        AdImageResponse uploadResponse = api.userSteps.imagesSteps().adImageUpload(
                new AdImageURLMap(api.type())
                        .withLogin(login)
                        .withName(name)
                        .withURL(ImagesSteps.getDefaultImageUrl(1, ImageType.REGULAR))
        );
        AdImageUpload[] uploads = api.userSteps.imagesSteps().getAdImageUploads(
                new AdImageSelectionCriteriaMap(api.type()).withAdImageUploadTaskIDS(
                        ImagesSteps.extractAdImageUploadTaskIDS(uploadResponse.getActionsResult())
                ));
        List<String> names = collect(uploads, on(AdImageUpload.class).getName());
        assertThat("Неверное название картинки", names, contains(name));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7495")
    public void uploadImagesWithSameNames() {
        String name = RandomStringUtils.randomAlphabetic(5) + ".jpg";
        AdImageResponse uploadResponse = api.userSteps.imagesSteps().adImageUpload(
                new AdImageURLMap(api.type())
                        .withLogin(login)
                        .withName(name)
                        .withURL(ImagesSteps.getDefaultImageUrl(2, ImageType.REGULAR)),
                new AdImageURLMap(api.type())
                        .withLogin(login)
                        .withName(name)
                        .withURL(ImagesSteps.getDefaultImageUrl(3, ImageType.REGULAR))
        );
        AdImageUpload[] uploads = api.userSteps.imagesSteps().getAdImageUploads(
                new AdImageSelectionCriteriaMap(api.type()).withAdImageUploadTaskIDS(
                        ImagesSteps.extractAdImageUploadTaskIDS(uploadResponse.getActionsResult())
                ));
        List<String> names = collect(uploads, on(AdImageUpload.class).getName());
        assertThat("Неверное название картинки", names, contains(name, name));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7497")
    public void uploadImagesWithDifferentNames() {
        String name1 = RandomStringUtils.randomAlphabetic(5) + ".jpg";
        String name2 = RandomStringUtils.randomAlphabetic(5) + ".jpg";
        AdImageResponse uploadResponse = api.userSteps.imagesSteps().adImageUpload(
                new AdImageURLMap(api.type())
                        .withLogin(login)
                        .withName(name1)
                        .withURL(ImagesSteps.getDefaultImageUrl(4, ImageType.REGULAR)),
                new AdImageURLMap(api.type())
                        .withLogin(login)
                        .withName(name2)
                        .withURL(ImagesSteps.getDefaultImageUrl(5, ImageType.REGULAR))
        );
        AdImageUpload[] uploads = api.userSteps.imagesSteps().getAdImageUploads(
                new AdImageSelectionCriteriaMap(api.type()).withAdImageUploadTaskIDS(
                        ImagesSteps.extractAdImageUploadTaskIDS(uploadResponse.getActionsResult())
                ));
        List<String> names = collect(uploads, on(AdImageUpload.class).getName());
        assertThat("Неверное название картинки", names, contains(name1, name2));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7498")
    public void uploadImageWithUndefName() {
        AdImageResponse response = api.userSteps.imagesSteps().adImageUpload(
                new AdImageURLMap(api.type())
                        .withLogin(login)
                        .withName(null)
                        .withURL(ImagesSteps.getDefaultImageUrl(6, ImageType.REGULAR))
        );
        api.userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(71, AxisErrorDetails.FIELD_MUST_NOT_BE_EMPTY, "AdImageURLData[0].Name")
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7499")
    public void uploadImageWithEmptyName() {
        AdImageResponse response = api.userSteps.imagesSteps().adImageUpload(
                new AdImageURLMap(api.type())
                        .withLogin(login)
                        .withName("")
                        .withURL(ImagesSteps.getDefaultImageUrl(7, ImageType.REGULAR))
        );
        api.userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(71, AxisErrorDetails.STRING_FIELD_SHOULD_NOT_BE_EMPTY, "AdImageURLData[0].Name")
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7500")
    public void uploadImageWithSpacesName() {
        AdImageResponse response = api.userSteps.imagesSteps().adImageUpload(
                new AdImageURLMap(api.type())
                        .withLogin(login)
                        .withName("   ")
                        .withURL(ImagesSteps.getDefaultImageUrl(7, ImageType.REGULAR))
        );
        api.userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(71, AxisErrorDetails.STRING_FIELD_SHOULD_NOT_BE_EMPTY, "AdImageURLData[0].Name")
        );
    }
}
