package ru.yandex.autotests.directapi.images.adimage.uploadrawdata;

import org.apache.commons.lang.RandomStringUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannerImagesFormatsImageType;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45.AdImage;
import ru.yandex.autotests.directapi.common.api45.AdImageResponse;
import ru.yandex.autotests.directapi.enums.ImageType;
import ru.yandex.autotests.directapi.images.ImageLogins;
import ru.yandex.autotests.directapi.images.ImagesFeatures;
import ru.yandex.autotests.directapi.model.images.AdImageRawMap;
import ru.yandex.autotests.directapi.model.images.AdImageRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.ImagesSteps;
import ru.yandex.autotests.directapi.utils.ImageBase64Utils;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * User: mariabye
 * Date: 23.05.13
 */
@Aqua.Test
@Features(ImagesFeatures.AD_IMAGE_UPLOAD_RAW_DATA)
@Tag(TagDictionary.TRUNK)
public class UploadImageRawTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String login = ImageLogins.LOGIN_UPLOADRAW_3;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static String name = RandomStringUtils.randomAlphabetic(5);

    @BeforeClass
    public static void deleteImages() {
        api.userSteps.imagesSteps().adImageDeleteForLogin(login);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7532")
    public void uploadImageWithUndefRaw() {
        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(
                new AdImageRawMap(api.type())
                        .withName(name)
                        .withRawData((String) null)
        );
        api.userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(71, AxisErrorDetails.FIELD_MUST_NOT_BE_EMPTY, "AdImageRawData[0].RawData")
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7533")
    public void uploadImageWithEmptyRaw() {
        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(
                new AdImageRawMap(api.type())
                        .withName(name)
                        .withRawData("")
        );
        api.userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(71, AxisErrorDetails.IMAGE_FIELD_MUST_BE_BASE64_ENCODED, "AdImageRawData[0].RawData")
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7534")
    public void uploadImageWithStringDataRaw() {
        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(
                new AdImageRawMap(api.type())
                        .withName(name)
                        .withRawData(RandomStringUtils.randomAlphabetic(256))
        );
        api.userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(272, AxisErrorDetails.INVALID_FILE_TYPE)
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7535")
    public void uploadTextFile() {
        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(
                new AdImageRawMap(api.type())
                        .withName(name)
                        .withRawData(ImageBase64Utils.getBase64EncodedFileData(
                                "http://aqua.yandex-team.ru/storage/get/indefinitely/report/base/4/report1.xml"))
        );
        api.userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(272, AxisErrorDetails.INVALID_FILE_TYPE)
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7536")
    public void uploadSomeImages() {
        AdImageRawMap[] imageRaws = new AdImageRawMap[ImagesSteps.RAW_IMAGE_POOL_LENGTH];
        for (int i = 1; i < ImagesSteps.RAW_IMAGE_POOL_LENGTH + 1; i++) {
            AdImageRawMap imageRaw = new AdImageRawMap(api.type())
                    .withName("pic" + i)
                    .withRawData(ImageBase64Utils
                            .getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(i, ImageType.REGULAR)));
            imageRaws[i - 1] = imageRaw;
        }

        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(imageRaws);
        assertThat(response.getActionsResult().length, equalTo(imageRaws.length));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("8884")
    public void uploadSomeImagesDifferentTypes() {
        AdImageRawMap[] imageRaws = new AdImageRawMap[2];
        AdImageRawMap regularImageRaw = new AdImageRawMap(api.type())
                .withName("regular image")
                .withRawData(ImageBase64Utils
                        .getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(1, ImageType.REGULAR)));
        imageRaws[0] = regularImageRaw;
        AdImageRawMap wideImageRaw = new AdImageRawMap(api.type())
                .withName("wide image")
                .withRawData(
                        ImageBase64Utils.getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(1, ImageType.WIDE)));
        imageRaws[1] = wideImageRaw;

        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(imageRaws);
        assertThat(response.getActionsResult().length, equalTo(imageRaws.length));
    }


    @Test
    public void successUploadRegularImageIfHaveOldSmall() {
        AdImage adImage = api.userSteps.imagesSteps().uploadRawImage("captain",
                "https://direct-qa.s3.mds.yandex.net/images/captain600x450.jpg", login);
        String smallAdImageHash = adImage.getAdImageHash();
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(login)
                .imagesSteps().setImageType(smallAdImageHash, BannerImagesFormatsImageType.small);
        AdImageRawMap regularImageRaw = new AdImageRawMap(api.type())
                .withName("regular image")
                .withRawData(ImageBase64Utils
                        .getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(1, ImageType.REGULAR)));

        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(regularImageRaw);
        assertThat(response.getActionsResult().length, equalTo(1));

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(login)
                .imagesSteps().deleteBannerImagesFormatsRecord(smallAdImageHash);
    }

    @Test
    public void successUploadWideImageIfHaveOldSmall() {
        AdImage adImage = api.userSteps.imagesSteps().uploadRawImage("bob",
                "https://direct-qa.s3.mds.yandex.net/images/bob600x450.jpg", login);
        String smallAdImageHash = adImage.getAdImageHash();

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(login)
                .imagesSteps().setImageType(smallAdImageHash, BannerImagesFormatsImageType.small);

        AdImageRawMap wideImageRaw = new AdImageRawMap(api.type())
                .withName("wide image")
                .withRawData(
                        ImageBase64Utils.getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(1, ImageType.WIDE)));

        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(wideImageRaw);
        assertThat(response.getActionsResult().length, equalTo(1));

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(login)
                .imagesSteps().deleteBannerImagesFormatsRecord(smallAdImageHash);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7537")
    public void uploadOverLimitImageNumber() {
        AdImageRawMap[] imageRaws = new AdImageRawMap[ImagesSteps.RAW_IMAGE_POOL_LENGTH + 1];
        for (int i = 1; i < ImagesSteps.RAW_IMAGE_POOL_LENGTH + 2; i++) {
            AdImageRawMap imageRaw = new AdImageRawMap(api.type())
                    .withName("pic" + i)
                    .withRawData(ImageBase64Utils
                            .getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(1, ImageType.REGULAR)));
            imageRaws[i - 1] = imageRaw;
        }

        api.userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE,
                new AdImageRequestMap(api.type())
                        .uploadRawData(imageRaws),
                new AxisError(241, AxisErrorDetails.THE_ARRAY_MUST_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                        "AdImageRawData", ImagesSteps.RAW_IMAGE_POOL_LENGTH)
        );
    }
}
