package ru.yandex.autotests.directapi.images.association.set;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang.RandomStringUtils;
import org.hamcrest.Matchers;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.db.models.jooq.ppc.enums.BannerImagesFormatsImageType;
import ru.yandex.autotests.direct.utils.tags.TagDictionary;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45.AdImage;
import ru.yandex.autotests.directapi.common.api45.AdImageActionResult;
import ru.yandex.autotests.directapi.common.api45.AdImageAssociationResponse;
import ru.yandex.autotests.directapi.common.api45.AdImageResponse;
import ru.yandex.autotests.directapi.enums.ImageType;
import ru.yandex.autotests.directapi.images.ImageLogins;
import ru.yandex.autotests.directapi.images.ImagesFeatures;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.images.AdImageAssociationMap;
import ru.yandex.autotests.directapi.model.images.AdImageAssociationRequestMap;
import ru.yandex.autotests.directapi.model.images.AdImageRawMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.ImagesSteps;
import ru.yandex.autotests.directapi.utils.ImageBase64Utils;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.Tag;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static java.util.Arrays.asList;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 22.05.13
 */
@Aqua.Test
@Features(ImagesFeatures.AD_IMAGE_ASSOCIATION_SET)
@Tag(TagDictionary.TRUNK)
public class AssignImageToAdTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    static String login = ImageLogins.LOGIN_WITH_ASSIGNED_IMAGES_8;

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).as(login);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static Long campaignID;
    private static Long adID1;
    private static Long adID2;
    private static Long adID3;
    private static Long adID4;
    private static String bigImageHash;
    private static List<String> hashes;
    private static String smallAdImageHash;

    @BeforeClass
    public static void initCampaign() {
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(login);
        Long adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID, login);
        List<Long> adIDs = api.userSteps.adsSteps().addDefaultTextAds(adGroupID, 4);

        adID1 = adIDs.get(0);
        adID2 = adIDs.get(1);
        adID3 = adIDs.get(2);
        adID4 = adIDs.get(3);

        AdImageResponse response = api.userSteps.imagesSteps().adImageUploadRawData(
                new AdImageRawMap(api.type())
                        .withLogin(login)
                        .withName(RandomStringUtils.randomAlphabetic(5))
                        .withRawData(ImageBase64Utils
                                .getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(1, ImageType.REGULAR))),
                new AdImageRawMap(api.type())
                        .withLogin(login)
                        .withName(RandomStringUtils.randomAlphabetic(5))
                        .withRawData(ImageBase64Utils
                                .getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(2, ImageType.REGULAR)))
        );
        hashes = Arrays.stream(response.getActionsResult())
                .map(AdImageActionResult::getAdImageHash)
                .collect(Collectors.toList());

        api.userSteps.imagesSteps().setAdImageAssociation(adID3, hashes.get(0));

        response = api.userSteps.imagesSteps().adImageUploadRawData(
                new AdImageRawMap(api.type())
                        .withLogin(login)
                        .withName(RandomStringUtils.randomAlphabetic(5))
                        .withRawData(ImageBase64Utils
                                .getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(1, ImageType.WIDE)))
        );
        bigImageHash = response.getActionsResult()[0].getAdImageHash();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7601")
    public void assignImageToAd() {
        api.as(login).userSteps.imagesSteps().setAdImageAssociation(adID1, hashes.get(0));
        String imageHash = api.userSteps.adsSteps().getAdImageHash(adID1, login);

        assertThat("Неверное значение AdImageHash", imageHash, equalTo(hashes.get(0)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("8886")
    public void assignBigImageToAd() {
        api.as(login).userSteps.imagesSteps().assignImageToBannerIfNot(adID4, bigImageHash);
        String imageHash = api.userSteps.adsSteps().getAdImageHash(adID4, login);

        assertThat("Неверное значение AdImageHash", imageHash, equalTo(bigImageHash));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7599")
    public void assignImageToNegativeAdID() {
        AdImageAssociationResponse response = api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                -adID1, hashes.get(0)
        );
        api.as(login).userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(71,
                        AxisErrorDetails.FIELD_SHOULD_CONTAIN_AN_UNSIGNED_INTEGER_VALUE,
                        "AdImageAssociations[0].AdID"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7600")
    public void assignImageToNoRightAdID() {
        AdImageAssociationResponse response = api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                3054L, hashes.get(0)
        );
        api.as(login).userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @Issue("DIRECT-49672")
    public void tryAssignOldSmallImageToAd() {
        AdImage adImage = api.userSteps.imagesSteps().uploadRawImage("car",
                "https://direct-qa.s3.mds.yandex.net/images/car600x450.jpg", login);
        smallAdImageHash = adImage.getAdImageHash();
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(login)
                .imagesSteps().setImageType(smallAdImageHash, BannerImagesFormatsImageType.small);
        AdImageAssociationResponse response = api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                adID4, smallAdImageHash
        );
        api.as(login).userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(272, AxisErrorDetails.IMAGE_SIZE_INVALID));
    }

    @Test
    @Issue("DIRECT-53453")
    public void tryAssignBrokenImageToAd() {
        AdImageResponse uploadResponse = api.userSteps.imagesSteps().adImageUploadRawData(
                new AdImageRawMap(api.type())
                        .withLogin(login)
                        .withName(RandomStringUtils.randomAlphabetic(5))
                        .withRawData(ImageBase64Utils
                                .getBase64EncodedFileData(ImagesSteps.getDefaultImageUrl(3, ImageType.REGULAR)))
        );
        String brokenImageHash = uploadResponse.getActionsResult()[0].getAdImageHash();

        api.userSteps.getDirectJooqDbSteps().useShardForLogin(login)
                .imagesSteps().deleteBannerImagesFormatsRecord(brokenImageHash);
        AdImageAssociationResponse setResponse = api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                adID4, brokenImageHash
        );
        api.as(login).userSteps.imagesSteps().responseShouldContainErrors(
                setResponse,
                new AxisError(271, AxisErrorDetails.IMAGE_NOT_FOUND));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7602")
    public void assignImageToNoExistAdID() {
        Long adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID, login);
        Long adID = api.userSteps.adsSteps().addDefaultTextAd(adGroupID, login);
        api.userSteps.adsSteps().adsDelete(adID);

        AdImageAssociationResponse response = api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                adID, hashes.get(0));
        api.as(login).userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(54, AxisErrorDetails.EMPTY_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7603")
    public void assignNoRightImageToAdID() {
        String imageHash = "H2DhotudJ4d68hbvXd-WRQ";
        assumeThat("подобран hash картинки, которая не загружена клиенту",
                hashes, Matchers.not(hasItem(imageHash)));
        AdImageAssociationResponse response = api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                adID1, imageHash
        );
        api.as(login).userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(271, AxisErrorDetails.IMAGE_NOT_FOUND));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7604")
    public void assignNoExistImageToAdID() {
        String imageHash = RandomStringUtils.randomAlphabetic(22);
        AdImageAssociationResponse response = api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                adID1, imageHash
        );
        api.as(login).userSteps.imagesSteps().responseShouldContainErrors(
                response,
                new AxisError(271, AxisErrorDetails.IMAGE_NOT_FOUND));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7605")
    public void assignSomeImagesToAds() {
        api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                new AdImageAssociationMap(api.type())
                        .withAdID(adID1)
                        .withAdImageHash(hashes.get(0)),
                new AdImageAssociationMap(api.type())
                        .withAdID(adID2)
                        .withAdImageHash(hashes.get(1))
        );

        String imageHash1 = api.userSteps.adsSteps().getAdImageHash(adID1, login);
        String imageHash2 = api.userSteps.adsSteps().getAdImageHash(adID2, login);
        List<String> bannerHashes = asList(imageHash1, imageHash2);
        assertThat("Неверное значение AdImageHash", bannerHashes, equalTo(asList(hashes.get(0), hashes.get(1))));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7606")
    public void assignOneImageToAds() {
        api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                new AdImageAssociationMap(api.type())
                        .withAdID(adID1)
                        .withAdImageHash(hashes.get(0)),
                new AdImageAssociationMap(api.type())
                        .withAdID(adID2)
                        .withAdImageHash(hashes.get(0))
        );

        String imageHash1 = api.userSteps.adsSteps().getAdImageHash(adID1, login);
        assertThat("Неверное значение AdImageHash", imageHash1, equalTo(hashes.get(0)));

        String imageHash2 = api.userSteps.adsSteps().getAdImageHash(adID2, login);
        assertThat("Неверное значение AdImageHash", imageHash2, equalTo(hashes.get(0)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7607")
    public void assignImagesToOneAd() {
        api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                new AdImageAssociationMap(api.type())
                        .withAdID(adID1)
                        .withAdImageHash(hashes.get(0)),
                new AdImageAssociationMap(api.type())
                        .withAdID(adID1)
                        .withAdImageHash(hashes.get(1))
        );

        String imageHash = api.userSteps.adsSteps().getAdImageHash(adID1, login);
        assertThat("Неверное значение AdImageHash", imageHash, equalTo(hashes.get(1)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7608")
    public void setAndUnsetImagesToAds() {
        api.as(login).userSteps.imagesSteps().setAdImageAssociation(
                new AdImageAssociationMap(api.type())
                        .withAdID(adID1)
                        .withAdImageHash(hashes.get(0)),
                new AdImageAssociationMap(api.type())
                        .withAdID(adID2)
                        .withAdImageHash(hashes.get(1)),
                new AdImageAssociationMap(api.type())
                        .withAdID(adID3)
                        .withAdImageHash(null)
        );

        List<String> imageHashes = asList(
                api.userSteps.adsSteps().getAdImageHash(adID1, login),
                api.userSteps.adsSteps().getAdImageHash(adID2, login),
                api.userSteps.adsSteps().getAdImageHash(adID3, login));
        assertThat("Неверное значение AdImageHash", imageHashes, equalTo(
                asList(hashes.get(0), hashes.get(1), null)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7609")
    public void setAndUnsetImagesToAdsLimitRequestItems() {
        AdImageAssociationMap[] associations = new AdImageAssociationMap[ImagesSteps.AD_IMAGE_HASHES_MAX_VALUE + 1];
        for (int i = 0; i < associations.length; i++) {
            associations[i] = new AdImageAssociationMap(api.type()).withAdID(adID1).withAdImageHash(hashes.get(0));
        }

        api.as(login).userSteps.imagesSteps().shouldGetErrorOn(
                Method.AD_IMAGE_ASSOCIATION,
                new AdImageAssociationRequestMap<>(api.type()).set(associations),
                new AxisError(241, AxisErrorDetails.THE_ARRAY_MUST_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                        "AdImageAssociations", ImagesSteps.AD_IMAGE_HASHES_MAX_VALUE)
        );
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7610")
    public void managerAssignImageToAd() {
        String subClient = Logins.CLIENT_FREE_YE_DEFAULT;
        Long campaignID = api.as(Logins.MANAGER_DEFAULT).userSteps.campaignSteps().addDefaultTextCampaign(subClient);
        Long adGroupID =
                api.as(Logins.MANAGER_DEFAULT).userSteps.adGroupsSteps().addDefaultGroup(campaignID, subClient);
        Long adID = api.userSteps.adsSteps().addDefaultTextAd(adGroupID, subClient);
        api.userSteps.imagesSteps().configLoginImages(subClient, ImageType.REGULAR, 1);

        String[] hash = api.userSteps.imagesSteps().getAccountImageHashes(subClient);
        assumeThat("достаточно картинок для проведения теста", hash.length, greaterThan(0));
        api.userSteps.imagesSteps().setAdImageAssociation(adID, hash[0]);

        String imageHash = api.userSteps.adsSteps().getAdImageHash(adID, subClient);
        assertThat("Неверное значение AdImageHash", imageHash, equalTo(hash[0]));
    }

    @AfterClass
    public static void removeSmallImage() {
        api.userSteps.getDirectJooqDbSteps().useShardForLogin(login)
                .imagesSteps().deleteBannerImagesFormatsRecord(smallAdImageHash);
    }
}
