package ru.yandex.autotests.directapi.sandbox.finance;

import java.util.Arrays;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.httpclient.data.sandbox.APISandboxClientTypeEnum;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.ApiFeatures;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.ShortClientInfo;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.sandbox.SandboxLogins;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.greaterThan;

/**
 * Created by chicos on 26.09.14.
 */
@Aqua.Test
@Features(ApiFeatures.SANDBOX)
public class CreateInvoiceTest {
    private static LogSteps log = LogSteps.getLogger(CreateInvoiceTest.class);

    private final Money minInvoiceSumRubCurrency = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();
    private static final String agency = SandboxLogins.AGENCY_SANDBOX_2;
    private static String subclient1;
    private static String subclient2;

    private static Long campaignId1;
    private static Long campaignId2;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(agency);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void initSandbox() {
        api.userSteps.sandboxSteps().enableSandboxByApi(agency, APISandboxClientTypeEnum.AGENCY, true, Currency.RUB, false);
        ShortClientInfo[] clients = api.userSteps.clientSteps().getSubClients(agency);
        TestSteps.assumeThat("агентство имеет несколько субклиентов", clients.length, greaterThan(1));

        subclient1 = clients[0].getLogin();
        subclient2 = clients[1].getLogin();

        campaignId1 = api.userSteps.addActiveCampaign(subclient1, 1);
        campaignId2 = api.userSteps.addActiveCampaign(subclient2, 1);
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignId1, campaignId2));
    }

    @Test
    public void createInvoiceOneCampaignTest() {
        PayCampElementMap campaignInvoice = new PayCampElementMap(api.type())
                .withCampaignID(campaignId1.intValue())
                .withCurrency(Currency.RUB)
                .withSum(minInvoiceSumRubCurrency.floatValue());
        String invoiceURL = api.userSteps.financeSteps().createInvoiceSandbox(campaignInvoice);
        TestSteps.assertThat("сформирован корректный URL выставленного счета", invoiceURL, containsString("request_id"));
    }

    @Test
    public void twoInvoicesTwoCampaignsTest() {
        PayCampElementMap campaignInvoice1 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId1.intValue())
                .withCurrency(Currency.RUB)
                .withSum(minInvoiceSumRubCurrency.floatValue());
        PayCampElementMap campaignInvoice2 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId2.intValue())
                .withCurrency(Currency.RUB)
                .withSum(minInvoiceSumRubCurrency.floatValue() + 1.0f);

        log.info("Выставление счета на разные кампании в одном запросе");
        String invoiceURL = api.userSteps.financeSteps().createInvoiceSandbox(campaignInvoice1, campaignInvoice2);
        TestSteps.assertThat("сформирован корректный URL выставленного счета", invoiceURL, containsString("request_id"));
    }

    @Test
    public void noRightsCampaignTest() {
        int campaignNoRights = 263;
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoiceSandbox(
                new AxisError(1, null),
                new PayCampElementMap(api.type())
                        .withCampaignID(campaignNoRights)
                        .withCurrency(Currency.RUB)
                        .withSum(minInvoiceSumRubCurrency.floatValue())
        );
    }

    @Test
    public void oneCampaignTwoInvoicesTest() {
        PayCampElementMap campaignInvoice1 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId1.intValue())
                .withCurrency(Currency.RUB)
                .withSum(minInvoiceSumRubCurrency.floatValue());
        PayCampElementMap campaignInvoice2 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId1.intValue())
                .withCurrency(Currency.RUB)
                .withSum(minInvoiceSumRubCurrency.floatValue() + 1.0f);

        log.info("Выставление двух счетов на одну кампанию в одном запросе");
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoiceSandbox(
                new AxisError(354, AxisErrorDetails.PAYCAMPELEMENT_CAMPAIGNID_MUST_BE_UNIQUE),
                campaignInvoice1, campaignInvoice2
        );
    }

    @Test
    public void twoInvoicesOneNoRightsCampaignTest() {
        int campaignNoRights = 263;
        PayCampElementMap campaignInvoice1 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId1.intValue())
                .withCurrency(Currency.RUB)
                .withSum(minInvoiceSumRubCurrency.floatValue());
        PayCampElementMap campaignInvoice2 = new PayCampElementMap(api.type())
                .withCampaignID(campaignNoRights)
                .withCurrency(Currency.RUB)
                .withSum(minInvoiceSumRubCurrency.floatValue() + 1.0f);

        log.info("Выставление счета на две кампании, на одну из которых нет прав");
        api.userSteps.financeSteps().shouldGetErrorOnCreateInvoiceSandbox(
                new AxisError(1, null),
                campaignInvoice1, campaignInvoice2
        );
    }

    @AfterClass
    public static void closeSandbox() {
        api.userSteps.sandboxSteps().closeSandbox(agency);
    }
}
