package ru.yandex.autotests.directapi.sandbox.finance;

import java.util.Arrays;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.httpclient.data.sandbox.APISandboxClientTypeEnum;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.ApiFeatures;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.common.api45.CreditLimitsInfo;
import ru.yandex.autotests.directapi.common.api45.ShortClientInfo;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.sandbox.SandboxLogins;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.greaterThan;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;


/**
 * Created by chicos on 29.09.14.
 */
@Aqua.Test
@Features(ApiFeatures.SANDBOX)
public class GetCreditLimitsTest {
    private static LogSteps log = LogSteps.getLogger(GetCreditLimitsTest.class);

    private final Money minInvoiceSumRubCurrency = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();
    private final float EXTRA_SUM = 7.03f;

    private static final String agency = SandboxLogins.AGENCY_SANDBOX_3;
    private static String subclient1;
    private static String subclient2;

    private static Long campaignId1;
    private static Long campaignId2;

    private CreditLimitsInfo limitsBefore;
    private CreditLimitsInfo limitsAfter;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(agency);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void initSandbox() {
        api.userSteps.sandboxSteps().enableSandboxByApi(agency, APISandboxClientTypeEnum.AGENCY, true, Currency.RUB, false);
        ShortClientInfo[] clients = api.userSteps.clientSteps().getSubClients(agency);
        TestSteps.assumeThat("агентство имеет несколько субклиентов", clients.length, greaterThan(1));

        subclient1 = clients[0].getLogin();
        subclient2 = clients[1].getLogin();

        campaignId1 = api.userSteps.addActiveCampaign(subclient1, 1);
        campaignId2 = api.userSteps.addActiveCampaign(subclient2, 1);
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignId1, campaignId2));
    }

    @Before
    public void storeLimitsBefore() {
        limitsBefore = api.userSteps.financeSteps().getCreditLimitsSandbox();
    }

    @After
    public void checkLimitsAfter() {
        limitsAfter = api.userSteps.financeSteps().getCreditLimitsSandbox();
        TestSteps.assertThat("кредитные остатки не изменились", limitsAfter, beanEquals(limitsBefore));
    }

    @Test
    public void getLimitsSpentOnePayTest() {
        float spentSum = minInvoiceSumRubCurrency.floatValue() + EXTRA_SUM;
        log.info("Осуществим платеж на кампанию");
        api.userSteps.financeSteps().payCampaignSandbox(spentSum, Currency.RUB, campaignId1.intValue());
    }

    @Test
    public void getLimitsSpentTwoPaysTest() {
        float spentSum = minInvoiceSumRubCurrency.floatValue() + EXTRA_SUM;
        log.info("Осуществим два платежа на кампанию");
        api.userSteps.financeSteps().payCampaignSandbox(spentSum, Currency.RUB, campaignId1.intValue());
        api.userSteps.financeSteps().payCampaignSandbox(spentSum, Currency.RUB, campaignId1.intValue());
    }

    @Test
    public void getLimitsSpentTwoPaysTwoCampaignsTest() {
        float spentSum = minInvoiceSumRubCurrency.floatValue() + EXTRA_SUM;
        log.info("Осуществим два платежа на разные кампании");
        api.userSteps.financeSteps().payCampaignSandbox(spentSum, Currency.RUB, campaignId1.intValue());
        api.userSteps.financeSteps().payCampaignSandbox(spentSum, Currency.RUB, campaignId2.intValue());
    }

    @AfterClass
    public static void closeSandbox() {
        api.userSteps.sandboxSteps().closeSandbox(agency);
    }
}
