package ru.yandex.autotests.directapi.sandbox.finance;

import java.util.Arrays;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import org.apache.commons.lang3.StringUtils;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.httpclient.data.sandbox.APISandboxClientTypeEnum;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.money.Money;
import ru.yandex.autotests.direct.utils.money.MoneyCurrency;
import ru.yandex.autotests.directapi.ApiFeatures;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.common.api45.ShortClientInfo;
import ru.yandex.autotests.directapi.model.finances.PayCampElementMap;
import ru.yandex.autotests.directapi.model.finances.PayCampaignsInfoMap;
import ru.yandex.autotests.directapi.model.finances.PayMethod;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.sandbox.SandboxLogins;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.greaterThan;


/**
 * Created by chicos on 30.09.14.
 */
@Aqua.Test
@Features(ApiFeatures.SANDBOX)
public class PayCampaignTest {
    private static LogSteps log = LogSteps.getLogger(PayCampaignTest.class);

    private final Money SUM_SPENT = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount().addVAT();
    private final Money SUM_SPENT_NO_VAT = MoneyCurrency.get(Currency.RUB).getMinInvoiceAmount();

    private static final String agency = SandboxLogins.AGENCY_SANDBOX_4;
    private static String subclient1;
    private static String subclient2;
    private static String subclient3;

    private static Long campaignId1;
    private static Long campaignId2;
    private static Long campaignId3;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(agency);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @BeforeClass
    public static void initSandbox() {
        api.userSteps.sandboxSteps().enableSandboxByApi(agency, APISandboxClientTypeEnum.AGENCY, true, Currency.RUB, false);
        ShortClientInfo[] clients = api.userSteps.clientSteps().getSubClients(agency);
        TestSteps.assertThat("агентство имеет несколько субклиентов", clients.length, greaterThan(1));

        subclient1 = clients[0].getLogin();
        subclient2 = clients[1].getLogin();
        subclient3 = clients[2].getLogin();

        campaignId1 = api.userSteps.addActiveCampaign(subclient1, 1);
        campaignId2 = api.userSteps.addActiveCampaign(subclient2, 1);
        campaignId3 = api.userSteps.addActiveCampaign(subclient3, 1);
        api.userSteps.campaignSteps().removeCampaignsFromQueue(Arrays.asList(campaignId1, campaignId2, campaignId3));
    }

    @Test
    public void payCampaignTest() {
        CampaignGetItem campaignBefore = api.userSteps.campaignSteps().campaignsGet(
                subclient1, campaignId1, CampaignFieldEnum.FUNDS, CampaignFieldEnum.ID);
        log.info("Оплата кампании по кредитному договору на сумму: " + SUM_SPENT);
        api.userSteps.financeSteps().payCampaignSandbox(SUM_SPENT.floatValue(), Currency.RUB, campaignId1.intValue());
        log.info("Подождем изменение баланса кампании");
        Long expectedSum = SUM_SPENT_NO_VAT.bidLong().longValueForSum();
        ConditionFactories.SANDBOX_STATUS.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(subclient1, campaignBefore, expectedSum, true));
    }

    @Test
    public void payCampaignZeroSumTest() {
        log.info("Оплата кампании по кредитному договору нулевой суммой");
        PayCampElementMap payInfo = new PayCampElementMap(api.type())
                .withCampaignID(campaignId1.intValue())
                .withCurrency(Currency.RUB)
                .withSum(0f);
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaignsSandbox(
                new AxisError(354, AxisErrorDetails.INVALID_AMOUNT_SPECIFIED_FOR_CAMPAIGN, campaignId1),
                payInfo
        );
    }

    @Test
    public void payTwoCampaignsTest() {
        CampaignGetItem campaignBefore3 = api.userSteps.campaignSteps().campaignsGet(
                subclient3, campaignId3, CampaignFieldEnum.FUNDS, CampaignFieldEnum.ID);
        CampaignGetItem campaignBefore2 = api.userSteps.campaignSteps().campaignsGet(
                subclient2, campaignId2, CampaignFieldEnum.FUNDS, CampaignFieldEnum.ID);

        PayCampElementMap payCampaign1 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId3.intValue())
                .withCurrency(Currency.RUB)
                .withSum(SUM_SPENT.floatValue());
        PayCampElementMap payCampaign2 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId2.intValue())
                .withCurrency(Currency.RUB)
                .withSum(SUM_SPENT.floatValue());

        log.info("Оплата двух кампании по кредитному договору");
        api.userSteps.financeSteps().payCampaignSandbox(payCampaign1, payCampaign2);
        log.info("Подождем изменение баланса кампании: " + campaignId3);
        Long expectedSum = SUM_SPENT_NO_VAT.bidLong().longValueForSum();
        ConditionFactories.SANDBOX_STATUS.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(subclient3, campaignBefore3, expectedSum, true));
        ConditionFactories.SANDBOX_STATUS.until(
                api.userSteps.campaignSteps().campaignFundsChangedOnFunds(subclient2, campaignBefore2, expectedSum, true));
    }

    @Test
    public void severalPaysOneCampaignTest() {
        log.info("Несколько платежей на кампанию в одном запросе");
        PayCampElementMap pay1 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId1.intValue())
                .withCurrency(Currency.RUB)
                .withSum(SUM_SPENT.floatValue());
        PayCampElementMap pay2 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId1.intValue())
                .withCurrency(Currency.RUB)
                .withSum(SUM_SPENT.floatValue());

        api.userSteps.financeSteps().shouldGetErrorOnPayCampaignsSandbox(
                new AxisError(354, AxisErrorDetails.PAYCAMPELEMENT_CAMPAIGNID_MUST_BE_UNIQUE),
                pay1, pay2
        );
    }

    @Test
    public void payNonExistingCampaignTest() {
        log.info("Попытка оплаты несуществующей кампании");
        PayCampElementMap pay1 = new PayCampElementMap(api.type())
                .withCampaignID(0)
                .withCurrency(Currency.RUB)
                .withSum(SUM_SPENT.floatValue());
        PayCampElementMap pay2 = new PayCampElementMap(api.type())
                .withCampaignID(campaignId2.intValue())
                .withCurrency(Currency.RUB)
                .withSum(SUM_SPENT.floatValue());

        api.userSteps.financeSteps().shouldGetErrorOnPayCampaignsSandbox(
                new AxisError(1, null),
                pay1, pay2
        );
    }

    @Test
    public void payCampaignInvalidContractIDTest() {
        String invalidContractId = "00000/00";
        PayCampaignsInfoMap payInfoMap = new PayCampaignsInfoMap(api.type())
                .withPayMethod(PayMethod.BANK)
                .withContractID(invalidContractId)
                .withPayments(new PayCampElementMap(api.type())
                        .withCampaignID(campaignId2.intValue())
                        .withCurrency(Currency.RUB)
                        .withSum(SUM_SPENT.floatValue()));

        log.info("Попытка оплаты кампании несуществующим договором");
        api.userSteps.financeSteps().shouldGetErrorOnPayCampaignsSandbox(
                new AxisError(71, AxisErrorDetails.FIELD_COMPLETED_INCORRECTLY, StringUtils.capitalize(PayCampaignsInfoMap.CONTRACT_ID)),
                payInfoMap
        );
    }

    @AfterClass
    public static void closeSandbox() {
        api.userSteps.sandboxSteps().closeSandbox(agency);
    }
}
