package ru.yandex.autotests.directapi.sandbox.v5;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.ads.AdFieldEnum;
import com.yandex.direct.api.v5.ads.AdGetItem;
import com.yandex.direct.api.v5.ads.GetResponse;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.general.StatusEnum;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.httpclient.data.sandbox.APISandboxClientTypeEnum;
import ru.yandex.autotests.directapi.ApiFeatures;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.ads.AdAddItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdGetItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdUpdateItemMap;
import ru.yandex.autotests.directapi.model.api5.ads.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.AdsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.ads.ArchiveRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdAddMap;
import ru.yandex.autotests.directapi.model.api5.ads.TextAdUpdateMap;
import ru.yandex.autotests.directapi.model.api5.ads.UpdateRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.sandbox.SandboxLogins;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 05.05.15.
 * https://st.yandex-team.ru/TESTIRT-5383
 */
@Aqua.Test
@Features(ApiFeatures.SANDBOX)
@Issue("https://st.yandex-team.ru/DIRECT-38497")
@Description("Проверка работоспособности сервиса Ads в песочнице")
public class AdsTest {

    private static final String LOGIN = SandboxLogins.SINGLE_CLIENT_SANDBOX_7;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static long adToArchiveID;
    private static long adToDeleteID;
    private static long adToGetID;
    private static long adToModerateID;
    private static long adToSuspendID;
    private static long adToUnarchiveID;
    private static long adToUpdateID;
    private static long adToResumeID;

    @BeforeClass
    public static void initSandBox() {
        api.userSteps.sandboxSteps().enableSandboxByApi(LOGIN, APISandboxClientTypeEnum.CLIENT, true);
        List<CampaignGetItem> campaigns = api.userSteps.campaignSteps().getCampaigns(
                new ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap()
                        .withSelectionCriteria(new CampaignsSelectionCriteriaMap())
                        .withAllFieldNames()
        );
        Long[] campaignIDs = campaigns.stream().mapToLong(campaign->campaign.getId()).boxed().toArray(Long[]::new);
        List<AdGetItem> ads = api.userSteps.adsSteps().adsGet(
                new GetRequestMap()
                        .withSelectionCriteria(new AdsSelectionCriteriaMap().withCampaignIds(campaignIDs))
                        .withFieldNames(AdFieldEnum.ID, AdFieldEnum.STATUS)
        ).getAds();
        Long[] draftAdId = ads.stream()
                .filter(ad -> ad.getStatus() == StatusEnum.DRAFT).mapToLong(ad -> ad.getId()).boxed().toArray(Long[]::new);
        assumeThat("пользователь имеет не менее 4 объявлений-черновиков", draftAdId.length, greaterThanOrEqualTo(4));
        adToDeleteID = draftAdId[0];
        adToModerateID = draftAdId[1];
        adToGetID = draftAdId[2];
        adToUpdateID = draftAdId[3];

        Long[] moderatedAdId = ads.stream()
                .filter(ad -> ad.getStatus() == StatusEnum.ACCEPTED).mapToLong(ad -> ad.getId()).boxed().toArray(Long[]::new);
        assumeThat("пользователь имеет не менее 4 промодерированных объявлений", moderatedAdId.length, greaterThanOrEqualTo(4));
        adToArchiveID = moderatedAdId[0];
        api.userSteps.adsSteps().adsSuspend(adToArchiveID);
        adToSuspendID = moderatedAdId[1];
        adToResumeID = moderatedAdId[2];
        api.userSteps.adsSteps().adsSuspend(adToResumeID);
        adToUnarchiveID = moderatedAdId[3];
        api.userSteps.adsSteps().adsArchive(adToUnarchiveID);
    }

    @Test
    @Description("Создание объявления в песочнице")
    public void addAdInSandbox() {
        Long campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        Long adGroupId = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
        api.userSteps.adsSteps().addDefaultTextAd(adGroupId);

        api.userSteps.adsSteps().shouldGetResultOnAdd(new AddRequestMap()
                        .withAds(new AdAddItemMap()
                                .withAdGroupId(adGroupId)
                                .withTextAd(new TextAdAddMap()
                                        .defaultTextAd())),
                ExpectedResult.success());
    }

    @Test
    @Description("Архивирование объявления в песочнице")
    public void archiveAdInSandbox() {
        api.userSteps.adsSteps().shouldGetResultOnArchive(new ArchiveRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(adToArchiveID)),
                ExpectedResult.success(adToArchiveID));
    }

    @Test
    @Description("Удаление объявления в песочнице")
    public void deleteAdInSandbox() {
        api.userSteps.adsSteps().shouldGetResultOnDelete(new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(adToDeleteID)),
                ExpectedResult.success(adToDeleteID));
    }

    @Test
    @Description("Получение объявления в песочнице")
    public void getAdInSandbox() {
        GetResponse response = api.userSteps.adsSteps().adsGet(new GetRequestMap()
                .withFieldNames(AdFieldEnum.ID)
                .withSelectionCriteria(new AdsSelectionCriteriaMap()
                        .withIds(adToGetID)));
        assertThat("объявление было успешно получено", response.getAds(), beanEquivalent(Arrays.asList(
                (AdGetItem) new AdGetItemMap()
                        .withId(adToGetID).getBean())));
    }

    @Test
    @Description("Отправка объявления на модерацию в песочнице")
    public void moderateAdInSandbox() {
        api.userSteps.adsSteps().shouldGetResultOnModerate(new IdsCriteriaMap()
                .withIds(adToModerateID), ExpectedResult.success(adToModerateID));
    }

    @Test
    @Description("Запуск объявления в песочнице")
    public void resumeAdInSandbox() {
        api.userSteps.adsSteps().shouldGetResultOnResume(new IdsCriteriaMap()
                        .withIds(adToResumeID),
                ExpectedResult.success(adToResumeID));
    }

    @Test
    @Description("Остановка объявления в песочнице")
    public void suspendAdInSandbox() {
        api.userSteps.adsSteps().shouldGetResultOnSuspend(new IdsCriteriaMap()
                        .withIds(adToSuspendID),
                ExpectedResult.success(adToSuspendID));
    }

    @Test
    @Description("Разархивация объявления в песочнице")
    public void unarchiveAdInSandbox() {
        api.userSteps.adsSteps().shouldGetResultOnUnarchive(new IdsCriteriaMap()
                        .withIds(adToUnarchiveID),
                ExpectedResult.success(adToUnarchiveID));
    }

    @Test
    @Description("Обновить объявление в песочнице")
    public void updateAdInSandbox() {
        api.userSteps.adsSteps().shouldGetResultOnUpdate(new UpdateRequestMap()
                        .withAds(new AdUpdateItemMap()
                                .withId(adToUpdateID)
                                .withTextAd(new TextAdUpdateMap()
                                        .defaultAd())),
                ExpectedResult.success(adToUpdateID));
    }

    @AfterClass
    public static void closeSandbox() {
        api.userSteps.sandboxSteps().closeSandboxByApi(LOGIN);
    }
}
