package ru.yandex.autotests.directapi.sandbox.v5;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.httpclient.data.sandbox.APISandboxClientTypeEnum;
import ru.yandex.autotests.directapi.ApiFeatures;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.IdsCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.keywords.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.DeleteRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordAddMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordGetItemMap;
import ru.yandex.autotests.directapi.model.api5.keywords.KeywordsSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.api5.keywords.ResumeRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywords.SuspendRequestMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.sandbox.SandboxLogins;
import ru.yandex.autotests.irt.testutils.RandomUtils;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created by semkagtn on 05.05.15.
 * https://st.yandex-team.ru/TESTIRT-5383
 */
@Aqua.Test
@Features(ApiFeatures.SANDBOX)
@Issue("https://st.yandex-team.ru/DIRECT-38497")
@Description("Проверка работоспособности сервиса Keywords в песочнице")
public class KeywordsTest {

    private static final String LOGIN = SandboxLogins.SINGLE_CLIENT_SANDBOX_10;

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    private static Long campaignID;
    private static Long adGroupID;

    @BeforeClass
    public static void initSandBox() {
        api.userSteps.sandboxSteps().enableSandboxByApi(LOGIN, APISandboxClientTypeEnum.CLIENT, false);
        campaignID = api.userSteps.campaignSteps().addDefaultTextCampaign(LOGIN);
        adGroupID = api.userSteps.adGroupsSteps().addDefaultGroup(campaignID);
    }

    @Test
    @Description("Создание ключевого слова в песочнице")
    public void addKeywordInSandbox() {
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.ADD,
                new AddRequestMap()
                        .withKeywords(new KeywordAddMap()
                                .withKeyword(RandomUtils.getString(7))
                                .withAdGroupId(adGroupID)),
                ExpectedResult.success());
    }

    @Test
    @Description("Удаление ключевого слова в песочнице")
    public void deleteKeywordInSandbox() {
        Long keywordForDeleteId = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.DELETE,
                new DeleteRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(keywordForDeleteId)),
                ExpectedResult.success(keywordForDeleteId));
    }

    @Test
    @Description("Получение ключевого слова в песочнице")
    public void getKeywordInSandbox() {
        Long keywordID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        List<KeywordGetItem> keywords = api.userSteps.keywordsSteps().keywordsGet(new KeywordsSelectionCriteriaMap()
                .withIds(keywordID));
        assertThat("ключевое слово успешно получено", keywords, beanEquivalent(Arrays.asList(
                (KeywordGetItem) new KeywordGetItemMap()
                        .withId(keywordID).getBean())));
    }

    @Test
    @Description("Запуск ключевого слова в песочнице")
    public void resumeKeywordInSandbox() {
        Long keywordForResumeID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        api.userSteps.keywordsSteps().keywordsSuspend(keywordForResumeID);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.RESUME,
                new ResumeRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(keywordForResumeID)),
                ExpectedResult.success(keywordForResumeID));
    }

    @Test
    @Description("Остановка ключевого слова в песочнице")
    public void suspendKeywordInSandbox() {
        Long keywordForSuspendID = api.userSteps.keywordsSteps().addDefaultKeyword(adGroupID);
        api.userSteps.keywordsSteps().shouldGetResultOn(
                Action.SUSPEND,
                new SuspendRequestMap()
                        .withSelectionCriteria(new IdsCriteriaMap()
                                .withIds(keywordForSuspendID)),
                ExpectedResult.success(keywordForSuspendID));
    }

    @AfterClass
    public static void closeSandbox() {
        api.userSteps.sandboxSteps().closeSandboxByApi(LOGIN);
    }
}
