package ru.yandex.autotests.directapi.stat.createforecast;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.model.ShardNumbers;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45.BannerPhraseInfo;
import ru.yandex.autotests.directapi.common.api45.ForecastStatusInfo;
import ru.yandex.autotests.directapi.common.api45.GetForecastInfo;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.forecast.NewForecastInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.directapi.stat.StatLogins;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.Arrays;
import java.util.List;

import static ch.lambdaj.Lambda.extract;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.assertThat;

/**
 * Created by chicos on 07.03.14.
 * https://jira.yandex-team.ru/browse/TESTIRT-1595
 */
@Aqua.Test
@Features(StatFeatures.CREATE_NEW_FORECAST)
@Stories(ApiStories.SHARDING)
public class CreateNewForecastShardingTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static final String clientShard2 = StatLogins.CLIENT_FORECAST_SHARD_2;
    private static int forecastIdToDelete;

    private static int forecastId1;
    private static int forecastId2;
    private final int fakeForecastId = 51174941;

    private static final String[] phrases = new String[]{"котики", "мимими"};

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(clientShard2);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void reshardUser() {
        api.userSteps.clientFakeSteps().enableAPI(StatLogins.CLIENT_FORECAST_SHARD_2);
        api.userSteps.clientFakeSteps().reshardUserWithLock(StatLogins.CLIENT_FORECAST_SHARD_2, ShardNumbers.EXTRA_SHARD);
    }

    @BeforeClass
    public static void createRequiredForecasts() {
        api.as(Logins.SUPER_LOGIN, StatLogins.CLIENT_FORECAST_SHARD_2)
                .userSteps.forecastSteps().deleteForecastReports();
        forecastId1 = api.userSteps.forecastSteps().createNewForecast(
                new NewForecastInfoMap(api.type())
                        .withPhrases(phrases)
                        .withCurrency(Currency.RUB.value())
        );

        forecastId2 = api.userSteps.forecastSteps().createNewForecast(
                new NewForecastInfoMap(api.type())
                        .withPhrases("спелеология")
                        .withCurrency(Currency.RUB.toString())
        );

        forecastIdToDelete = api.userSteps.forecastSteps().createNewForecast(
                new NewForecastInfoMap(api.type())
                        .withPhrases(phrases)
                        .withCurrency(Currency.RUB.value())
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7039")
    public void createForecast() {
        int reportId;
        NewForecastInfoMap reportInfo = new NewForecastInfoMap(api.type())
                .withCurrency(Currency.RUB.value())
                .withPhrases("8 марта");
        reportId = api.userSteps.forecastSteps().createNewForecast(reportInfo);
        assertThat("Вернулся некорректный идентификатор прогноза", reportId, not(equalTo(0)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7043")
    public void getForecast() {
        GetForecastInfo report = api.userSteps.forecastSteps().getForecast(forecastId1);
        List<String> phrasesReturned = extract(report.getPhrases(), on(BannerPhraseInfo.class).getPhrase());
        assertThat("Некорректный набор фраз", phrasesReturned, contains(phrases));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7040")
    public void getForecastNotExist() {
        api.userSteps.shouldGetErrorOn(
                Method.GET_FORECAST,
                fakeForecastId,
                new AxisError(73, AxisErrorDetails.NULL_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7041")
    public void deleteForecast() {
        //DIRECT-41059
        api.userSteps.forecastSteps().deleteForecast(forecastIdToDelete);
        api.userSteps.shouldGetErrorOn(
                Method.GET_FORECAST,
                forecastIdToDelete,
                new AxisError(73, AxisErrorDetails.NULL_STRING));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7042")
    public void getForecastList() {
        ForecastStatusInfo[] reports = api.userSteps.forecastSteps().getForecastList();
        List<Integer> reportIds = extract(Arrays.asList(reports), on(ForecastStatusInfo.class).getForecastID());
        assertThat("Некорректный набор прогнозов", reportIds, hasItems(
                forecastId1,
                forecastId2)
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7038")
    public void deleteForecastNotExist() {
        api.userSteps.shouldGetErrorOn(
                Method.DELETE_FORECAST_REPORT,
                fakeForecastId,
                new AxisError(24, AxisErrorDetails.NULL_STRING));
    }
}
