package ru.yandex.autotests.directapi.stat.createforecast;

import java.math.BigDecimal;
import java.util.List;

import ch.lambdaj.function.convert.PropertyExtractor;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45mng.BannerPhraseInfo;
import ru.yandex.autotests.directapi.common.api45mng.GetForecastInfo;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.banners.phrases.Categories;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.model.forecast.NewForecastInfoExtendedMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.convert;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.hasItems;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.closeTo;
import static ru.yandex.autotests.direct.utils.ForecastCalculator.getCTR;
import static ru.yandex.autotests.direct.utils.ForecastCalculator.getTotal;

/**
 * User: mariabye
 * Date: 06.08.12
 *
 * https://jira.yandex-team.ru/browse/TESTIRT-20
 */
@Aqua.Test
@Features(StatFeatures.NOT_REGRESSION_YET) // because of https://st.yandex-team.ru/DIRECT-60332
public class ForecastCommonTest {
    private static final BigDecimal ALLOWED_ERROR = new BigDecimal(0.01);

    protected LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class)
            .as(Logins.SUPER_LOGIN);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    private static GetForecastInfo getForecastInfo;

    @BeforeClass
    public static void getForecast() {
        NewForecastInfoExtendedMap newForecastInfoMap = new NewForecastInfoExtendedMap(api.type());
        newForecastInfoMap.withPhrases("новогодние подарки");
        newForecastInfoMap.withCategories(Categories.OKNA);
        newForecastInfoMap.withCurrency(Currency.RUB.value());

        getForecastInfo = api.userSteps.forecastSteps().getForecastSync(newForecastInfoMap);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7049")
    public void getReportPhraseTest() {
        List<String> phrases = convert(getForecastInfo.getPhrases(),
                new PropertyExtractor<Object, String>("phrase"));
        assertThat("Неверный отчет forecast", phrases, hasItems("новогодние подарки"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7050")
    public void getReportIsRubricTest() {
        assertThat("Неверный отчет forecast", getForecastInfo.getPhrases()[0].getIsRubric(), equalTo(Value.NO));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7051")
    public void getReportClicksTest() {
        assertThat("Ошибка в полях Clicks. Cуммарное количество кликов не совпадает с их фактической суммой",
                getForecastInfo.getCommon().getClicks(),
                equalTo(getForecastInfo.getPhrases()[0].getClicks()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7048")
    public void getReportFirstPlaceClicksTest() {
        assertThat("Ошибка в полях FirstPlaceClicks. Cуммарное количество кликов не совпадает с их фактической суммой",
                getForecastInfo.getCommon().getFirstPlaceClicks(),
                equalTo(getForecastInfo.getPhrases()[0].getFirstPlaceClicks()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7052")
    public void getReportPremiumClicksTest() {
        assertThat("Ошибка в полях PremiumClicks. Cуммарное количество кликов не совпадает с их фактической суммой",
                getForecastInfo.getCommon().getPremiumClicks(),
                equalTo(getForecastInfo.getPhrases()[0].getPremiumClicks()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7053")
    public void getReportShowsTest() {
        assertThat("Ошибка в полях Shows. Cуммарное количество показов не совпадает с их фактической суммой",
                getForecastInfo.getCommon().getShows(),
                equalTo(getForecastInfo.getPhrases()[0].getShows()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7054")
    public void getReportMinTest() {
        assertThat("Ошибка в полях Min. Поле Min в Common не соответствует отдельным полям во фразах и категориях",
                getForecastInfo.getCommon().getMin(),
                equalTo(
                        getTotal(
                                getForecastInfo.getPhrases()[0].getClicks(),
                                getForecastInfo.getPhrases()[0].getMin())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7055")
    public void getReportMaxTest() {
        BannerPhraseInfo firstPhrase = getForecastInfo.getPhrases()[0];
        BigDecimal premiumMax = new BigDecimal(getForecastInfo.getCommon().getMax());
        BigDecimal clicksToMax = new BigDecimal(getTotal(
                                                            firstPhrase.getFirstPlaceClicks(),
                                                            firstPhrase.getMax()));

        assertThat("Ошибка в полях Max. Поле Max в Common не соответствует отдельным полям во фразах и категориях",
                premiumMax,
                is(closeTo(clicksToMax,
                        ALLOWED_ERROR)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7056")
    public void getReportPremiumMinTest() {
        BannerPhraseInfo firstPhrase = getForecastInfo.getPhrases()[0];
        BigDecimal premiumMin = new BigDecimal(getForecastInfo.getCommon().getPremiumMin());
        BigDecimal clicksToMin = new BigDecimal(getTotal(
                                                            firstPhrase.getPremiumClicks(),
                                                            firstPhrase.getPremiumMin()));

        assertThat("Ошибка в полях Min. Поле Max в Common не соответствует отдельным полям во фразах и категориях",
                premiumMin,
                is(closeTo(clicksToMin,
                        ALLOWED_ERROR)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7057")
    public void getReportCTRTest() {
        assertThat("Ошибка в поле CTR во фразах",
                getForecastInfo.getPhrases()[0].getCTR(), equalTo(
                getCTR(getForecastInfo.getPhrases()[0].getClicks(), getForecastInfo.getPhrases()[0].getShows()))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7058")
    public void getReportFirstPlaceCTRTest() {
        assertThat("Ошибка в поле FirstPlaceCTR во фразах",
                getForecastInfo.getPhrases()[0].getFirstPlaceCTR(), equalTo(
                getCTR(
                        getForecastInfo.getPhrases()[0].getFirstPlaceClicks(),
                        getForecastInfo.getPhrases()[0].getShows()))
        );
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7059")
    public void getReportPremiumCTRTest() {
        assertThat("Ошибка в поле PremiumCTR во фразах",
                getForecastInfo.getPhrases()[0].getPremiumCTR(), equalTo(
                getCTR(
                        getForecastInfo.getPhrases()[0].getPremiumClicks(),
                        getForecastInfo.getPhrases()[0].getShows()))
        );
        log.info("Отчет создан верно");
    }

    /**
     * В ответе прогноза используется таже структура BannerPhraseInfo
     * что и для фраз баннера
     * Поэтому поле phraseID  в WSDL определяется как обязательное
     * При парсинге ответа в него по дефолту проставляется 0
     */
    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7060")
    public void couldNoPhraseIDFieldInReport() {
      assertThat("Поле PhraseID присутствует в отчете по фразе",
                getForecastInfo.getPhrases()[0].getPhraseID(), equalTo(0l));
    }

}

