package ru.yandex.autotests.directapi.stat.createforecast;

import java.util.Arrays;

import org.apache.commons.lang.RandomStringUtils;
import org.hamcrest.collection.IsArrayWithSize;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.PhraseTexts;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45.GetForecastInfo;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.banners.phrases.Categories;
import ru.yandex.autotests.directapi.model.forecast.NewForecastInfoMap;
import ru.yandex.autotests.directapi.model.forecast.NewWordstatReportInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.directapi.stat.StatLogins;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.directapi.steps.forecast.ForecastSteps;
import ru.yandex.autotests.directapi.utils.ErrorTextUtils;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static ch.lambdaj.Lambda.having;
import static ch.lambdaj.Lambda.on;
import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;

/**
 * User: mariabye
 * Date: 06.08.12
 *
 * https://jira.yandex-team.ru/browse/TESTIRT-20
 */
@Aqua.Test
@Features(StatFeatures.CREATE_NEW_FORECAST)
public class ForecastTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @BeforeClass
    public static void enableAPI(){
        api.userSteps.clientFakeSteps().enableAPI(StatLogins.CREATE_NEW_FORECAST_LOGIN);
        api.as(Logins.SUPER_LOGIN, StatLogins.CREATE_NEW_FORECAST_LOGIN)
                .userSteps.forecastSteps().deleteForecastReports();
    }

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104)
            .as(Logins.SUPER_LOGIN, StatLogins.CREATE_NEW_FORECAST_LOGIN)
            .protocol(ProtocolType.SOAP);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7064")
    public void couldCreateNewForecastWithOnePhrase() {
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        newForecastInfoMap.withPhrases("новогодние подарки");
        newForecastInfoMap.withCategories(new int[0]);
        newForecastInfoMap.withCurrency(Currency.RUB.value());
        log.info("Вызываем метод CreateNewForecast с одной фразой");

        GetForecastInfo getForecastInfo = api.userSteps.forecastSteps().getForecast(newForecastInfoMap);

        log.info("Получили идентификатор отчета. Проверяем правильность созданного отчета");
        assertThat("Неверный отчет forecast",
                getForecastInfo,
                allOf(
                        having(on(GetForecastInfo.class).getPhrases(), IsArrayWithSize.arrayWithSize(1))
                        , having(on(GetForecastInfo.class).getCategories(), IsArrayWithSize.emptyArray())
                ));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7065")
    public void couldCreateNewForecastWithOnePhraseWithCurrency() {
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        newForecastInfoMap.withPhrases("новогодние подарки");
        newForecastInfoMap.withCategories(new int[0]);
        newForecastInfoMap.withCurrency(Currency.RUB.toString());
        log.info("Вызываем метод CreateNewForecast с одной фразой");

        GetForecastInfo getForecastInfo = api.userSteps.forecastSteps().getForecast(newForecastInfoMap);

        log.info("Получили идентификатор отчета. Проверяем правильность созданного отчета");
        assertThat("Неверный отчет forecast",
                getForecastInfo,
                allOf(
                        having(on(GetForecastInfo.class).getPhrases(), IsArrayWithSize.arrayWithSize(1))
                        , having(on(GetForecastInfo.class).getCategories(), IsArrayWithSize.emptyArray())
                ));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7066")
    public void couldCreateNewForecastWithMassPhrasesAndMassCategories() {
        int phraseCount = 5;
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        newForecastInfoMap.withPhrases(PhraseTexts.getTexts(phraseCount));
        newForecastInfoMap.withCategories(new int[]{Categories.OKNA, Categories.BIT_TECHNIKA});
        newForecastInfoMap.withCurrency(Currency.RUB.value());
        log.info("Вызываем метод CreateNewForecast с массивом фраз и массивом категорий");

        GetForecastInfo getForecastInfo = api.userSteps.forecastSteps().getForecast(newForecastInfoMap);

        log.info("Получили идентификатор отчета. Проверяем правильность созданного отчета");
        assertThat("Неверный отчет forecast",
                getForecastInfo,
                allOf(
                        having(on(GetForecastInfo.class).getPhrases(), IsArrayWithSize.arrayWithSize(phraseCount))
                        , having(on(GetForecastInfo.class).getCategories(), IsArrayWithSize.emptyArray())
                ));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7067")
    public void couldNotCreateNewForecastWithEmptyArraysPhrasesAndCategories() {
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        newForecastInfoMap.withPhrases(new String[]{});
        newForecastInfoMap.withCategories(new int[]{});
        newForecastInfoMap.withCurrency(Currency.RUB.value());

        log.info("Вызываем метод CreateNewForecast с пустыми массивами фраз и категорий");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_FORECAST,
                newForecastInfoMap,
                new AxisError(71, AxisErrorDetails.PHRASES_FIELD_SHOULD_NOT_BE_AN_EMPTY_ARRAY,
                        NewForecastInfoMap.PHRASES, NewForecastInfoMap.CATEGORIES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7068")
    public void expectErrorCreateNewForecastWithIncorrectNumberOfPhrases() {
        String[] phrases = new String[ForecastSteps.NEW_FORECAST_REPORT_PHRASES_LIMIT + 1];
        Arrays.fill(phrases, "новогодние подраки");
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        newForecastInfoMap.withPhrases(phrases);
        newForecastInfoMap.withCategories(new int[]{});
        newForecastInfoMap.withCurrency(Currency.RUB.value());

        log.info("Вызываем метод CreateNewForecast с количеством фраз, превышающим допустимое значение");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_FORECAST,
                newForecastInfoMap,
                new AxisError(241, AxisErrorDetails.THE_ARRAY_SHOULD_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                        "'" + NewWordstatReportInfoMap.PHRASES + "'",
                        ForecastSteps.NEW_FORECAST_REPORT_PHRASES_LIMIT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7069")
    public void expectErrorCreateNewForecastWithMaxNumberOfPhrases() {
        String[] phrases = new String[ForecastSteps.NEW_FORECAST_REPORT_PHRASES_LIMIT];
        Arrays.fill(phrases, "новогодние подраки");
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        newForecastInfoMap.withPhrases(phrases);
        newForecastInfoMap.withCategories(new int[]{});
        newForecastInfoMap.withCurrency(Currency.RUB.value());

        GetForecastInfo getForecastInfo = api.userSteps.forecastSteps().getForecast(newForecastInfoMap);

        log.info("Получили идентификатор отчета. Проверяем правильность созданного отчета");
        assertThat("Неверный отчет forecast",
                getForecastInfo,
                having(on(GetForecastInfo.class).getPhrases(), IsArrayWithSize.arrayWithSize(100))
        );
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7071")
    public void expectErrorCreateNewForecastWithIncorrectCharacterInPhrase() {
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        newForecastInfoMap.withPhrases("некорректный символ #");
        newForecastInfoMap.withCurrency(Currency.RUB.value());

        log.info("Вызываем метод CreateNewForecast с фразой, содержащей некорректный символ");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_FORECAST,
                newForecastInfoMap,
                new AxisError(71, AxisErrorDetails.WORDSTAT_KEYWORD_PHRASES_INVALID, "некорректный символ #"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7072")
    public void expectErrorWithIncorrectLengthOfPhrase() {
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        String inputPhrase = RandomStringUtils.randomAlphabetic(4097);
        newForecastInfoMap.withPhrases(inputPhrase);
        newForecastInfoMap.withCurrency(Currency.RUB.value());

        log.info("Вызываем метод CreateNewForecast с фразой, длина которой превышает максимально допустимую");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_FORECAST,
                newForecastInfoMap,
                new AxisError(71, AxisErrorDetails.KEYWORD_PHRASE_LENGTH_HAS_BEEN_EXCEEDED, 4096
                        , ErrorTextUtils.getShortTextParameter(inputPhrase)));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7073")
    public void getReportGeoTest() {
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1, 2);
        newForecastInfoMap.withPhrases("новогодние подарки");
        String geoIDString = "1,2";
        newForecastInfoMap.withCurrency(Currency.RUB.value());
        log.info("Вызываем метод CreateNewForecast с двумя GeoID");
        GetForecastInfo getForecastInfo = api.userSteps.forecastSteps().getForecast(newForecastInfoMap);

        log.info("Получили идентификатор отчета. Проверяем правильность поля Geo в созданном отчете");
        assertThat("Ошибка в поле Geo", getForecastInfo.getCommon().getGeo(), equalTo(geoIDString));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7074")
    public void validateGetForecastResponseBySchema() {
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withGeoID(1);
        newForecastInfoMap.withPhrases("новогодние подарки");
        newForecastInfoMap.withCategories(new int[0]);
        newForecastInfoMap.withCurrency(Currency.RUB.value());
        int forecastId = api.userSteps.forecastSteps().createNewForecast(newForecastInfoMap);
        ConditionFactories.FORECAST.until(api.userSteps.forecastSteps().forecastIsReady(forecastId));
        api.userSteps.statisticsSteps().validateJsonResponseBySchema(Method.GET_FORECAST, forecastId);

    }

}

