package ru.yandex.autotests.directapi.stat.createforecast;

import ch.lambdaj.function.convert.PropertyExtractor;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.ForecastCalculator;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.direct.utils.matchers.BeanCompareStrategy;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.ApiStories;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45mng.ForecastCommonInfo;
import ru.yandex.autotests.directapi.common.api45mng.GetForecastInfo;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.banners.phrases.BannerPhraseInfoMap;
import ru.yandex.autotests.directapi.model.forecast.NewForecastInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.directapi.stat.StatLogins;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.matchers.IsCloseTo;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.security.SecureRandom;
import java.util.*;

import static ch.lambdaj.Lambda.convert;
import static org.apache.commons.lang.StringUtils.uncapitalize;
import static org.hamcrest.CoreMatchers.everyItem;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;

/**
 * User: pavryabov
 * Date: 29.12.13
 */
@Aqua.Test
@Features(StatFeatures.CREATE_NEW_FORECAST)
@Stories(ApiStories.CURRENCY)
@RunWith(Parameterized.class)
public class ForecastWithCurrencyTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static HashMap<Currency, GetForecastInfo> reports = new HashMap<>();
    private static List<Currency> currenciesToUse = getRandomCurrenciesList();

    @BeforeClass
    public static void enableAPI() {
        api.userSteps.getDarkSideSteps().getRunScriptSteps().runFillCurrencyRates();
        api.userSteps.clientFakeSteps().enableAPI(StatLogins.CLIENT_FREE_YE_DEFAULT);
    }

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class)
            .as(Logins.CLIENT_FREE_YE_DEFAULT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter(value = 0)
    public Currency currency;

    @Parameterized.Parameters(name = "Currency = {0}")
    public static Collection strategies() {
        List<Currency[]> currenciesWithArray = new ArrayList<>();
        for (Currency cur : currenciesToUse){
            currenciesWithArray.add(new Currency[]{cur});
        }
        return currenciesWithArray;
    }

    /**
     * Due to limitation of 5 forecast reports per user let's use randomizing not to lose test coverage
     *
     */
    private static List<Currency> getRandomCurrenciesList(){
        List<Currency> allCurrencies = Arrays.asList(
                Currency.RUB,
                Currency.USD,
                Currency.UAH,
                Currency.EUR,
                Currency.CHF,
                Currency.TRY,
                Currency.KZT,
                Currency.BYN
        );
        Collections.shuffle(allCurrencies, new SecureRandom());
        return allCurrencies.subList(0, 5);
    }

    @BeforeClass
    public static void createReports() {
        api.userSteps.forecastSteps().deleteForecastReports();

        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withPhrases("разгильдяй");
        newForecastInfoMap.withCategories(new int[0]);

        //create the last 5 reports with some currencies
        for (Currency currency : currenciesToUse) {
            newForecastInfoMap.withCurrency(currency.toString());
            reports.put(currency, api.userSteps.forecastSteps().getForecast(newForecastInfoMap));
        }
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7076")
    public void getReportWithCurrencyTest() {
        log.info("Проверка значения в поле Currency в отчете");
        List<String> currencies = convert(reports.get(currency).getPhrases(),
                new PropertyExtractor<Object, String>("currency"));

        assertThat("Неверный отчет forecast",
                currencies, everyItem(equalTo(currency.toString())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7077")
    public void checkForecastCommonTest() {
        ForecastCommonInfo expectedCommon = ForecastCalculator.calculate(reports.get(currency));
        BeanCompareStrategy strategy = new BeanCompareStrategy();
        strategy.putFieldMatcher(uncapitalize(BannerPhraseInfoMap.MIN),
                IsCloseTo.closeTo(expectedCommon.getMin(), 0.01f));
        strategy.putFieldMatcher(uncapitalize(BannerPhraseInfoMap.MAX),
                IsCloseTo.closeTo(expectedCommon.getMax(), 0.01f));
        strategy.putFieldMatcher(uncapitalize(BannerPhraseInfoMap.PREMIUM_MIN),
                IsCloseTo.closeTo(expectedCommon.getPremiumMin(), 0.01f));

        assertThat("Неверные значения итоговых данных отчета",
                reports.get(currency).getCommon(), beanEquals(expectedCommon).accordingStrategy(strategy));

    }

}
