package ru.yandex.autotests.directapi.stat.createnewwordstatreport;

import org.apache.commons.lang.RandomStringUtils;
import org.junit.*;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.directapi.stat.StatLogins;
import ru.yandex.autotests.directapi.utils.ErrorTextUtils;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.forecast.NewWordstatReportInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.hazelcast.SemaphoreRule;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.forecast.ForecastSteps;
import ru.yandex.qatools.allure.annotations.Features;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * User: mariabye
 * Date: 30.11.12
 *
 * https://jira.yandex-team.ru/browse/TESTIRT-70
 */
@Aqua.Test
@Features(StatFeatures.CREATE_NEW_WORDSTAT)
@RunWith(Parameterized.class)
public class CreateNewWordstatReporErrorsTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.SUPER_LOGIN, StatLogins.CREATE_NEW_WORDSTAT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @Parameterized.Parameter
    public int version;

    @Parameterized.Parameters
    public static Collection reportRequests() {
        List<Object[]> params = new ArrayList<Object[]>();
        params.add(new Object[]{4});
        params.add(new Object[]{104});
        return params;
    }

    @BeforeClass
    public static void enableAPI(){
        api.userSteps.clientFakeSteps().enableAPI(StatLogins.CREATE_NEW_WORDSTAT);
    }

    @Before
    public void setVersion(){
        api.version(version);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7153")
    public void expectErrorCreateNewWordstatReportWithIncorrectNumberOfPhrases() {
        String[] phrases = new String[ForecastSteps.NEW_WORDSTAT_REPORT_PHRASES_LIMIT + 1];
        Arrays.fill(phrases, "новогодние подраки");
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases(phrases);

        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_WORDSTAT_REPORT,
                reportInfo,
                new AxisError(241, AxisErrorDetails.THE_ARRAY_MUST_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                        NewWordstatReportInfoMap.PHRASES, ForecastSteps.NEW_WORDSTAT_REPORT_PHRASES_LIMIT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7155")
    public void expectErrorWithEmptyArrayOfPhrases() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases(new String[0]);
        log.info("Вызываем метод CreateNewWordstatReport с пустым массивом фраз");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_WORDSTAT_REPORT,
                reportInfo,
                new AxisError(30, AxisErrorDetails.THE_ARRAY_CANNOT_BE_EMPTY, NewWordstatReportInfoMap.PHRASES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7154")
    public void expectErrorWithNullArrayOfPhrases() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1);
        log.info("Вызываем метод CreateNewWordstatReport без массива фраз");

        //https://jira.yandex-team.ru/browse/DIRECT-24223
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_WORDSTAT_REPORT,
                reportInfo,
                new AxisError(71, AxisErrorDetails.FIELD_PHRASES_MUST_NOT_BE_EMPTY));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7156")
    public void expectErrorWithIncorrectCharacterInPhrase() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases("некорректный символ #");
        log.info("Вызываем метод CreateNewWordstatReport с фразой, содержащей некорректный символ");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_WORDSTAT_REPORT,
                reportInfo,
                new AxisError(71, AxisErrorDetails.WORDSTAT_KEYWORD_PHRASES_INVALID, "некорректный символ #"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7157")
    public void expectErrorWithIncorrectLengthOfPhrase() {
        String phrase = RandomStringUtils.randomAlphabetic(ForecastSteps.NEW_WORDSTAT_REPORT_PHRASES_LENGTH_LIMIT + 1);
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases(phrase);
        log.info("Вызываем метод CreateNewWordstatReport с фразой, содержащей некорректный символ");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_WORDSTAT_REPORT,
                reportInfo,
                new AxisError(71, AxisErrorDetails.KEYWORD_PHRASE_LENGTH_HAS_BEEN_EXCEEDED,
                        ForecastSteps.NEW_WORDSTAT_REPORT_PHRASES_LENGTH_LIMIT
                        , ErrorTextUtils.getShortTextParameter(phrase)));
    }


}
