package ru.yandex.autotests.directapi.stat.createnewwordstatreport;

import org.junit.*;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.PhraseTexts;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.forecast.NewWordstatReportInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.directapi.stat.StatLogins;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;

/**
 * User: mariabye
 * Date: 30.11.12
 *
 * https://jira.yandex-team.ru/browse/TESTIRT-70
 */
@Aqua.Test
@Features(StatFeatures.CREATE_NEW_WORDSTAT)
@RunWith(Parameterized.class)
public class CreateNewWordstatReportTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(Logins.SUPER_LOGIN, StatLogins.CREATE_NEW_WORDSTAT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);


    @Parameterized.Parameters
    public static Collection reportRequests() {
        List<Object[]> params = new ArrayList<Object[]>();
        params.add(new Object[]{4});
        params.add(new Object[]{104});
        return params;
    }

    @Parameterized.Parameter
    public int version;

    @BeforeClass
    public static void enableAPI(){
        api.userSteps.clientFakeSteps().enableAPI(StatLogins.CREATE_NEW_WORDSTAT);
    }

    @Before
    public void setVersion() {
        api.version(version);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7167")
    public void couldCreateNewWordstatReportWithOnePhrase() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases("новогодние подарки");
        log.info("Вызываем метод CreateNewWordstatReport с одной фразой");
        List<String> gotPhrases = api.userSteps.forecastSteps().getWordstatReportPhrases(reportInfo);
        assertThat("Неправильный порядок фраз в отчете", gotPhrases, equalTo(Arrays.asList(reportInfo.getPhrases())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7168")
    public void couldCreateNewWordstatReportWithMassPhrases() {
        String[] phrases = PhraseTexts.getTexts(5);
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases(phrases);

        List<String> gotPhrases = api.userSteps.forecastSteps().getWordstatReportPhrases(reportInfo);
        assertThat("Неправильный порядок фраз в отчете", gotPhrases, equalTo(Arrays.asList(phrases)));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7169")
    public void couldCreateNewWordstatReportWithNullGeoID() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withPhrases("новогодние подарки");
        log.info("Вызываем метод CreateNewWordstatReport с GeoID == null");
        List<String> gotPhrases = api.userSteps.forecastSteps().getWordstatReportPhrases(reportInfo);
        assertThat("Неправильный порядок фраз в отчете", gotPhrases, equalTo(Arrays.asList(reportInfo.getPhrases())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7170")
    public void couldCreateNewWordstatReportWithQuotesInPrases() {
        //DIRECT-40729
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases("\"новогодние подарки\"", "новог\'одние пода\'рки");
        log.info("Вызываем метод CreateNewWordstatReport с двумя фразами: фраза в двойных кавычках и в одинарных");
        List<String> gotPhrases = api.userSteps.forecastSteps().getWordstatReportPhrases(reportInfo);
        assertThat("Неправильный порядок фраз в отчете", gotPhrases, equalTo(Arrays.asList(reportInfo.getPhrases())));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7172")
    public void expectErrorWhenApostropheAtStartOfPhrase() {
        String phrase = "\'новогодние подарки\'";
        api.userSteps.forecastSteps().deleteWordstatReports();
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases(phrase);
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.CREATE_NEW_WORDSTAT_REPORT,
                reportInfo,
                new AxisError(71, AxisErrorDetails.PHRASE_CANT_START_WITH_DOT_OR_APOSTROPHE, phrase));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7171")
    public void createNewWordstatReportOrderOfPhrasesTest() {
        String[] phrases = new String[]{
                "фраза 1",
                "фраза 2",
                "фраза 3",
                "фраза 4",
                "фраза 5",
                "фраза 6",
                "фраза 7",
                "фраза 8",
                "фраза 9",
                "фраза 10"
        };
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases(phrases);
        List<String> gotPhrases = api.userSteps.forecastSteps().getWordstatReportPhrases(reportInfo);
        assertThat("Неправильный порядок фраз в отчете", gotPhrases, equalTo(Arrays.asList(reportInfo.getPhrases())));

    }
}
