package ru.yandex.autotests.directapi.stat.getforecastsync;

import org.apache.commons.lang.RandomStringUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.utils.ErrorTextUtils;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.banners.phrases.Categories;
import ru.yandex.autotests.directapi.model.forecast.NewForecastInfoExtendedMap;
import ru.yandex.autotests.directapi.model.forecast.NewWordstatReportInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.qatools.hazelcast.SemaphoreRule;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.forecast.ForecastSteps;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.directapi.stat.StatLogins;
import ru.yandex.qatools.allure.annotations.Features;


import java.util.Arrays;

import static org.apache.commons.lang3.StringUtils.capitalize;

/**
 * User: ginger
 * Date: 09.04.13
 *
 * тесты на логику аналогично CreateNewForecast (не указан PeriodType)
 * https://jira.yandex-team.ru/browse/DIRECT-20989
 */

@Aqua.Test
@Features(StatFeatures.GET_FORECAST_SYNC)
public class GetForecastSyncErrorsTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class).as(Logins.MANAGER_DEFAULT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void enableAPI(){
        api.userSteps.clientFakeSteps().enableAPI(StatLogins.MANAGER_DEFAULT);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7268")
    public void expectErrorGetForecastSyncWithIncorrectNumberOfPhrases() {
        String[] phrases = new String[ForecastSteps.NEW_FORECAST_SYNC_REPORT_PHRASES_LIMIT + 1];
        Arrays.fill(phrases, "отдых в Италии");

        NewForecastInfoExtendedMap newForecastInfo = new NewForecastInfoExtendedMap(api.type())
                .withGeoID(1)
                .withCategories(new int[0])
                .withPhrases(phrases);
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_FORECAST_SYNC,
                newForecastInfo,
                new AxisError(241, AxisErrorDetails.THE_ARRAY_SHOULD_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                        "'" + NewWordstatReportInfoMap.PHRASES + "'", ForecastSteps.NEW_FORECAST_SYNC_REPORT_PHRASES_LIMIT));
    }


    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7269")
    public void expectErrorGetForecastSyncWithIncorrectCharacterInPhrase() {
        NewForecastInfoExtendedMap newForecastInfo = new NewForecastInfoExtendedMap(api.type())
                .withGeoID(1)
                .withPhrases("некорректный символ #");
        log.info("Вызываем метод GetForecastSync с фразой, содержащей некорректный символ");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_FORECAST_SYNC,
                newForecastInfo,
                new AxisError(71, AxisErrorDetails.WORDSTAT_KEYWORD_PHRASES_INVALID, "некорректный символ #"));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7267")
    public void expectErrorWithIncorrectLengthOfPhrase() {
        String inputPhrase = RandomStringUtils.randomAlphabetic(4097);
        NewForecastInfoExtendedMap newForecastInfo = new NewForecastInfoExtendedMap(api.type())
                .withGeoID(1)
                .withPhrases(inputPhrase);
        log.info("Вызываем метод GetForecastSync с фразой, длина которой превышает максимально допустимую");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_FORECAST_SYNC,
                newForecastInfo,
                new AxisError(71, AxisErrorDetails.KEYWORD_PHRASE_LENGTH_HAS_BEEN_EXCEEDED,
                        4096, ErrorTextUtils.getShortTextParameter(inputPhrase)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7270")
    public void expectedErrorGetForecastSyncWithEmptyArraysPhrasesAndCategories() {
        NewForecastInfoExtendedMap newForecastInfo = new NewForecastInfoExtendedMap(api.type())
                .withGeoID(1)
                .withPhrases(new String[0])
                .withCategories(new int[0]);
        log.info("Вызываем метод GetForecastSync с пустыми массивами фраз и категорий");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_FORECAST_SYNC,
                newForecastInfo,
                new AxisError(71, AxisErrorDetails.PHRASES_FIELD_SHOULD_NOT_BE_AN_EMPTY_ARRAY));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7271")
    public void expectedErrorGetForecastSyncWithNullPhrasesAndCategories() {
        NewForecastInfoExtendedMap newForecastInfo = new NewForecastInfoExtendedMap(api.type())
                .withGeoID(1);
        log.info("Вызываем метод GetForecastSync с пустыми массивами фраз и категорий");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_FORECAST_SYNC,
                newForecastInfo,
                new AxisError(71, AxisErrorDetails.PHRASES_FIELD_SHOULD_BE_SPECIFIED));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7272")
    public void expectedErrorGetForecastSyncWithEmptyStringPhrases() {
        NewForecastInfoExtendedMap newForecastInfo = new NewForecastInfoExtendedMap(api.type())
                .withGeoID(1)
                .withPhrases(" ");
        log.info("Вызываем метод GetForecastSync с пустыми массивами фраз и категорий");
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_FORECAST_SYNC,
                newForecastInfo,
                new AxisError(71, AxisErrorDetails.VALUE_FIELD_NOT_ENTERED,
                        capitalize(NewForecastInfoExtendedMap.PHRASES)));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7273")
    public void expectedErrorGetForecastSyncWithOneCategory() {
        NewForecastInfoExtendedMap newForecastInfo = new NewForecastInfoExtendedMap(api.type())
                .withGeoID(1)
                .withCategories(Categories.OKNA);

        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_FORECAST_SYNC,
                newForecastInfo,
                new AxisError(71, AxisErrorDetails.PHRASES_FIELD_SHOULD_BE_SPECIFIED));
    }
}
