package ru.yandex.autotests.directapi.stat.getwordstatsync;

/**
 * User: mariabye
 * Date: 29.10.12
 *
 * https://jira.yandex-team.ru/browse/DIRECT-18934
 */

import org.apache.commons.lang.RandomStringUtils;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorDetails;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45mng.WordstatReportInfo;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.forecast.NewWordstatReportInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.banners.KeywordPhraseGenerator;
import ru.yandex.autotests.directapi.steps.banners.KeywordsSteps;
import ru.yandex.autotests.directapi.steps.forecast.ForecastSteps;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.directapi.stat.StatLogins;
import ru.yandex.autotests.directapi.utils.ErrorTextUtils;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import java.util.Arrays;

import static org.hamcrest.Matchers.arrayWithSize;
import static org.hamcrest.Matchers.equalTo;

@Aqua.Test
@Features(StatFeatures.GET_WORDSTAT_SYNC)
public class GetWordstatSyncTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @ClassRule
    public static ApiSteps api = new ApiSteps().version(104).wsdl(APIPort_PortType.class).as(Logins.MANAGER_DEFAULT);

    @ClassRule
    public static SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trasher = new Trashman(api);

    @BeforeClass
    public static void enableAPI(){
        api.userSteps.clientFakeSteps().enableAPI(StatLogins.MANAGER_DEFAULT);
        api.as(Logins.SUPER_LOGIN, StatLogins.MANAGER_DEFAULT).userSteps.forecastSteps().deleteWordstatReports();
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7332")
    public void getWordstatSyncForAWord() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withPhrases("подвеска -автомобиль");
        api.userSteps.forecastSteps().compareWordstatToSyncReport(reportInfo, 30, 30, true);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7333")
    public void getWordstatSyncForEmptyGeo() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(new int[0])
                .withPhrases("подвеска -автомобиль");
        api.userSteps.forecastSteps().compareWordstatToSyncReport(reportInfo, 30, 30, true);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7330")
    public void getWordstatSyncForGeo() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(215, 219)
                .withPhrases("москва -столица");

        api.userSteps.forecastSteps().compareWordstatToSyncReport(reportInfo, 30, 30, true);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7334")
    public void getWordstatSyncForNegativeGeo() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(-215, 219)
                .withPhrases("москва -столица");
        api.userSteps.forecastSteps().compareWordstatToSyncReport(reportInfo, 30, 30, true);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7331")
    public void getWordstatSyncForNull() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type());
        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_WORDSTAT_SYNC,
                reportInfo,
                new AxisError(71, AxisErrorDetails.FIELD_MUST_NOT_BE_EMPTY, NewWordstatReportInfoMap.PHRASES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7335")
    public void getWordstatSyncForEmptyPhrases() {
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withPhrases(new String[0]);

        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_WORDSTAT_SYNC,
                reportInfo,
                new AxisError(30, AxisErrorDetails.THE_ARRAY_CANNOT_BE_EMPTY, NewWordstatReportInfoMap.PHRASES));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7336")
    public void getWordStatSyncForOverLimitPhrasesCount() {
        String[] phrases = new String[ForecastSteps.NEW_WORDSTAT_SYNC_REPORT_PHRASES_LIMIT + 1];
        Arrays.fill(phrases, "новогодние подарки");
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases(phrases);

        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_WORDSTAT_SYNC,
                reportInfo,
                new AxisError(241, AxisErrorDetails.THE_ARRAY_MUST_CONTAIN_NO_MORE_THAN_N_ELEMENTS,
                        NewWordstatReportInfoMap.PHRASES, ForecastSteps.NEW_WORDSTAT_SYNC_REPORT_PHRASES_LIMIT));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7337")
    @Ignore("https://st.yandex-team.ru/DIRECT-167449")
    public void getWordstatSyncMatchesAsyncHyphenPhraseTest() {
        String hyphenPhrase = "юго-запад";
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withGeoID(1)
                .withPhrases(hyphenPhrase);
        api.userSteps.forecastSteps().compareWordstatToSyncReport(reportInfo, 30, 30, true);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7338")
    public void getWordstatSyncHyphenPhraseTest() {
        //https://st.yandex-team.ru/DIRECT-35408
        String hyphenPhrase = "Нью-Йорк";
        WordstatReportInfo[] reportInfo = api.userSteps.forecastSteps().getWordstatSyncReport(
                new NewWordstatReportInfoMap(api.type())
                        .withGeoID(1)
                        .withPhrases(hyphenPhrase)
        );
        TestSteps.assertThat("фраза корректна", reportInfo[0].getPhrase(), equalTo(hyphenPhrase));
    }

    @Test
    @Description("Проверка того, что отчёт успевает построиться для фразы \"фото\"")
    @Issue("https://st.yandex-team.ru/DIRECT-40320")
    @ru.yandex.qatools.allure.annotations.TestCaseId("7339")
    public void getWordstatSyncPhrasePhoto() throws InterruptedException {
        String phrase = "фото";
        WordstatReportInfo[] reportInfo = api.userSteps.forecastSteps().getWordstatSyncReport(
                new NewWordstatReportInfoMap(api.type())
                        .withGeoID(1)
                        .withPhrases(phrase));
        TestSteps.assertThat("отчёт успел построиться", reportInfo, arrayWithSize(1));
    }

    @Test
    public void getWordstatSyncReportMaxSymbolsNotExceed() {
        KeywordPhraseGenerator keywordPhraseGenerator =
                new KeywordPhraseGenerator(KeywordsSteps.CYRILLIC_CHARS_NO_SPACE);

        String phrase = keywordPhraseGenerator
                .generatePhraseOfSpecificLength(
                        new Integer[]{KeywordsSteps.ACCEPTABLE_KEYWORD_LENGTH},
                        ForecastSteps.NEW_WORDSTAT_SYNC_REPORT_LENGTH_LIMIT,
                        KeywordsSteps.ACCEPTABLE_KEYWORD_LENGTH)
                .toString();

        WordstatReportInfo[] reportInfo = api.userSteps.forecastSteps().getWordstatSyncReport(
                new NewWordstatReportInfoMap(api.type())
                        .withPhrases(phrase));

        TestSteps.assertThat("ограничение на длину фразы работает", reportInfo, arrayWithSize(1));
    }

    @Test
    public void getWordstatSyncReportMaxSymbolsExceed() {
        String phrase = RandomStringUtils.randomAlphabetic(ForecastSteps.NEW_WORDSTAT_SYNC_REPORT_LENGTH_LIMIT + 1);
        NewWordstatReportInfoMap reportInfo = new NewWordstatReportInfoMap(api.type())
                .withPhrases(phrase);

        api.userSteps.forecastSteps().shouldGetErrorOn(
                Method.GET_WORDSTAT_SYNC,
                reportInfo,
                new AxisError(71, AxisErrorDetails.THE_MAXIMUM_STRING_LENGTH_OF_N_CHARACTERS_IN_THE_KEYWORD_PHRASE,
                        ForecastSteps.NEW_WORDSTAT_SYNC_REPORT_LENGTH_LIMIT
                        , ErrorTextUtils.getShortTextParameter(phrase)));
    }
}
