package ru.yandex.autotests.directapi.stat.production;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.utils.ForecastCalculator;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.direct.utils.matchers.BeanCompareStrategy;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.ApiFeatures;
import ru.yandex.autotests.directapi.common.api45mng.ForecastCommonInfo;
import ru.yandex.autotests.directapi.common.api45mng.APIPort_PortType;
import ru.yandex.autotests.directapi.common.api45mng.GetForecastInfo;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.banners.phrases.BannerPhraseInfoMap;
import ru.yandex.autotests.directapi.model.forecast.NewForecastInfoMap;
import ru.yandex.autotests.directapi.rules.ApiSteps;
import ru.yandex.autotests.directapi.stat.StatFeatures;
import ru.yandex.autotests.irt.testutils.matchers.IsCloseTo;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Stories;

import java.security.SecureRandom;
import java.util.*;

import static org.apache.commons.lang.StringUtils.uncapitalize;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.direct.utils.matchers.BeanEquals.beanEquals;

/**
 * Created by chicos on 28.04.14.
 * <p/>
 * Smoke-тесты на проверку некоторых методов на production окружении
 * https://jira.yandex-team.ru/browse/TESTIRT-1800
 */
@Aqua.Test
@Features(ApiFeatures.PRODUCTION)
@Stories(StatFeatures.CREATE_NEW_FORECAST)
public class CreateNewForecastProdTest {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    private static String clientLogin = Logins.CLIENT_SMOKE_PROD;
    private static GetForecastInfo forecastReport;
    private static int forecastReportId;
    private static Currency currency = getRandomCurrency();

    @ClassRule
    public static ApiSteps api = new ApiSteps().as(clientLogin).version(104).wsdl(APIPort_PortType.class);

    /**
     * Get one random Currency to test
     */
    private static Currency getRandomCurrency(){
        List<Currency> currenciesList = Arrays.asList(
                Currency.RUB,
                Currency.USD,
                Currency.UAH,
                Currency.EUR,
                Currency.CHF,
                Currency.TRY,
                Currency.KZT
        );
        Collections.shuffle(currenciesList, new SecureRandom());
        return currenciesList.get(0);
    }

    @BeforeClass
    public static void createReports() {
        //create one report with random currency
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(api.type());
        newForecastInfoMap.withPhrases("разгильдяй");
        newForecastInfoMap.withCategories(new int[0]);
        newForecastInfoMap.withCurrency(currency.value());

        forecastReportId = api.userSteps.forecastSteps().createNewForecast(newForecastInfoMap);
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7341")
    public void getForecastReportTest() {
        forecastReport = api.userSteps.forecastSteps().getForecast(forecastReportId);
        log.info("Проверка значения поля Currency в отчете");

        assertThat("Неверный отчет forecast",
                forecastReport.getPhrases()[0].getCurrency(), equalTo(currency.value()));
    }

    @Test
    @ru.yandex.qatools.allure.annotations.TestCaseId("7340")
    public void checkForecastCommonTest() {
        forecastReport = api.userSteps.forecastSteps().getForecast(forecastReportId);
        ForecastCommonInfo expectedCommon = ForecastCalculator.calculate(forecastReport);
        BeanCompareStrategy strategy = new BeanCompareStrategy();
        strategy.putFieldMatcher(uncapitalize(BannerPhraseInfoMap.MIN),
                IsCloseTo.closeTo(expectedCommon.getMin(), 0.01f));
        strategy.putFieldMatcher(uncapitalize(BannerPhraseInfoMap.MAX),
                IsCloseTo.closeTo(expectedCommon.getMax(), 0.01f));
        strategy.putFieldMatcher(uncapitalize(BannerPhraseInfoMap.PREMIUM_MIN),
                IsCloseTo.closeTo(expectedCommon.getPremiumMin(), 0.01f));

        assertThat("Неверные значения итоговых данных отчета",
                forecastReport.getCommon(), beanEquals(expectedCommon).accordingStrategy(strategy));
    }

    @AfterClass
    public static void deleteReportsManually(){
        api.userSteps.forecastSteps().deleteForecastReports();
    }
}
