package ru.yandex.autotests.directapi.apiclient.balancesimple;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.databind.ObjectMapper;
import freemarker.cache.ClassTemplateLoader;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.DefaultObjectWrapper;
import freemarker.template.Template;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Consts;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.autotests.direct.utils.config.balancesimple.BalanceSimpleEnvironment;
import ru.yandex.autotests.directapi.apiclient.HttpMessage;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.model.balancesimple.request.BalanceSimpleRequest;
import ru.yandex.autotests.httpclient.lite.core.config.HttpClientConnectionConfig;
import ru.yandex.autotests.httpclient.lite.core.config.HttpClientFactory;
import ru.yandex.qatools.allure.annotations.Attachment;

/**
 * Created by semkagtn on 15.09.15.
 */
public class BalanceSimpleClient {

    public static Log log = LogFactory.getLog(BalanceSimpleClient.class);

    private static ObjectMapper objectMapper = new ObjectMapper();

    private String token;
    private BalanceSimpleEnvironment environment;

    public BalanceSimpleClient(String token, BalanceSimpleEnvironment environment) {
        this.token = token;
        this.environment = environment;
    }

    public <T extends BalanceSimpleRequest> String sendRequest(String methodName, T request) {
        String fullUrl = environment.getUrl() + methodName;

        request.setToken(token);
        Map<String, Object> mapRequestParams = new HashMap<>();
        mapRequestParams.put("params", request);
        String requestParams;
        try {
            requestParams = objectMapper.writeValueAsString(mapRequestParams);
        } catch (IOException e) {
            throw new DirectAPIException("Не удалось преобразовать запрос к формату JSON", e);
        }

        HttpMessage httpMessage = new HttpMessage();
        httpMessage.setRequest(requestParams);
        httpMessage.setUrl(fullUrl);

        HttpClientConnectionConfig clientConnectionConfig =
                ConnectionConfig.getHttpClientConnectionConfig(fullUrl).timeout(360);
        CloseableHttpClient httpClient = HttpClientFactory.getHttpClient(clientConnectionConfig);

        HttpPost httpRequest = new HttpPost();
        StringEntity entity = new StringEntity(requestParams,
                ContentType.create("application/x-www-form-urlencoded", Consts.UTF_8));
        httpRequest.setEntity(entity);
        httpRequest.setURI(URI.create(fullUrl));

        String result = null;
        InputStream responseStream = null;
        try {
            HttpResponse response = httpClient.execute(httpRequest);
            StatusLine status = response.getStatusLine();
            httpMessage.setStatusCode(String.valueOf(status.getStatusCode()));
            if (status.getStatusCode() != HttpStatus.SC_OK) {
                httpMessage.setResponse(status.getReasonPhrase());
                traceHttpMessage(httpMessage);
                throw new DirectAPIException("Сервер ответил ошибкой");
            }
            responseStream = response.getEntity().getContent();
            result = IOUtils.toString(responseStream, Consts.UTF_8.toString());
            httpMessage.setResponse(result);
            traceHttpMessage(httpMessage);
        } catch (IOException e) {
            traceHttpMessage(httpMessage);
            throw new DirectAPIException("Неизвестная ошибка при получении ответа");
        } finally {
            if (responseStream != null) {
                try {
                    responseStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return result;
    }

    @Attachment(value = "[JSON/CURL]: Request", type = "text/html")
    public String traceHttpMessage(HttpMessage message) {
        log.info(message.toString());
        Map<String, Object> root = new HashMap<>();
        root.put("usualFormatString", message.toString());
        root.put("curlString", message.toCurlStringForJson());
        root.put("buttonName", "JSON");

        String attachmentHTML;
        try {
            TemplateLoader file = new ClassTemplateLoader(this.getClass(), "/");
            Configuration cfg = new Configuration();
            cfg.setTemplateLoader(file);
            cfg.setObjectWrapper(new DefaultObjectWrapper());
            cfg.setDefaultEncoding("UTF-8");
            Template template = cfg.getTemplate("request2curlTemplate.ftl");
            Writer fileWriter = new StringWriter();
            try {
                template.process(root, fileWriter);
            } finally {
                fileWriter.close();
            }
            attachmentHTML = fileWriter.toString();
        } catch (Exception e) {
            throw new RuntimeException("Error", e);
        }

        return attachmentHTML;
    }

    public void setToken(String token) {
        this.token = token;
    }
}
