package ru.yandex.autotests.directapi.apiclient.errors;

import java.util.List;
import java.util.regex.Pattern;

import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

/**
 * Author pavryabov
 * Date 17.07.14
 */
public class Api5ErrorMatcher extends TypeSafeDiagnosingMatcher<Api5Error> {
    private final Api5Error expectedError;
    private final boolean isCodeInIgnoreList;
    private static final Pattern START_CODE_REGEXP = Pattern.compile("^#[0-9]*(: )?");
    private final boolean ignoreDetails;

    private Api5ErrorMatcher(Api5Error expected) {
        this.expectedError = expected;
        this.isCodeInIgnoreList = false;
        this.ignoreDetails = false;
    }

    private Api5ErrorMatcher(Api5Error expected,
                             List<String> ignoreDetailForCodes,
                             boolean ignoreDetails) {
        this.expectedError = expected;
        this.isCodeInIgnoreList = ignoreDetailForCodes != null
                && ignoreDetailForCodes.contains(String.valueOf(expectedError.getFaultInfo().getErrorCode()));
        this.ignoreDetails = ignoreDetails;
    }

    @Override
    public void describeTo(Description description) {
        description.appendText(
                (isCodeInIgnoreList) ?
                        ("error comparing only codes " + expectedError.toString()) :
                        ("error with details " + expectedError.toString()));
    }

    protected boolean matchesSafely(Api5Error item, Description mismatchDescription) {
        String itemDetailNoCode = START_CODE_REGEXP
                .matcher(item.getFaultInfo().getErrorDetail())
                .replaceFirst("");

        item.getFaultInfo().setErrorDetail(itemDetailNoCode);
        boolean matchesResult;
        //compare error message
        if (expectedError.getMessage() == null || expectedError.getMessage().equals("")) {
            matchesResult = item.getMessage() == null || item.getMessage().equals("");
        } else {
            matchesResult = item.getMessage() != null
                    && !item.getMessage().equals("")
                    && item.getMessage().equals(expectedError.getMessage());
        }

        //compare details only if error code not in ignore details list
        matchesResult =
                expectedError.getFaultInfo().getErrorCode() == item.getFaultInfo().getErrorCode() &&
                (isCodeInIgnoreList || matchesResult &&
                        (ignoreDetails || expectedError.getFaultInfo().equals(item.getFaultInfo())));
        if (!matchesResult) {
            mismatchDescription.appendText("\nerror was ");
            mismatchDescription.appendText(item.toString());
        }
        return matchesResult;
    }

    @Factory
    public static Matcher<Api5Error> equalToIgnoreLocale(Api5Error expectedError) {
        return new Api5ErrorMatcher(expectedError);
    }

    @Factory
    public static Matcher<Api5Error> equalToIgnoreLocale(
            Api5Error expectedError,
            List<String> ignoreDetailForCodes) {
        return new Api5ErrorMatcher(expectedError, ignoreDetailForCodes, false);
    }

    @Factory
    public static Matcher<Api5Error> equalToIgnoreLocaleAndDetails(
            Api5Error expectedError,
            List<String> ignoreDetailForCodes) {
        return new Api5ErrorMatcher(expectedError, ignoreDetailForCodes, true);
    }
}
