package ru.yandex.autotests.directapi.apiclient.errors;

import java.util.List;

import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

/**
 * Created by chicos on 28.11.14.
 */
public class Api5JsonErrorMatcher extends TypeSafeDiagnosingMatcher<Api5JsonError> {
    private final Api5JsonError expectedError;
    private final boolean isCodeInIgnoreList;
    private final String START_CODE_REGEXP = "^#[0-9]*(: )?";
    private final boolean ignoreDetails;

    public Api5JsonErrorMatcher(Api5JsonError expected) {
        this.expectedError = expected;
        this.isCodeInIgnoreList = false;
        this.ignoreDetails = false;
    }

    public Api5JsonErrorMatcher(Api5JsonError expected,
                                List<String> ignoreDetailForCodes,
                                boolean ignoreDetails) {
        this.expectedError = expected;
        this.isCodeInIgnoreList = ignoreDetailForCodes != null
                && ignoreDetailForCodes.contains(expectedError.getErrorCode().toString());
        this.ignoreDetails = ignoreDetails;
    }

    @Override
    public void describeTo(Description description) {
        description.appendText(
                (isCodeInIgnoreList) ?
                        ("error comparing only codes " + expectedError.toString()) :
                        ("error with details " + expectedError.toString()));
    }

    protected boolean matchesSafely(Api5JsonError item, Description mismatchDescription) {
        String itemDetailNoCode = item.getDetails().replaceFirst(START_CODE_REGEXP, "");
        //workaround for json - waiting for new fields mapping in errors
        //removes redundant brackets in phrases like - 'required value for element `Country' missing at {http://direct.yandex.com/api/v5/vcards}AddRequest/VCards'
        //DIRECT-33530
        itemDetailNoCode = itemDetailNoCode
                .replace("{", "")
                .replace("}", "");

        item.setDetails(itemDetailNoCode);
        boolean matchesResult;
        if (ignoreDetails) {
            matchesResult = true;
        } else if (expectedError.getDetails() == null || expectedError.getDetails().equals("")) {
            matchesResult = item.getDetails() == null || item.getDetails().equals("");
        } else if (item.getDetails() == null || item.getDetails().equals("")) {
            matchesResult = false;
        } else {
            matchesResult = item.getDetails().equals(expectedError.getDetails());
        }

        matchesResult = expectedError.getErrorCode().equals(item.getErrorCode()) &&
                (isCodeInIgnoreList || matchesResult && expectedError.getMessage().equals(item.getMessage()));
        if (!matchesResult) {
            mismatchDescription.appendText("\nerror was ");
            mismatchDescription.appendText(item.toString());
        }
        return matchesResult;
    }

    @Factory
    public static Matcher<Api5JsonError> equalToIgnoreLocale(Api5JsonError expectedError) {
        return new Api5JsonErrorMatcher(expectedError);
    }

    @Factory
    public static Matcher<Api5JsonError> equalToIgnoreLocale(
            Api5JsonError expectedError,
            List<String> ignoreDetailForCodes) {
        return new Api5JsonErrorMatcher(expectedError, ignoreDetailForCodes, false);
    }

    @Factory
    public static Matcher<Api5JsonError> equalToIgnoreLocaleAndDetails(
            Api5JsonError expectedError,
            List<String> ignoreDetailForCodes) {
        return new Api5JsonErrorMatcher(expectedError, ignoreDetailForCodes, true);
    }
}
