package ru.yandex.autotests.directapi.apiclient.errors;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableMap;
import com.google.gson.annotations.SerializedName;
import com.jayway.jsonpath.JsonPath;
import org.apache.axis.AxisFault;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

import ru.yandex.autotests.direct.utils.DirectUtilsException;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.textresource.TextResourceFormatter;
import ru.yandex.autotests.directapi.apiclient.config.ApiLocale;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;

import static ru.yandex.autotests.directapi.apiclient.errors.AxisErrorMessage.*;

/**
 * Created with IntelliJ IDEA.
 * User: mariabye
 * Date: 17.05.13
 * Time: 17:11
 * To change this template use File | Settings | File Templates.
 */
public class AxisError extends RuntimeException {
    private static Logger log = LogManager.getLogger(AxisError.class);

    public AxisError() {
        super(new Exception());
    }

    @SerializedName("error_code")
    private Integer errorCode;

    @SerializedName("error_str")
    private String message;

    @SerializedName("error_detail")
    private String details;

    private transient Map<String, String> detailsLocale;

    public Integer getErrorCode() {
        return errorCode;
    }

    public AxisError clone() {
        AxisError error = new AxisError((int) errorCode);
        error.setDetails(details);
        return error;
    }

    public void setErrorCode(Integer errorCode) {
        this.errorCode = errorCode;
    }

    public String getMessage() {
        return message;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    public String getDetails() {
        return details;
    }

    public void setDetails(String details) {
        this.details = details;
    }

    /**
     * Создает объект AxisError (ошибка сервиса) с заданным кодом и пустым описанием
     *
     * @param errorCode код ошибки
     */
    public AxisError(int errorCode) {
        setAxisError(errorCode, null, null);
    }

    /**
     * Создает объект AxisError (ошибка сервиса)  с заданным кодом и описанием
     *
     * @param errorCode    код ошибки
     * @param errorDetails описание ошибки
     */
    public AxisError(int errorCode, AxisErrorDetails errorDetails) {
        setAxisError(errorCode, errorDetails, (Object) null);
    }

    /**
     * Создает объект AxisError (ошибка сервиса)  с заданным кодом и описанием, сформированным
     * из шаблона текста ошибки и заданных параметров detailsParams
     * <p>
     * Например ошибка,
     * 71, "Invalid campaignID 1234567"
     * описывается как
     * 71, "Invalid campaignID %s"
     * и объект AxisError формируется как
     * new AxisError(71, AxisErrorMessage.INVALID_CAMPAIGN_ID, new Object[]{1243567})
     * где  ресурс INVALID_CAMPAIGN_ID = Invalid campaignID %s
     * </p>
     *
     * @param errorCode     код ошибки
     * @param errorDetails  описание ошибки
     * @param detailsParams параметры в описании ошибки
     */

    public AxisError(int errorCode, AxisErrorDetails errorDetails, Object... detailsParams) {
        setAxisError(errorCode, errorDetails, detailsParams);
    }


    /**
     * Создает объект ошибки на основе полученного объекта Error
     *
     * @param error ошибка полученная от сервиса
     */
    public <T> AxisError(T error) {
        if (error == null) {
            throw new DirectAPIException("Сервер ответил 502 ошибкой");
        }
        if (error instanceof AxisFault) {
            AxisFault axisFault = (AxisFault) error;
            try {
                this.errorCode = Integer.valueOf(axisFault.getFaultCode().getLocalPart());
                this.message = axisFault.getMessage();
                this.details = getAxisFaultDetail(axisFault);
                if (this.details != null && this.details.equals("")) {
                    this.details = null;
                }
            } catch (Exception e) {
                //incorrect AxisFault format
                throw new DirectAPIException("Неверный ответ сервиса (SOAP)" + axisFault.getMessage(), e);
            }
        } else if (error instanceof AxisError) {
            this.errorCode = ((AxisError) error).getErrorCode();
            this.message = ((AxisError) error).getMessage();
            this.details = ((AxisError) error).getDetails();
        } else {
            String jsonError = JsonUtils.toString(error);
            this.errorCode = JsonPath.read(jsonError, "$.FaultCode");
            this.message = JsonPath.read(jsonError, "$.FaultString");
            this.details = JsonPath.read(jsonError, "$.FaultDetail");
        }
    }

    /**
     * Создает список ошибок на основе полученного массива Error
     */
    public static <T> List<AxisError> getAxisErrors(T[] errors) {
        List<AxisError> result = new ArrayList<AxisError>();
        for (T error : errors) {
            AxisError axis = new AxisError(error);
            result.add(axis);
        }
        return result;
    }

    /**
     * Инициализирует объект AxisError (ошибка сервиса)  с заданным кодом и описанием
     *
     * @param errorCode    код ошибки
     * @param errorDetails описание ошибки
     */
    private void setAxisError(int errorCode, AxisErrorDetails errorDetails, Object... params) {
        this.errorCode = errorCode;
        this.detailsLocale = new HashMap<>();
        TextResourceFormatter stringResource = TextResourceFormatter.resource(ERROR_MESSAGES.get(errorCode))
                .args(params)
                .locale(DirectTestRunProperties.getInstance().getDirectAPILocale());
        this.message = stringResource.toString();
        if (errorDetails == null) {
            this.details = null;
        } else {
            try {
                TextResourceFormatter resourceFormatter = TextResourceFormatter.resource(errorDetails)
                        .args(params)
                        .locale(DirectTestRunProperties.getInstance().getDirectAPILocale());
                this.details = resourceFormatter.toString();

                detailsLocale.put(ApiLocale.EN.toString(), TextResourceFormatter.resource(errorDetails)
                        .args(params)
                        .locale(ApiLocale.EN.toString()).toString());
                detailsLocale.put(ApiLocale.RU.toString(), TextResourceFormatter.resource(errorDetails)
                        .args(params)
                        .locale(ApiLocale.RU.toString()).toString());
            } catch (DirectUtilsException noResourceKey) {
                log.error("Не хватает описания ошибки в ресурсах!");
                this.details = errorDetails.toString();
            }
        }
    }

    /**
     * Извлекает текст описания ошибки из объекта AxisFault
     *
     * @param fault полученный объект AxisFault
     * @return String текст описания ошибки
     */
    public static String getAxisFaultDetail(AxisFault fault) {
        String faultDetail = null;
        for (int i = 0; i < fault.getFaultDetails().length; i++) {
            if (fault.getFaultDetails()[i].getNodeName().equals("detail")
                    || fault.getFaultDetails()[i].getNodeName().equals("text"))
            {
                faultDetail = fault.getFaultDetails()[i].getTextContent();
                break;
            }
        }
        return faultDetail;
    }

    /**
     * Добавляет к описанию ошибки ведущий текст:
     * In banner[0](BannerID=%s)
     *
     * @param detailsParam
     * @return объект ошибки
     */
    public AxisError withInBanner(Object... detailsParam) {
        TextResourceFormatter resourceFormatter = TextResourceFormatter.resource(AxisErrorDetails.IN_BANNER)
                .args(detailsParam)
                .locale(DirectTestRunProperties.getInstance().getDirectAPILocale());

        this.details = String.format(String.format("%s %s", resourceFormatter.toString(), this.details), detailsParam);
        detailsLocale.put(ApiLocale.EN.toString(),
                String.format(String.format("%s %s", resourceFormatter.toString(),
                        detailsLocale.get(ApiLocale.EN.toString())), detailsParam));
        detailsLocale.put(ApiLocale.RU.toString(),
                String.format(String.format("%s %s", resourceFormatter.toString(),
                        detailsLocale.get(ApiLocale.RU.toString())), detailsParam));
        return this;
    }

    /**
     * Добавляет к описанию ошибки ведущий текст:
     * In banner[0]
     *
     * @return объект ошибки
     */
    public AxisError withInBanner() {
        TextResourceFormatter resourceFormatter = TextResourceFormatter.resource(AxisErrorDetails.IN_BANNER_NO_ID)
                .locale(DirectTestRunProperties.getInstance().getDirectAPILocale());
        this.details = String.format("%s %s", resourceFormatter.toString(), this.details);
        detailsLocale.put(ApiLocale.EN.toString(),
                String.format(String.format("%s %s", resourceFormatter.toString(),
                        detailsLocale.get(ApiLocale.EN.toString()))));
        detailsLocale.put(ApiLocale.RU.toString(),
                String.format(String.format("%s %s", resourceFormatter.toString(),
                        detailsLocale.get(ApiLocale.RU.toString()))));

        return this;
    }

    public AxisError withInPhrase(Object... phraseIDs) {
        TextResourceFormatter resourceFormatter = TextResourceFormatter.resource(AxisErrorDetails.IN_PHRASE)
                .args(phraseIDs)
                .locale(DirectTestRunProperties.getInstance().getDirectAPILocale());
        this.details = String.format("%s %s", resourceFormatter.toString(), this.details);
        detailsLocale.put(ApiLocale.EN.toString(),
                String.format(String.format("%s %s", resourceFormatter.toString(),
                        detailsLocale.get(ApiLocale.EN.toString()))));
        detailsLocale.put(ApiLocale.RU.toString(),
                String.format(String.format("%s %s", resourceFormatter.toString(),
                        detailsLocale.get(ApiLocale.RU.toString()))));
        return this;
    }

    public String toString() {
        return JsonUtils.toString(this);
    }

    public String toExpandString() {
        StringBuilder sb = new StringBuilder(JsonUtils.toString(this));
        sb.replace(sb.indexOf("}"), sb.length(), "\t\"error_details_local\":\n");
        sb.append(JsonUtils.toString(detailsLocale));
        return sb.toString();
    }

    private java.lang.Object __equalsCalc = null;

    @Override
    public synchronized boolean equals(Object got) {
        if (!(got instanceof AxisError)) {
            return false;
        }
        AxisError other = (AxisError) got;
        if (got == null) {
            return false;
        }
        if (this == got) {
            return true;
        }
        if (__equalsCalc != null) {
            return (__equalsCalc == got);
        }
        __equalsCalc = got;
        boolean _equals;
        _equals = (((int) this.errorCode == (int) other.getErrorCode()) &&
                this.getMessage().equals(other.getMessage()) &&
                (
                        (
                                this.getDetails() == null
                                        && (other.getDetails() == null || other.getDetails().equals(""))
                        )
                                ||
                                (
                                        other.getDetails() == null
                                                && (this.getDetails() == null || this.getDetails().equals(""))
                                )
                                ||
                                (this.getDetails() != null
                                        && other.getDetails() != null
                                        && this.getDetails().equals(other.getDetails())
                                )
                )
        );
        __equalsCalc = null;
        return _equals;
    }

    private boolean __hashCodeCalc = false;

    public synchronized int hashCode() {
        if (__hashCodeCalc) {
            return 0;
        }
        __hashCodeCalc = true;
        int _hashCode = 1;
        _hashCode += getErrorCode();
        if (getMessage() != null) {
            _hashCode += getMessage().hashCode();
        }
        if (getDetails() != null) {
            _hashCode += getDetails().hashCode();
        }
        __hashCodeCalc = false;
        return _hashCode;
    }

    private static final Map<Integer, AxisErrorMessage> ERROR_MESSAGES = ImmutableMap.<Integer, AxisErrorMessage>builder()
            .put(1, INVALID_CAMPAIGN_ID)
            .put(2, THERE_ARE_NO_STATISTICS_FOR_THIS_CAMPAIGN)
            .put(3, INVALID_DATE)
            .put(5, INVALID_TIME_INTERVAL)
            .put(8, INVALID_LIMITS)
            .put(9, THIS_FIELD_MUST_BE_AN_ARRAY)
            .put(10, BAD_BANNER_FILTER)
            .put(11, BAD_GEO_FILTER)
            .put(12, BAD_PAGE_NAME_FILTER)
            .put(13, BAD_PAGE_TYPE_FILTER)
            .put(14, BAD_PHRASE_FILTER)
            .put(15, BAD_PAGE_TYPE)
            .put(16, INVALID_GROUP_BY_DATE_SETTINGS)
            .put(17, INVALID_REPORT_FIELD)
            .put(18, INVALID_OFFSET)
            .put(20, BAD_TYPE_RESULT_REPORT)
            .put(21, BAD_COMPRESSION_REPORT)
            .put(22, INVALID_REPORT_ID)
            .put(23, BAD_BANNER_ID)
            .put(24, NO_REPORT)
            .put(25, FIELD_MUST_CONTAIN_THE_FOLLOWING_VALUES_YES_OR_NO)
            .put(26, THE_VALUE_MUST_BE_BETWEEN_0_AND_100_IN_MULTIPLE_OF_10)
            .put(27, INVALID_BANNER_ID)
            .put(28, INVALID_PHRASE_ID)
            .put(29, INVALID_CAMPAIGN_TYPE)
            .put(30, THE_ARRAY_CANNOT_BE_EMPTY)
            .put(31, REPORT_QUEUE_LIMIT_REACHED)
            .put(32, BAD_VALUE_FOR_FIELD_POSITION_TYPE)
            .put(34, INVALID_EMAIL_FORMAT)
            .put(36, INVALID_TIMESTAMP_FORMAT)
            .put(37, INVALID_STRATEGY_PARAMETERS)
            .put(39, INVALID_AD_ID)
            .put(40, INVALID_MEDIA_PLAN_AD_ID)
            .put(41, INVALID_MEDIA_PLAN_KEYWORD_ID)
            .put(42, INVALID_MEDIA_PLAN_CATEGORY_ID)
            .put(43, INCORRECT_MEDIA_PLAN)
            .put(44, INCORRECT_MEDIA_PLAN_AD)
            .put(47, INVALID_RETARGETING_CONDITION_ID)
            .put(48, INVALID_FILTER_PARAMETER)
            .put(49, INVALID_AD_GROUP_ID)
            .put(53, AUTHORIZATION_ERROR)
            .put(54, NOT_ENOUGH_RIGHTS)
            .put(55, THIS_METHOD_DOES_NOT_EXIST)
            .put(56, REQUEST_LIMIT_EXCEEDED)
            .put(58, NO_ACCESS)
            .put(71, INVALID_REQUEST_PARAMETERS)
            .put(72, INVALID_FORECAST_ID)
            .put(73, FORECAST_REPORT_NOT_EXIST)
            .put(74, FORECAST_IS_NOT_READY)
            .put(75, BAD_CATEGORY_ID)
            .put(76, INVALID_PHRASE)
            .put(77, INVALID_GEO_ID)
            .put(79, INVALID_CATALOG_CATEGORY)
            .put(80, UNABLE_TO_DELETE_OBJECT)
            .put(81, ARRAY_VALUES_INCORRECT)
            .put(91, THE_SPECIFIED_WORDSTAT_REPORT_DOES_NOT_EXIST)
            .put(93, INVALID_WORDSTAT_REPORT_ID)
            .put(111, INVALID_CAMPAIGN_SETTINGS)
            .put(112, CANNOT_CREATE_CAMPAIGN_IN_BALANCE)
            .put(114, THE_MAXIMUM_NUMBER_OF_CAMPAIGNS_HAS_BEEN_EXCEEDED)
            .put(115, SET_STRATEGY_NOT_AVAILABLE)
            .put(151, INVALID_BANNER_SETTINGS)
            .put(152, NOT_ENOUGH_UNITS_FOR_CLIENT)
            .put(153, MAXIMUM_BANNER_LIMIT_EXCEEDED_IN_REQUEST)
            .put(154, INVALID_POINT_ON_MAP_STRUCTURE)
            .put(155, THIS_ACTION_CANNOT_BE_PERFORMED_ON_THIS_CAMPAIGN_OR_AD)
            .put(156, CHANGING_OF_ARCHIVE_CAMPAIGN_OR_BANNER_NOT_ALLOWED)
            .put(157, LIMIT_ADS_IN_CAMPAIGN_EXCEEDED)
            .put(158, LIMIT_ADS_IN_GROUP_EXCEEDED)
            .put(191, THE_CONTEXT_PRICE_FOR_PHRASES_ACTIVE_ON_SEARCH_CANNOT_BE_CHANGED)
            .put(192, INVALID_NEGATIVE_KEYWORDS)
            .put(193, OVER_LIMIT_KEYWORD_IDS)
            .put(194, KEYWORD_NOT_FOUND)
            .put(195, NOT_AVAILABLE_SUSPEND_LAST_KEYWORD)
            .put(241, MAXIMUM_ARRAY_SIZE_EXCEEDED)
            .put(242, INVALID_COST)
            .put(243, BAD_AUTO_BROKER)
            .put(244, NO_PHRASES_FOUND)
            .put(245, INVALID_CURRENCY)
            .put(251, INVALID_USER_LOGIN)
            .put(252, LOGIN_IS_OCCUPIED)
            .put(254, AN_ERROR_OCCURRED_WHILE_CREATING_CLIENT)
            .put(257, USER_RIGHTS_COULD_NOT_BE_UPDATED)
            .put(259, THIS_CLIENT_DOES_NOT_EXIST)
            .put(271, IMAGE_NOT_FOUND)
            .put(272, INVALID_IMAGE)
            .put(273, THE_IMAGE_POOL_SIZE_WAS_EXCEED)
            .put(350, INVALID_FINANCIAL_TRANSACTION_TOKEN)
            .put(352, FINANCIAL_TRANSACTIONS_ARE_NOT_PERMITTED_FOR_YOUR_ACCOUNT)
            .put(353, INVALID_MONEY_TRANSFER_REQUEST)
            .put(354, INVALID_INVOICE_REQUEST)
            .put(355, CREDIT_LIMIT_EXCEEDED)
            .put(356, INVALID_CONTRACT_ID)
            .put(358, FINANCIAL_TRANSACTIONS_ARE_TEMPORARILY_UNAVAILABLE)
            .put(360, PAYMENT_TOKEN_MISSING)
            .put(361, INVALID_PAYMENT_TOKEN)
            .put(363, OVERDRAFT_UNAVAILABLE)
            .put(364, OVERDRAFT_LIMIT_EXCEEDED)
            .put(365, UNABLE_TO_PERFORM_TRANSACTION)
            .put(366, PAYMENT_OF_DIFF_TYPE_CAMPAIGNS_NOT_ALLOWED)
            .put(368, INVALID_PAYMENT_METHOD)
            .put(370, TRANSACTION_DOESNT_EXIST)
            .put(371, PAY_IN_CURRENCY_NOT_AVAILABLE_FOR_CLIENT)
            .put(372, SOMEONE_ELSE_PAYMENT_TOKEN)
            .put(500, INTERNAL_SERVER_ERROR)
            .put(501, INVALID_REQUEST)
            .put(503, API_IS_TEMPORARY_UNAVAILABLE)
            .put(506, TOO_MANY_SIMULTANEOUS_REQUESTS)
            .put(508, REQUESTED_API_S_VERSION_IS_NOT_SUPPORTED)
            .put(509, THIS_METHOD_IS_NOT_AVAILABLE_IN_THIS_API_VERSION)
            .put(510, ACCESS_DENIED)
            .put(511, UNKNOWN_LANGUAGE)
            .put(513, YOUR_LOGIN_IS_NOT_CONNECTED_TO_YANDEX_DIRECT)
            .put(515, ENABLE_SHARED_ACCOUNT_IS_REQUIRED)
            .put(519, ENABLE_SHARED_ACCOUNT_HAS_BEEN_ON)
            .put(520, CAMPAIGN_REQUIRED_TO_ENABLE_SHARED_ACCOUNT)
            .put(521, WORK_WITH_SHARED_ACCOUNT_RESTRICTED)
            .put(522, NO_CAMPAIGN_WITH_ACCOUNT)
            .put(3500, DOES_NOT_SUPPORT)
            .put(5005, FIELD_SET_INCORRECTLY)
            .build();

    public Map<String, String> getDetailsLocale() {
        return detailsLocale;
    }
}
