package ru.yandex.autotests.directapi.apiclient.version4;

import java.io.StringWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.gson.JsonSyntaxException;
import freemarker.cache.ClassTemplateLoader;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.DefaultObjectWrapper;
import freemarker.template.Template;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Consts;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.json.JSONObject;

import ru.yandex.autotests.directapi.apiclient.ApiResponseHandler;
import ru.yandex.autotests.directapi.apiclient.HttpMessage;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.apiclient.version5.MethodInvocationResult;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.httpclient.lite.core.config.HttpClientConnectionConfig;
import ru.yandex.autotests.httpclient.lite.core.config.HttpClientFactory;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Attachment;

/**
 * Created with IntelliJ IDEA.
 * User: mariabye
 * Date: 14.05.13
 * Time: 10:44
 * Soap implementation of Direct.API service client
 */
public class JsonClient extends BaseApiV4Client {
    public static Log log = LogFactory.getLog(JsonClient.class);

    public JsonClient clone() {
        return new JsonClient(connectionConfig.clone(), requestHeader.clone());
    }

    public JsonClient(ConnectionConfig connectionConfig) {
        super(connectionConfig);
    }

    public JsonClient(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        super(connectionConfig, requestHeader);
    }

    @Override
    protected ProtocolType getProtocolType() {
        return ProtocolType.JSON;
    }

    public String sendRequest(Method methodName, Object params) {
        return sendRequest(methodName.toString(), params, false)[0];
    }

    public String[] sendRequestAndGetResponseWithHeaders(Method methodName, Object params) {
        return sendRequest(methodName.toString(), params, false);
    }

    public String sendRequest(String methodName, Object params) {
        return sendRequest(methodName, params, false)[0];
    }

    public String sendRequestForNonParsableResponse(Method methodName, Object params) {
        return sendRequest(methodName.toString(), params, true)[0];
    }

    private String[] sendRequest(String methodName, Object params, boolean nonParsableResponseExpected) {
        JsonRequest request = new JsonRequest();
        request.setHeader(requestHeader.getHeaderParameters());
        request.setMethod(methodName);
        request.setParameters(params);

        HttpMessage message = new HttpMessage();
        message.setUrl(connectionConfig.getEndPoint(getProtocolType()));
        message.setRequest(request.toString());

        String response = null;
        try {
            HttpClientConnectionConfig clientConnectionConfig =
                    ConnectionConfig.getHttpClientConnectionConfig(connectionConfig.getEndPoint(getProtocolType()))
                            .timeout(360);
            CloseableHttpClient httpClient = HttpClientFactory.getHttpClient(clientConnectionConfig);

            HttpPost httpRequest = new HttpPost(connectionConfig.getEndPoint(getProtocolType()));
            StringEntity entity = new StringEntity(request.toString(),
                    ContentType.create("plain/text", Consts.UTF_8));
            httpRequest.setEntity(entity);
            MethodInvocationResult<String> methodInvocationResult = httpClient.execute(httpRequest, new ApiResponseHandler(message));
            response = methodInvocationResult.getResponseObject();

            if (response == null) {
                throw new DirectAPIException("Response contains no content");
            }

            if (!nonParsableResponseExpected) {
                try {
                    message.setResponse(new JSONObject(response).toString(2));
                } catch (JsonSyntaxException e) {
                    throw new DirectAPIException("Получен некорреткный ответ от сервера", e);
                }
            }
            traceHttpMessage(message);
            checkAxisFaultPresence(message);
        } catch (AxisError axisError) {
            throw axisError;
        } catch (DirectAPIException knownException) {
            traceHttpMessage(message);
            throw new DirectAPIException("Неверный ответ сервера: " + knownException.getMessage());
        } catch (Exception e) {
            traceHttpMessage(message);
            throw new DirectAPIException("Неизвестная ошибка при выполнении JSON запроса", e);
        }
        response = (nonParsableResponseExpected) ? (response) : (getResponseData(response));
        return new String[]{response, message.getResponseHeaders()};
    }

    @Attachment(value = "[JSON/CURL]: Request", type = "text/html")
    public String traceHttpMessage(HttpMessage message) {
        log.info(message.toString());
        Map root = new HashMap();
        root.put("usualFormatString", message.toString());
        root.put("curlString", message.toCurlStringForJson());
        root.put("buttonName", "JSON");

        String attachmentHTML;
        try {
            TemplateLoader file = new ClassTemplateLoader(this.getClass(), "/");
            Configuration cfg = new Configuration();
            cfg.setTemplateLoader(file);
            cfg.setObjectWrapper(new DefaultObjectWrapper());
            cfg.setDefaultEncoding("UTF-8");
            Template template = cfg.getTemplate("request2curlTemplate.ftl");
            Writer fileWriter = new StringWriter();
            try {
                template.process(root, fileWriter);
            } finally {
                fileWriter.close();
            }
            attachmentHTML = fileWriter.toString();
        } catch (Exception e) {
            throw new RuntimeException("Error", e);
        }

        return attachmentHTML;
    }

    public <T> T invokeMethod(String methodName, Object params) {
        Class resultType = getReturnClass(methodName);
        T result = null;
        result = invokeMethod(methodName, params, resultType);
        return result;
    }

    public <T> T invokeMethod(String methodName, Object params, Class resultType) {
        String response = null;
        response = sendRequest(methodName, params);
        Object result = JsonUtils.getObject(response, resultType);
        return (T) result;
    }

    public <T> T invokeMethod(Method methodName, Object params, Class resultType) {
        String response = null;
        response = sendRequest(methodName.toString(), params);
        Object result = JsonUtils.getObject(response, resultType);
        return (T) result;
    }

    private Class getReturnClass(String methodName) {
        java.lang.reflect.Method method = null;
        try {
            java.lang.reflect.Method[] methods =
                    Class.forName(getPackageName() + ".APIPort_PortType").getDeclaredMethods();
            for (java.lang.reflect.Method m : methods) {
                if (m.getName().toLowerCase().equals(methodName.toLowerCase())) {
                    method = m;
                }
            }
        } catch (ClassNotFoundException e) {
            throw new DirectAPIException("Метод не определен в интерфейсе", e);
        }
        if (method == null) {
            return String.class;
        } else {
            return method.getReturnType();
        }
    }

    /**
     * Look http message for error codes and throw AxisError if found one
     *
     * @param message
     */
    private void checkAxisFaultPresence(HttpMessage message) {
        try {
            AxisError error = JsonUtils.getObject(message.getResponse(), AxisError.class);
            if (error.getErrorCode() != null) {
                throw error;
            }
        } catch (JsonSyntaxException e) {
            log.debug("Non parsable response error");
        }
    }

    /**
     * Возвращает содержимое структуры data {}, полученное в ответе сервиса
     *
     * @param response ответ сервиса
     * @return содержимое структуры data{}, полученное в ответе сервиса в формате json
     * @throws DirectAPIException ошибка парсинга data в ответе
     */
    public String getResponseData(String response) {
        response = response.replace("\"x\"", "\"X\"");
        response = response.replace("\"x1\"", "\"X1\"");
        response = response.replace("\"x2\"", "\"X2\"");
        response = response.replace("\"y\"", "\"Y\"");
        response = response.replace("\"y1\"", "\"Y1\"");
        response = response.replace("\"y2\"", "\"Y2\"");
        response = response.replace("\"uid\"", "\"Uid\"");
        Pattern p = Pattern.compile("data.*?:(.*)}");
        Matcher m = p.matcher(response);
        if (m.find()) {
            return (m.group(1));
        }
        throw new DirectAPIException("Не удалось извлечь data из ответа:\n" + response);
    }
}
