package ru.yandex.autotests.directapi.apiclient.version5;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.autotests.directapi.model.api5.Action;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Конфигурация проксирования запросов JSON-клиента API v5.
 * Проксирование селективное – по определённым сервисам и методам.
 * <p>
 * Предполагается применение для запуска тестов на разработческой среде,
 * довольно частый кейс: нужно прокинуть пару методов на приложение,
 * запущенное на локальной машине, остальные запросы – на какую-то бету.
 * <p>
 * Данный класс позволяет решить эту задачу без необходимости вносить изменения
 * в код {@link JSONClientV5} и пересобирать модуль {@code directapi-steps},
 * просто указывая локальные настройки в конфигах модулей, откуда запускаются тесты.
 * <p>
 * Пример конфига можно посмотреть в тестовых ресурсах. Как с конфигом работает
 * данный класс, соответственно, можно увидеть в тесте.
 *
 * @see JSONClientV5
 */
@ParametersAreNonnullByDefault
class JsonClientProxyConfig {

    private static final String PROXY_CONFIG_NAME = "api5-client-proxy.conf";

    private final Map<Pair<ServiceNames, Action>, String> proxyTable;

    JsonClientProxyConfig() {
        Config conf = ConfigFactory.load(PROXY_CONFIG_NAME);
        if (conf.isEmpty() || !conf.hasPath("proxies")) {
            proxyTable = ImmutableMap.of();
            return;
        }

        ImmutableMap.Builder<Pair<ServiceNames, Action>, String> builder = ImmutableMap.builder();

        for (Config proxy : conf.getConfigList("proxies")) {
            try {
                builder.putAll(buildProxyTable(proxy));
            } catch (MalformedURLException ex) {
                throw new IllegalArgumentException(ex);
            }
        }

        proxyTable = builder.build();
    }

    private Map<Pair<ServiceNames, Action>, String> buildProxyTable(Config proxyItem) throws MalformedURLException {
        Map<Pair<ServiceNames, Action>, String> table = new HashMap<>();
        URL endpoint = new URL(ensureEndsWithSlash(proxyItem.getString("endpoint")));
        for (String entry : proxyItem.getStringList("service.methods")) {
            String[] serviceMethod = entry.split("\\.");
            checkArgument(serviceMethod.length == 2);
            String service = serviceMethod[0], method = serviceMethod[1];

            Pair<ServiceNames, Action> key = Pair.of(ServiceNames.fromString(service), Action.fromString(method));
            table.put(key, new URL(endpoint, service.toLowerCase()).toString());
        }
        return table;
    }

    private static String ensureEndsWithSlash(String s) {
        return s.endsWith("/") ? s : s + "/";
    }

    /**
     * @param service           сервис, в который ходим
     * @param method            вызываемый метод
     * @param defaultEndpoint   адрес, по которому пойдём, если соответствующей записи
     *                          нет в таблице проксирования
     * @return Строка URL, по которому следует адресовать запрос в API в соответствии
     * с конфигурацией проксирования. В случае, если конфигурация прокси не содержит
     * записи для заданных сервиса и метода, будет возвращён исходный адрес,
     * переданный в аргументе, {@code defaultEndpoint}.
     */
    String getEndpoint(ServiceNames service, Action method, String defaultEndpoint) {
        return proxyTable.getOrDefault(Pair.of(service, method), defaultEndpoint);
    }

}
