package ru.yandex.autotests.directapi.apiclient.version5;

import java.lang.reflect.UndeclaredThrowableException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.handler.Handler;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import com.sun.xml.ws.developer.WSBindingProvider;
import com.sun.xml.ws.handler.SOAPMessageContextImpl;
import com.yandex.direct.api.v5.adextensions.AdExtensions;
import com.yandex.direct.api.v5.adextensions.AdExtensionsPort;
import com.yandex.direct.api.v5.adgroups.AdGroups;
import com.yandex.direct.api.v5.adgroups.AdGroupsPort;
import com.yandex.direct.api.v5.adimages.AdImages;
import com.yandex.direct.api.v5.adimages.AdImagesPort;
import com.yandex.direct.api.v5.ads.Ads;
import com.yandex.direct.api.v5.ads.AdsPort;
import com.yandex.direct.api.v5.agencyclients.AgencyClients;
import com.yandex.direct.api.v5.agencyclients.AgencyClientsPort;
import com.yandex.direct.api.v5.audiencetargets.AudienceTargets;
import com.yandex.direct.api.v5.audiencetargets.AudienceTargetsPort;
import com.yandex.direct.api.v5.bidmodifiers.BidModifiers;
import com.yandex.direct.api.v5.bidmodifiers.BidModifiersPort;
import com.yandex.direct.api.v5.bids.Bids;
import com.yandex.direct.api.v5.bids.BidsPort;
import com.yandex.direct.api.v5.campaigns.Campaigns;
import com.yandex.direct.api.v5.campaigns.CampaignsPort;
import com.yandex.direct.api.v5.campaignsext.CampaignsExt;
import com.yandex.direct.api.v5.campaignsext.CampaignsExtPort;
import com.yandex.direct.api.v5.changes.Changes;
import com.yandex.direct.api.v5.changes.ChangesPort;
import com.yandex.direct.api.v5.clients.Clients;
import com.yandex.direct.api.v5.clients.ClientsPort;
import com.yandex.direct.api.v5.dictionaries.Dictionaries;
import com.yandex.direct.api.v5.dictionaries.DictionariesPort;
import com.yandex.direct.api.v5.dynamictextadtargets.DynamicTextAdTargets;
import com.yandex.direct.api.v5.dynamictextadtargets.DynamicTextAdTargetsPort;
import com.yandex.direct.api.v5.keywordbids.KeywordBids;
import com.yandex.direct.api.v5.keywordbids.KeywordBidsPort;
import com.yandex.direct.api.v5.keywords.Keywords;
import com.yandex.direct.api.v5.keywords.KeywordsPort;
import com.yandex.direct.api.v5.retargetinglists.RetargetingLists;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListsPort;
import com.yandex.direct.api.v5.sitelinks.Sitelinks;
import com.yandex.direct.api.v5.sitelinks.SitelinksPort;
import com.yandex.direct.api.v5.turbopages.TurboPages;
import com.yandex.direct.api.v5.turbopages.TurboPagesPort;
import com.yandex.direct.api.v5.vcards.VCards;
import com.yandex.direct.api.v5.vcards.VCardsPort;

import ru.yandex.autotests.direct.utils.ReflectionUtils;
import ru.yandex.autotests.direct.utils.model.ApiException;
import ru.yandex.autotests.directapi.apiclient.ApiResponseHeaders;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.SSLUtilities;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.model.api5.Action;

/**
 * Created by mariabye on 17.04.14.
 */
public class SOAPClientV5 extends BaseApiV5Client {
    static {
        SSLUtilities.trustAllHostnames();
        SSLUtilities.trustAllHttpsCertificates();
        System.setProperty("com.sun.xml.ws.transport.http.client.HttpTransportPipe.dump", "true");
    }

    private static ThreadLocal<Boolean> isJavaResponse = ThreadLocal.withInitial(() -> false);

    public SOAPClientV5(ConnectionConfig connectionConfig) {
        super(connectionConfig);
    }

    public SOAPClientV5(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        super(connectionConfig, requestHeader);
    }

    @Override
    public <T> MethodInvocationResult<T> invokeMethodEx(ServiceNames serviceName, String login, Action methodName, Object params) {
        Object soapService = getSoapServiceV5(serviceName, login);
        try {
            @SuppressWarnings("unchecked")
            T responseObject = (T) ReflectionUtils.invokeMethodV5(soapService, methodName.toString(), params);
            return new MethodInvocationResult<>(responseObject, isJavaResponse.get());
        } catch (ApiException apiException) {
            throw new Api5Error(apiException, isJavaResponse.get());
        } catch (NumberFormatException | UndeclaredThrowableException parseException) {
            throw new DirectAPIException("Неверный ответ сервера (SOAP)", parseException);
        }
    }

    @Override
    protected ProtocolType getProtocolType() {
        return ProtocolType.SOAP;
    }

    private Object getSoapServiceV5(ServiceNames serviceName, String login) {
        switch (serviceName) {
            case AD_EXTENSIONS:
                return adExtensionsService(login);
            case AD_GROUPS:
                return adGroupsService(login);
            case AD_IMAGES:
                return adImagesService(login);
            case AUDIENCE_TARGETS:
                return audienceTargetsService(login);
            case ADS:
                return adsService(login);
            case BIDS:
                return bidsService(login);
            case KEYWORD_BIDS:
                return keywordBidsService(login);
            case BID_MODIFIERS:
                return bidModifiersService(login);
            case CHANGES:
                return changesService(login);
            case CAMPAIGNS:
                return campaignsService(login);
            case CLIENTS:
                return clientsService(login);
            case AGENCY_CLIENTS:
                return agencyClientsService(login);
            case DICTIONARIES:
                return dictionariesService(login);
            case DYNAMIC_TEXT_AD_TARGETS:
                return dynamicTextAdTargetsService(login);
            case KEYWORDS:
                return keywordsService(login);
            case SITELINKS:
                return sitelinksService(login);
            case RETARGETING_LISTS:
                return retargetingListsService(login);
            case VCARDS:
                return vCardsService(login);
            case CAMPAIGNS_EXT:
                return campaignsExtService(login);
            case TURBO_PAGES:
                return turboPagesService(login);
        }

        throw new DirectAPIException("Реализация тестовых степов для сервиса " + serviceName + " отсутствует");
    }

    public AdExtensionsPort adExtensionsService(String clientLogin) {
        AdExtensions adExtensions =
                new AdExtensions(connectionConfig.getV5WsdlLocation(ServiceNames.AD_EXTENSIONS.toString()));
        AdExtensionsPort adExtensionsPort = adExtensions.getAdExtensionsSOAP();
        setHttpHeaders((WSBindingProvider) adExtensionsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) adExtensionsPort, ServiceNames.AD_EXTENSIONS.toString());
        return adExtensionsPort;
    }

    public AdGroupsPort adGroupsService(String clientLogin) {
        AdGroups adGroups = new AdGroups(connectionConfig.getV5WsdlLocation(ServiceNames.AD_GROUPS.toString()));
        AdGroupsPort adGroupsPort = adGroups.getAdGroupsSOAP();
        setHttpHeaders((WSBindingProvider) adGroupsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) adGroupsPort, ServiceNames.AD_GROUPS.toString());
        return adGroupsPort;
    }

    public AdImagesPort adImagesService(String clientLogin) {
        AdImages adImage = new AdImages(connectionConfig.getV5WsdlLocation(ServiceNames.AD_IMAGES.toString()));
        AdImagesPort adImagesPort = adImage.getAdImagesSOAP();
        setHttpHeaders((WSBindingProvider) adImagesPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) adImagesPort, ServiceNames.AD_IMAGES.toString());
        return adImagesPort;
    }

    public AudienceTargetsPort audienceTargetsService(String clientLogin) {
        AudienceTargets audienceTargets =
                new AudienceTargets(connectionConfig.getV5WsdlLocation(ServiceNames.AUDIENCE_TARGETS.toString()));
        AudienceTargetsPort audienceTargetsPort = audienceTargets.getAudienceTargetsSOAP();
        setHttpHeaders((WSBindingProvider) audienceTargetsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) audienceTargetsPort, ServiceNames.AUDIENCE_TARGETS.toString());
        return audienceTargetsPort;
    }

    public BidsPort bidsService(String clientLogin) {
        //"https://8788.beta1.direct.yandex.ru:14443/api/v5/bids"
        Bids bids = new Bids(connectionConfig.getV5WsdlLocation(ServiceNames.BIDS.toString()));
        BidsPort bidsPort = bids.getBidsSOAP();
        setHttpHeaders((WSBindingProvider) bidsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) bidsPort, ServiceNames.BIDS.toString());
        return bidsPort;
    }

    public KeywordBidsPort keywordBidsService(String clientLogin) {
        KeywordBids keywordBids =
                new KeywordBids(connectionConfig.getV5WsdlLocation(ServiceNames.KEYWORD_BIDS.toString()));
        KeywordBidsPort keywordBidsPort = keywordBids.getKeywordBidsSOAP();
        setHttpHeaders((WSBindingProvider) keywordBidsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) keywordBidsPort, ServiceNames.KEYWORD_BIDS.toString());
        return keywordBidsPort;
    }

    public DictionariesPort dictionariesService(String clientLogin) {
        Dictionaries dictionaries =
                new Dictionaries(connectionConfig.getV5WsdlLocation(ServiceNames.DICTIONARIES.toString()));
        DictionariesPort dictionariesPort = dictionaries.getDictionariesSOAP();
        setHttpHeaders((WSBindingProvider) dictionariesPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) dictionariesPort, ServiceNames.DICTIONARIES.toString());
        return dictionariesPort;
    }

    public VCardsPort vCardsService(String clientLogin) {
        VCards vCards = new VCards(connectionConfig.getV5WsdlLocation(ServiceNames.VCARDS.toString()));
        VCardsPort vCardsPort = vCards.getVCardsSOAP();
        setHttpHeaders((WSBindingProvider) vCardsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) vCardsPort, ServiceNames.VCARDS.toString());
        return vCardsPort;
    }

    public KeywordsPort keywordsService() {
        return keywordsService(null);
    }

    public KeywordsPort keywordsService(String clientLogin) {
        Keywords keywords = new Keywords(connectionConfig.getV5WsdlLocation(ServiceNames.KEYWORDS.toString()));
        KeywordsPort keywordsPort = keywords.getKeywordsSOAP();
        setHttpHeaders((WSBindingProvider) keywordsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) keywordsPort, ServiceNames.KEYWORDS.toString());
        return keywordsPort;
    }

    public SitelinksPort sitelinksService(String clientLogin) {
        Sitelinks sitelinks = new Sitelinks(connectionConfig.getV5WsdlLocation(ServiceNames.SITELINKS.toString()));
        SitelinksPort sitelinksPort = sitelinks.getSitelinksSOAP();
        setHttpHeaders((WSBindingProvider) sitelinksPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) sitelinksPort, ServiceNames.SITELINKS.toString());
        return sitelinksPort;
    }

    public AdsPort adsService(String clientLogin) {
        Ads ads = new Ads(connectionConfig.getV5WsdlLocation(ServiceNames.ADS.toString()));
        AdsPort adsPort = ads.getAdsSOAP();
        setHttpHeaders((WSBindingProvider) adsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) adsPort, ServiceNames.ADS.toString());
        return adsPort;
    }


    public ChangesPort changesService(String clientLogin) {
        Changes changes = new Changes(connectionConfig.getV5WsdlLocation(ServiceNames.CHANGES.toString()));
        ChangesPort changesPort = changes.getChangesSOAP();
        setHttpHeaders((WSBindingProvider) changesPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) changesPort, ServiceNames.CHANGES.toString());
        return changesPort;
    }


    public CampaignsPort campaignsService(String clientLogin) {
        Campaigns campaigns = new Campaigns(connectionConfig.getV5WsdlLocation(ServiceNames.CAMPAIGNS.toString()));
        CampaignsPort campaignsPort = campaigns.getCampaignsSOAP();
        setHttpHeaders((WSBindingProvider) campaignsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) campaignsPort, ServiceNames.CAMPAIGNS.toString());
        return campaignsPort;
    }

    public CampaignsExtPort campaignsExtService(String clientLogin) {
        CampaignsExt campaigns = new CampaignsExt(connectionConfig.getV5WsdlLocation(ServiceNames.CAMPAIGNS_EXT.toString()));
        CampaignsExtPort campaignsPort = campaigns.getCampaignsExtSOAP();
        setHttpHeaders((WSBindingProvider) campaignsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) campaignsPort, ServiceNames.CAMPAIGNS_EXT.toString());
        return campaignsPort;
    }

    public BidModifiersPort bidModifiersService(String clientLogin) {
        BidModifiers bidModifiers =
                new BidModifiers(connectionConfig.getV5WsdlLocation(ServiceNames.BID_MODIFIERS.toString()));
        BidModifiersPort bidModifiersPort = bidModifiers.getBidModifiersSOAP();
        setHttpHeaders((WSBindingProvider) bidModifiersPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) bidModifiersPort, ServiceNames.BID_MODIFIERS.toString());
        return bidModifiersPort;
    }

    public DynamicTextAdTargetsPort dynamicTextAdTargetsService(String clientLogin) {
        DynamicTextAdTargets dynamicTextAdTargets = new DynamicTextAdTargets(
                connectionConfig.getV5WsdlLocation(ServiceNames.DYNAMIC_TEXT_AD_TARGETS.toString()));
        DynamicTextAdTargetsPort dynamicTextAdTargetsPort = dynamicTextAdTargets.getDynamicTextAdTargetsSOAP();
        setHttpHeaders((WSBindingProvider) dynamicTextAdTargetsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) dynamicTextAdTargetsPort,
                ServiceNames.DYNAMIC_TEXT_AD_TARGETS.toString());
        return dynamicTextAdTargetsPort;
    }

    public ClientsPort clientsService(String clientLogin) {
        Clients clients = new Clients(connectionConfig.getV5WsdlLocation(ServiceNames.CLIENTS.toString()));
        ClientsPort clientsPort = clients.getClientsSOAP();
        setHttpHeaders((WSBindingProvider) clientsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) clientsPort, ServiceNames.CLIENTS.toString());
        return clientsPort;
    }

    public AgencyClientsPort agencyClientsService(String clientLogin) {
        AgencyClients agencyClients =
                new AgencyClients(connectionConfig.getV5WsdlLocation(ServiceNames.AGENCY_CLIENTS.toString()));
        AgencyClientsPort agencyClientsPort = agencyClients.getAgencyClientsSOAP();
        setHttpHeaders((WSBindingProvider) agencyClientsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) agencyClientsPort, ServiceNames.AGENCY_CLIENTS.toString());
        return agencyClientsPort;
    }

    public RetargetingListsPort retargetingListsService(String clientLogin) {
        RetargetingLists retargetingLists =
                new RetargetingLists(connectionConfig.getV5WsdlLocation(ServiceNames.RETARGETING_LISTS.toString()));
        RetargetingListsPort retargetingListsPort = retargetingLists.getRetargetingListsSOAP();
        setHttpHeaders((WSBindingProvider) retargetingListsPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) retargetingListsPort, ServiceNames.RETARGETING_LISTS.toString());
        return retargetingListsPort;
    }

    public TurboPagesPort turboPagesService(String clientLogin) {

        TurboPages turboPages = new TurboPages(connectionConfig.getV5WsdlLocation(ServiceNames.TURBO_PAGES.toString()));
        TurboPagesPort turboPagesPort = turboPages.getTurboPagesSOAP();
        setHttpHeaders((WSBindingProvider) turboPagesPort, clientLogin);
        setSoapLoggingHandler((WSBindingProvider) turboPagesPort, ServiceNames.TURBO_PAGES.toString());
        return turboPagesPort;
    }

    private void setHttpHeaders(
            WSBindingProvider bp,
            String authToken,
            String clientLogin,
            String locale,
            String fakeLogin,
            String useOperatorUnits,
            String tokenType,
            Boolean preferPerlImplementation)
    {
        Map<String, List<String>> requestHeaders = new HashMap<>();
        requestHeaders.put("Authorization", Arrays.asList("Bearer " + authToken));
        if (clientLogin != null) {
            requestHeaders.put("Client-Login", Arrays.asList(clientLogin));
        }
        requestHeaders.put("Accept-Language", Arrays.asList(locale));
        if (fakeLogin != null) {
            requestHeaders.put("Fake-Login", Arrays.asList(fakeLogin));
        }
        if (useOperatorUnits != null) {
            requestHeaders.put("Use-Operator-Units", Arrays.asList(useOperatorUnits));
        }
        if (tokenType != null) {
            requestHeaders.put("Token-Type", Arrays.asList(tokenType));
        }
        if (preferPerlImplementation != null) {
            requestHeaders.put("X-Prefer-Perl-Implementation", Arrays.asList(preferPerlImplementation.toString()));
        }

        bp.getRequestContext().put(MessageContext.HTTP_REQUEST_HEADERS, requestHeaders);
    }

    private void setHttpHeaders(WSBindingProvider bp, String clientLogin) {
        setHttpHeaders(
                bp,
                requestHeader.getToken(),
                clientLogin != null ? clientLogin : requestHeader.getClientLogin(),
                requestHeader.getLocale(),
                requestHeader.getFakeLogin(),
                requestHeader.getUseOperatorUnits(),
                requestHeader.getTokenType(),
                requestHeader.getPreferPerlImplementation()
        );
    }

    private void setSoapLoggingHandler(WSBindingProvider bp, String serviceName) {
        String endpointUrl = connectionConfig.getEndPointForV5(getProtocolType(), serviceName);
        bp.getRequestContext().put(
                BindingProvider.ENDPOINT_ADDRESS_PROPERTY, endpointUrl);

        List<Handler> handlers = bp.getBinding().getHandlerChain();
        handlers.add(new SOAPLoggingHandler());
        handlers.add(new DetectJavaResponseHandler());
        bp.getBinding().setHandlerChain(handlers);
    }

    public boolean isLastJavaResponse() {
        return isJavaResponse.get();
    }

    /**
     * Специальный handler для того чтобы определить откуда пришел ответ (из perl-а или java-ы)
     */
    private static class DetectJavaResponseHandler implements SOAPHandler<SOAPMessageContext> {

        @Override
        public Set<QName> getHeaders() {
            return null;
        }

        @Override
        public boolean handleMessage(SOAPMessageContext context) {
            tryDetectJavaResponse(context);
            return true;
        }

        @Override
        public boolean handleFault(SOAPMessageContext context) {
            tryDetectJavaResponse(context);
            return true;
        }

        @Override
        public void close(MessageContext context) {
        }

        private void tryDetectJavaResponse(SOAPMessageContext context) {
            boolean isOutboundMessage = (Boolean) context.get(MessageContext.MESSAGE_OUTBOUND_PROPERTY);

            if (!isOutboundMessage) {
                @SuppressWarnings("unchecked")
                Map<String, List<String>> responseHeaders = (Map)context.get(SOAPMessageContextImpl.HTTP_RESPONSE_HEADERS);

                boolean isPerlResponse = Optional.ofNullable(responseHeaders.get(ApiResponseHeaders.PERL_IMPLEMENTATION))
                        .map(l -> !l.isEmpty() ? l.get(0) : null)
                        .map(Boolean::valueOf)
                        .orElse(false);

                isJavaResponse.set(!isPerlResponse);
            }
        }
    }
}
