package ru.yandex.autotests.directapi.cases;

import java.util.Arrays;
import java.util.List;

import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;

public class TextGeoMatchingCases {
    private static final String UKR_TEXT_1 = "Ґорілка";
    private static final String UKR_TEXT_2 = "Українська мова";
    private static final String TUR_TEXT_1 = "Metin afiş";
    private static final String TUR_TEXT_2 = "şarap";
    private static final String KAZ_TEXT_1 = "арақ";
    private static final String KAZ_TEXT_2 = "Қазақ тілі";
    private static final String BEL_TEXT_1 = "Дорым падарўнкі";
    private static final String BEL_TEXT_2 = "Усё, наступіў";
    private static final String GER_TEXT_1 = "Überschrift öß";
    private static final String GER_TEXT_2 = "Erklärung";
    private static final String RUS_TEXT = "водка";
    private static final String UZB_TEXT_1 = "Қизиқ, момақалдироқ бўлармикан";
    private static final String UZB_TEXT_2 = "Узр, куттириб қўйдим";

    public enum TextField {
        TEXT("текстом"),
        TITLE("заголовком"),
        TITLE2("заголовком 2");

        final String fieldDescription;

        TextField(String fieldDesription) {
            this.fieldDescription = fieldDesription;
        }
    }

    private static String getActionDescription(TextField textField, String langName) {
        return String.format("Добавление баннера с %s %s", langName, textField.fieldDescription);
    }

    public static List<NegativeTestCase> getNegativeCases(TextField textField) {
        return Arrays.asList(
                new NegativeTestCase(
                        getActionDescription(textField, "украинским"),
                        UKR_TEXT_1,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_UKRAINE)
                ),
                new NegativeTestCase(
                        getActionDescription(textField, "турецким"),
                        TUR_TEXT_1,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_TURKEY)
                ),
                new NegativeTestCase(
                        getActionDescription(textField, "казахским"),
                        KAZ_TEXT_1,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_KAZAKHSTAN)
                ),
                new NegativeTestCase(
                        getActionDescription(textField, "белорусским"),
                        BEL_TEXT_1,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_BELARUS)
                ),
                new NegativeTestCase(
                        getActionDescription(textField, "узбекским"),
                        UZB_TEXT_1,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_UZBEKISTAN)
                )
        );
    }

    public static List<PositiveTestCase> getPositiveCases(TextField textField) {
        return Arrays.asList(
                new PositiveTestCase(
                        getActionDescription(textField, "украинским"),
                        UKR_TEXT_1,
                        RegionIDValues.UKRAINE.getId().longValue()
                ),
                new PositiveTestCase(
                        getActionDescription(textField, "турецким"),
                        TUR_TEXT_1,
                        RegionIDValues.TURKEY.getId().longValue()
                ),
                new PositiveTestCase(
                        getActionDescription(textField, "казахским"),
                        KAZ_TEXT_1,
                        RegionIDValues.KAZAKHSTAN.getId().longValue()
                ),
                new PositiveTestCase(
                        getActionDescription(textField, "белорусским"),
                        BEL_TEXT_1,
                        RegionIDValues.BELARUS.getId().longValue()
                ),
                new PositiveTestCase(
                        getActionDescription(textField, "узбекским"),
                        UZB_TEXT_1,
                        RegionIDValues.UZBEKISTAN.getId().longValue()
                )
        );
    }

    public static List<UpdateTestCase> getUpdateCases() {
        return Arrays.asList(
                new UpdateTestCase(
                        "украинский текст для украинского региона",
                        RegionIDValues.UKRAINE.getId().longValue(),
                        UKR_TEXT_1,
                        UKR_TEXT_2,
                        ExpectedResult.success()
                ),
                new UpdateTestCase(
                        "турецкий текст для турецкого региона",
                        RegionIDValues.TURKEY.getId().longValue(),
                        TUR_TEXT_1,
                        TUR_TEXT_2,
                        ExpectedResult.success()
                ),
                new UpdateTestCase(
                        "казахский текст для казахского региона",
                        RegionIDValues.KAZAKHSTAN.getId().longValue(),
                        KAZ_TEXT_1,
                        KAZ_TEXT_2,
                        ExpectedResult.success()
                ),
                new UpdateTestCase(
                        "немецкий текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        GER_TEXT_1,
                        GER_TEXT_2,
                        ExpectedResult.success()
                ),
                new UpdateTestCase(
                        "белорусский текст для белорусского региона",
                        RegionIDValues.BELARUS.getId().longValue(),
                        BEL_TEXT_1,
                        BEL_TEXT_2,
                        ExpectedResult.success()
                ),
                new UpdateTestCase(
                        "узбекский текст для узбекского региона",
                        RegionIDValues.UZBEKISTAN.getId().longValue(),
                        UZB_TEXT_1,
                        UZB_TEXT_2,
                        ExpectedResult.success()
                ),
                new UpdateTestCase(
                        "украинский текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT,
                        UKR_TEXT_2,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_UKRAINE))
                ),
                new UpdateTestCase(
                        "турецкий текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT,
                        TUR_TEXT_2,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_TURKEY))
                ),
                new UpdateTestCase(
                        "казахский текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT,
                        KAZ_TEXT_2,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_KAZAKHSTAN))
                ),
                new UpdateTestCase(
                        "белорусский текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT,
                        BEL_TEXT_2,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_BELARUS))
                ),
                new UpdateTestCase(
                        "узбекский текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT,
                        UZB_TEXT_2,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_UZBEKISTAN))
                )
        );
    }

    public static final class PositiveTestCase {
        public final String description;
        public final String text;
        public final Long regionId;

        PositiveTestCase(String description, String text, Long regionId)
        {
            this.description = description;
            this.text = text;
            this.regionId = regionId;
        }
    }

    public static final class NegativeTestCase {
        public final String description;
        public final String text;
        public final Notification expectedNotification;

        NegativeTestCase(String description, String text,
                Notification expectedNotification)
        {
            this.description = description;
            this.text = text;
            this.expectedNotification = expectedNotification;
        }
    }

    public static final class UpdateTestCase {
        public final String description;
        public final Long regionId;
        public final String oldText;
        public final String newText;
        public final ExpectedResult expectedResult;

        UpdateTestCase(String description, Long regionId, String oldText, String newText,
                ExpectedResult expectedResult)
        {
            this.description = description;
            this.regionId = regionId;
            this.oldText = oldText;
            this.newText = newText;
            this.expectedResult = expectedResult;
        }
    }
}
