package ru.yandex.autotests.directapi.cases;

import java.util.Arrays;
import java.util.List;

import ru.yandex.autotests.direct.utils.model.RegionIDValues;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.enums.CampaignLanguage;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;

public class TextGeoMatchingWithCampLangCases {
    private static final String UKR_TEXT = "Ґорілка";
    private static final String TUR_TEXT = "Metin afiş";
    private static final String KAZ_TEXT = "арақ";
    private static final String BEL_TEXT = "Дорым падарўнкі";
    private static final String GER_TEXT = "Überschrift öß";
    private static final String RUS_TEXT_1 = "водка";
    private static final String RUS_TEXT_2 = "квас";
    private static final String ENG_TEXT = "whiskey";

    public enum TextField {
        TEXT("текстом"),
        TITLE("заголовком"),
        TITLE2("заголовком 2");

        final String fieldDescription;

        TextField(String fieldDesription) {
            this.fieldDescription = fieldDesription;
        }
    }

    private static String getActionDescription(TextField textField, String langName) {
        return String.format("Добавление баннера с %s %s", langName, textField.fieldDescription);
    }

    public static List<NegativeTestCase> getNegativeCases(TextField textField) {
        return Arrays.asList(
                new NegativeTestCase(
                        getActionDescription(textField, "украинским"),
                        UKR_TEXT,
                        CampaignLanguage.BELARUSIAN,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_BELARUS)
                ),
                new NegativeTestCase(
                        getActionDescription(textField, "турецким"),
                        TUR_TEXT,
                        CampaignLanguage.UKRAINIAN,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_UKRAINE)
                ),
                new NegativeTestCase(
                        getActionDescription(textField, "казахским"),
                        KAZ_TEXT,
                        CampaignLanguage.TURKISH,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_TURKEY)
                ),
                new NegativeTestCase(
                        getActionDescription(textField, "белорусским"),
                        BEL_TEXT,
                        CampaignLanguage.KAZAKH,
                        new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_KAZAKHSTAN)
                )
        );
    }

    public static List<PositiveTestCase> getPositiveTestCases(TextField textField) {
        return Arrays.asList(
                new PositiveTestCase(
                        getActionDescription(textField, "украинским"),
                        UKR_TEXT,
                        CampaignLanguage.KAZAKH,
                        RegionIDValues.KAZAKHSTAN.getId().longValue()
                ),
                new PositiveTestCase(
                        getActionDescription(textField, "турецким"),
                        TUR_TEXT,
                        CampaignLanguage.UKRAINIAN,
                        RegionIDValues.UKRAINE.getId().longValue()
                ),
                new PositiveTestCase(
                        getActionDescription(textField, "казахским"),
                        KAZ_TEXT,
                        CampaignLanguage.TURKISH,
                        RegionIDValues.TURKEY.getId().longValue()
                ),
                new PositiveTestCase(
                        getActionDescription(textField, "немецким"),
                        GER_TEXT,
                        CampaignLanguage.BELARUSIAN,
                        RegionIDValues.BELARUS.getId().longValue()
                ),
                new PositiveTestCase(
                        getActionDescription(textField, "белорусским"),
                        BEL_TEXT,
                        CampaignLanguage.RUSSIAN,
                        RegionIDValues.RUSSIA.getId().longValue()
                )
        );
    }

    public static List<UpdateTestCase> getUpdateCases() {
        return Arrays.asList(
                new UpdateTestCase(
                        "текст для украинского региона",
                        RegionIDValues.UKRAINE.getId().longValue(),
                        UKR_TEXT,
                        TUR_TEXT,
                        CampaignLanguage.UKRAINIAN,
                        ExpectedResult.success()
                ),
                new UpdateTestCase("текст для турецкого региона",
                        RegionIDValues.TURKEY.getId().longValue(),
                        TUR_TEXT,
                        UKR_TEXT,
                        CampaignLanguage.TURKISH,
                        ExpectedResult.success()
                ),
                new UpdateTestCase("текст для казахского региона",
                        RegionIDValues.KAZAKHSTAN.getId().longValue(),
                        KAZ_TEXT,
                        ENG_TEXT,
                        CampaignLanguage.KAZAKH,
                        ExpectedResult.success()
                ),
                new UpdateTestCase("текст для белорусского региона",
                        RegionIDValues.BELARUS.getId().longValue(),
                        BEL_TEXT,
                        TUR_TEXT,
                        CampaignLanguage.BELARUSIAN,
                        ExpectedResult.success()
                ),
                new UpdateTestCase(
                        "текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT_1,
                        RUS_TEXT_2,
                        CampaignLanguage.UKRAINIAN,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_UKRAINE))
                ),
                new UpdateTestCase(
                        "текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT_1,
                        RUS_TEXT_2,
                        CampaignLanguage.TURKISH,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_TURKEY))
                ),
                new UpdateTestCase(
                        "текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT_1,
                        RUS_TEXT_2,
                        CampaignLanguage.KAZAKH,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_KAZAKHSTAN))
                ),
                new UpdateTestCase(
                        "текст для русского региона",
                        RegionIDValues.RUSSIA.getId().longValue(),
                        RUS_TEXT_1,
                        RUS_TEXT_2,
                        CampaignLanguage.BELARUSIAN,
                        ExpectedResult.errors(new Notification(6101, Api5ErrorDetails.AD_TEXT_ONLY_FOR_BELARUS))
                )
        );
    }

    public static final class PositiveTestCase {
        public final String description;
        public final String text;
        public final CampaignLanguage campLang;
        public final Long regionId;

        PositiveTestCase(String description, String text, CampaignLanguage campLang, Long regionId) {
            this.description = description;
            this.text = text;
            this.campLang = campLang;
            this.regionId = regionId;
        }
    }

    public static final class NegativeTestCase {
        public final String description;
        public final String text;
        public final CampaignLanguage campLang;
        public final Notification expectedNotification;

        NegativeTestCase(String description, String text, CampaignLanguage campLang,
                Notification expectedNotification)
        {
            this.description = description;
            this.text = text;
            this.campLang = campLang;
            this.expectedNotification = expectedNotification;
        }
    }

    public static final class UpdateTestCase {
        public final String description;
        public final Long regionId;
        public final String oldText;
        public final String newText;
        public final CampaignLanguage campLang;
        public final ExpectedResult expectedResult;

        UpdateTestCase(String description, Long regionId, String oldText, String newText, CampaignLanguage campLang,
                ExpectedResult expectedResult)
        {
            this.description = description;
            this.regionId = regionId;
            this.oldText = oldText;
            this.newText = newText;
            this.campLang = campLang;
            this.expectedResult = expectedResult;
        }
    }
}
