package ru.yandex.autotests.directapi.converters;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Created by andy-ilyin on 12.09.16.
 */
public final class MinusWordsSerializer {
    private static final ObjectMapper jsonMapper = new ObjectMapper();

    private enum Format {
        SEPARATED {
            @Override
            List<String> read(String serializedMinusWords) {
                return Arrays.asList(serializedMinusWords.split("[, ]"));
            }

            @Override
            String write(List<String> minusWords) {
                return minusWords.stream().collect(Collectors.joining(" "));
            }
        },
        JSON {
            @Override
            List<String> read(String serializedMinusWords) {
                String[] minusWords;
                try {
                    minusWords = jsonMapper.readValue(serializedMinusWords, String[].class);
                } catch (IOException e) {
                    throw new IllegalArgumentException(e);
                }
                return Arrays.asList(minusWords);
            }

            @Override
            String write(List<String> minusWords) {
                try {
                    return jsonMapper.writeValueAsString(minusWords);
                } catch (JsonProcessingException e) {
                    throw new IllegalArgumentException(e);
                }
            }
        };

        abstract List<String> read(String serializedMinusWords);

        abstract String write(List<String> minusWords);
    }

    private final Format format;

    private MinusWordsSerializer(Format format) {
        this.format = format;
    }

    private static final MinusWordsSerializer JSON_SERIALIZER = new MinusWordsSerializer(Format.JSON);
    private static final MinusWordsSerializer SEPARATED_SERIALIZER = new MinusWordsSerializer(Format.SEPARATED);

    public static final MinusWordsSerializer LEGACY_SERIALIZER = SEPARATED_SERIALIZER;
    public static final MinusWordsSerializer CURRENT_SERIALIZER = JSON_SERIALIZER;

    public final List<String> read(String s) {
        return Collections.unmodifiableList(format.read(s));
    }

    public final String write(List<String> minusWords) {
        return format.write(minusWords);
    }

    @Override
    public String toString() {
        return format.toString();
    }
}
