package ru.yandex.autotests.directapi.model.api5.bidmodifiers;

import java.util.ArrayList;
import java.util.List;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum;
import com.yandex.direct.api.v5.general.ExceptionNotification;
import org.hamcrest.Matcher;

import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.matchers.BeanCompareStrategy;
import ru.yandex.autotests.direct.utils.matchers.BeanEquals;
import ru.yandex.autotests.directapi.model.api5.general.ExceptionNotificationMap;

import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.isOneOf;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;

/**
 * Created by chicos on 31.07.15.
 */
public class ToggleExpectedResult extends ToggleResultMap {

    public ToggleExpectedResult() {
        super();
    }

    private BeanCompareStrategy resultCompareStrategy;

    public static ToggleExpectedResult successGroup() {
        return success(nullValue(), notNullValue());
    }

    public static ToggleExpectedResult successGroup(Long groupId) {
        return success(nullValue(), equalTo(groupId));
    }

    public static ToggleExpectedResult successGroup(Long groupId, BidModifierToggleTypeEnum type) {
        return success(nullValue(), equalTo(groupId), equalTo(type));
    }

    public static ToggleExpectedResult successCampaign() {
        return success(notNullValue(), nullValue());
    }

    public static ToggleExpectedResult successCampaign(Long campaignId) {
        return success(equalTo(campaignId), nullValue());
    }

    public static ToggleExpectedResult successCampaign(Long campaignId, BidModifierToggleTypeEnum type) {
        return success(equalTo(campaignId), nullValue(), equalTo(type));
    }

    private static ToggleExpectedResult success(Matcher campaignIdMatcher, Matcher groupIdMatcher) {
        return success(
                campaignIdMatcher,
                groupIdMatcher,
                isOneOf(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT,
                        BidModifierToggleTypeEnum.RETARGETING_ADJUSTMENT,
                        BidModifierToggleTypeEnum.SERP_LAYOUT_ADJUSTMENT,
                        BidModifierToggleTypeEnum.REGIONAL_ADJUSTMENT));
    }

    private static ToggleExpectedResult success(Matcher campaignIdMatcher, Matcher groupIdMatcher,
                                                Matcher typeMatcher) {
        ToggleExpectedResult map = new ToggleExpectedResult();
        map.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(CAMPAIGN_ID, campaignIdMatcher)
                .putFieldMatcher(AD_GROUP_ID, groupIdMatcher)
                .putFieldMatcher(TYPE, typeMatcher)
                .putFieldMatcher(WARNINGS, emptyIterable())
                .putFieldMatcher(ERRORS, emptyIterable());
        return map;
    }

    public static ToggleExpectedResult warnings(Long campaignId, Long groupId, ExceptionNotificationMap... warnings) {
        return warnings(equalTo(campaignId), equalTo(groupId), warnings);
    }

    public static ToggleExpectedResult warnings(ExceptionNotificationMap... warnings) {
        return warnings(notNullValue(), notNullValue(), warnings);
    }

    private static ToggleExpectedResult warnings(Matcher campaignIdMatcher, Matcher groupIdMatcher,
                                                 ExceptionNotificationMap... warnings) {
        ToggleExpectedResult map = new ToggleExpectedResult();
        List<BeanEquals> expectedWarningMatchers = new ArrayList<>();

        //replace details matcher to ignore code in messages like '#82518511 Warning message detail here'
        for (ExceptionNotificationMap warning : warnings) {
            BeanCompareStrategy strategy = new BeanCompareStrategy();
            if (((ExceptionNotification) warning.getBean()).getDetails() == null) {
                strategy.putFieldMatcher(ExceptionNotificationMap.DETAILS, nullValue());
            }
            BeanEquals warningMatcher = BeanEquals.beanEquals(warning.getBean()).accordingStrategy(strategy);

            //ignore message and details if ignore code set in properties
            if (DirectTestRunProperties.getInstance().getDirectApiIgnoreDetailForCodes() != null &&
                    DirectTestRunProperties.getInstance().getDirectApiIgnoreDetailForCodes()
                            .contains(String.valueOf(((ExceptionNotification) warning.getBean()).getCode()))) {
                warningMatcher.byFields(ExceptionNotificationMap.CODE);
            }
            expectedWarningMatchers.add(warningMatcher);
        }

        BeanEquals[] array = new BeanEquals[expectedWarningMatchers.size()];
        map.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(CAMPAIGN_ID, campaignIdMatcher)
                .putFieldMatcher(AD_GROUP_ID, groupIdMatcher)
                .putFieldMatcher(TYPE, isOneOf(BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT,
                        BidModifierToggleTypeEnum.RETARGETING_ADJUSTMENT,
                        BidModifierToggleTypeEnum.SERP_LAYOUT_ADJUSTMENT,
                        BidModifierToggleTypeEnum.REGIONAL_ADJUSTMENT))
                .putFieldMatcher(WARNINGS, containsInAnyOrder(expectedWarningMatchers.toArray(array)))
                .putFieldMatcher(ERRORS, emptyIterable());
        map.withWarnings(warnings);
        return map;
    }

    public static ToggleExpectedResult errors(ExceptionNotificationMap... errors) {
        ToggleExpectedResult map = new ToggleExpectedResult();
        List<BeanEquals> expectedErrorMatchers = new ArrayList<>();

        //replace details matcher to ignore code in messages like '#82518511 Error message detail here'
        for (ExceptionNotificationMap error : errors) {
            BeanCompareStrategy strategy = new BeanCompareStrategy();
            if (((ExceptionNotification) error.getBean()).getDetails() == null) {
                strategy.putFieldMatcher(ExceptionNotificationMap.DETAILS, nullValue());
            }

            BeanEquals errorMatcher = BeanEquals.beanEquals(error.getBean()).accordingStrategy(strategy);
            //ignore message and details if ignore code set in properties
            if (DirectTestRunProperties.getInstance().getDirectApiIgnoreDetailForCodes() != null &&
                    DirectTestRunProperties.getInstance().getDirectApiIgnoreDetailForCodes()
                            .contains(String.valueOf(((ExceptionNotification) error.getBean()).getCode()))) {
                errorMatcher.byFields(ExceptionNotificationMap.CODE);
            }
            expectedErrorMatchers.add(errorMatcher);
        }

        BeanEquals[] array = new BeanEquals[expectedErrorMatchers.size()];
        map.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(CAMPAIGN_ID, nullValue())
                .putFieldMatcher(AD_GROUP_ID, nullValue())
                .putFieldMatcher(TYPE, nullValue())
                .putFieldMatcher(WARNINGS, emptyIterable())
                .putFieldMatcher(ERRORS, containsInAnyOrder(expectedErrorMatchers.toArray(array)));
        map.withErrors(errors);
        return map;
    }

    public BeanCompareStrategy getActionResultCompareStrategy() {
        return resultCompareStrategy;
    }
}
