package ru.yandex.autotests.directapi.model.api5.bids;

import java.util.ArrayList;
import java.util.List;

import com.yandex.direct.api.v5.general.ExceptionNotification;

import ru.yandex.autotests.direct.utils.matchers.BeanCompareStrategy;
import ru.yandex.autotests.direct.utils.matchers.BeanEquals;
import ru.yandex.autotests.directapi.model.api5.general.ExceptionNotificationMap;

import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static ru.yandex.autotests.directapi.model.api5.general.NotificationDetailMatcher.equalToIgnoringCode;

/**
 * Created by ginger on 13.08.14.
 */
public class BidExpectedResult extends BidActionResultMap {

    public BidExpectedResult() {
        super();
    }

    private BeanCompareStrategy resultCompareStrategy;

    public static BidExpectedResult successWithKeywordId() {
        return success(BidBaseMap.KEYWORD_ID);
    }

    public static BidExpectedResult successWithCampaignId() {
        return success(BidBaseMap.CAMPAIGN_ID);
    }

    public static BidExpectedResult successWithAdGroupId() {
        return success(BidBaseMap.AD_GROUP_ID);
    }

    public static BidExpectedResult success(String fieldName) {
        BidExpectedResult map = new BidExpectedResult();
        map.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(fieldName, notNullValue())
                .putFieldMatcher(WARNINGS, emptyIterable())
                .putFieldMatcher(ERRORS, emptyIterable());
        return map;
    }

    public static BidExpectedResult warningsWithKeywordID(ExceptionNotificationMap... warnings) {
        return warnings(BidBaseMap.KEYWORD_ID, warnings);
    }

    public static BidExpectedResult warningsWithCampaignID(ExceptionNotificationMap... warnings) {
        return warnings(BidBaseMap.CAMPAIGN_ID, warnings);
    }

    public static BidExpectedResult warningsWithAdGroupID(ExceptionNotificationMap... warnings) {
        return warnings(BidBaseMap.AD_GROUP_ID, warnings);
    }

    public static BidExpectedResult warnings(String fieldName, ExceptionNotificationMap... warnings) {
        BidExpectedResult map = new BidExpectedResult();
        List<BeanEquals> expectedWarningMatchers = new ArrayList<>();

        //replace details matcher to ignore code in messages like '#82518511 Warning message detail here'
        for (ExceptionNotificationMap warning : warnings) {
            BeanCompareStrategy strategy = new BeanCompareStrategy();
            if (((ExceptionNotification) warning.getBean()).getDetails() == null) {
                strategy.putFieldMatcher(ExceptionNotificationMap.DETAILS, nullValue());
            }
            expectedWarningMatchers.add(
                    BeanEquals.beanEquals(warning.getBean())
                            .accordingStrategy(strategy)
            );
        }

        BeanEquals[] array = new BeanEquals[expectedWarningMatchers.size()];
        map.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(fieldName, notNullValue())
                .putFieldMatcher(WARNINGS, containsInAnyOrder(expectedWarningMatchers.toArray(array)))
                .putFieldMatcher(ERRORS, emptyIterable());
        map.withWarnings(warnings);
        return map;
    }

    public static BidExpectedResult errors(ExceptionNotificationMap... errors) {
        BidExpectedResult map = new BidExpectedResult();
        List<BeanEquals> expectedErrorMatchers = new ArrayList<>();

        //replace details matcher to ignore code in messages like '#82518511 Error message detail here'
        for (ExceptionNotificationMap error : errors) {
            BeanCompareStrategy strategy = new BeanCompareStrategy();
            if (((ExceptionNotification) error.getBean()).getDetails() == null) {
                strategy.putFieldMatcher(ExceptionNotificationMap.DETAILS, nullValue());
            }
            expectedErrorMatchers.add(
                    BeanEquals.beanEquals(error.getBean())
                            .accordingStrategy(strategy)
            );
        }

        BeanEquals[] array = new BeanEquals[expectedErrorMatchers.size()];
        map.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(BidBaseMap.CAMPAIGN_ID, nullValue())
                .putFieldMatcher(BidBaseMap.KEYWORD_ID, nullValue())
                .putFieldMatcher(BidBaseMap.AD_GROUP_ID, nullValue())
                .putFieldMatcher(WARNINGS, emptyIterable())
                .putFieldMatcher(ERRORS, containsInAnyOrder(expectedErrorMatchers.toArray(array)));
        map.withErrors(errors);
        return map;
    }

    public static BidExpectedResult warningWithError(ExceptionNotificationMap warning, ExceptionNotificationMap error) {
        BidExpectedResult map = new BidExpectedResult();
        List<BeanEquals> expectedErrorMatchers = new ArrayList<>();
        List<BeanEquals> expectedWarningMatchers = new ArrayList<>();
        expectedErrorMatchers.add(
                BeanEquals.beanEquals(error.getBean())
                        .ignoreFields(ExceptionNotificationMap.DETAILS)
                        .accordingStrategy(new BeanCompareStrategy()
                                .putFieldMatcher(
                                        ExceptionNotificationMap.DETAILS,
                                        equalToIgnoringCode((String) error.get(ExceptionNotificationMap.DETAILS)))));
        expectedWarningMatchers.add(
                BeanEquals.beanEquals(warning.getBean())
                        .ignoreFields(ExceptionNotificationMap.DETAILS)
                        .accordingStrategy(new BeanCompareStrategy()
                                .putFieldMatcher(
                                        ExceptionNotificationMap.DETAILS,
                                        equalToIgnoringCode((String) warning.get(ExceptionNotificationMap.DETAILS)))));
        BeanEquals[] errorArray = new BeanEquals[1];
        BeanEquals[] warningArray = new BeanEquals[1];
        map.resultCompareStrategy =
                new BeanCompareStrategy()
                        .putFieldMatcher(WARNINGS, containsInAnyOrder(expectedWarningMatchers.toArray(warningArray)))
                        .putFieldMatcher(ERRORS, containsInAnyOrder(expectedErrorMatchers.toArray(errorArray)));
        map.withWarnings(warning);
        map.withErrors(error);
        return map;
    }

    public BeanCompareStrategy getActionResultCompareStrategy() {
        return resultCompareStrategy;
    }
}
