package ru.yandex.autotests.directapi.model.api5.general;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.general.ActionResult;
import com.yandex.direct.api.v5.general.ExceptionNotification;
import org.hamcrest.Matcher;
import org.junit.Assert;

import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.matchers.BeanCompareStrategy;
import ru.yandex.autotests.direct.utils.matchers.BeanEquals;

import static java.util.Collections.emptyList;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;

/**
 * Created by chicos on 24.07.14.
 */
public class ExpectedResult extends ActionResultMap {

    private BeanCompareStrategy resultCompareStrategy;

    public ExpectedResult() {
        super();
    }

    public static ExpectedResult success() {
        return success(notNullValue());
    }

    public static ExpectedResult success(Long id) {
        return success(equalTo(id));
    }

    private static ExpectedResult success(Matcher idMatcher) {
        ExpectedResult map = new ExpectedResult();
        map.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(ID, idMatcher)
                .putFieldMatcher(WARNINGS, emptyIterable())
                .putFieldMatcher(ERRORS, emptyIterable());
        return map;
    }

    public static ExpectedResult warnings(Long id, ExceptionNotificationMap... warnings) {
        return warnings(equalTo(id), warnings);
    }

    public static ExpectedResult warnings(ExceptionNotificationMap... warnings) {
        return warnings(notNullValue(), warnings);
    }

    public static ExpectedResult warningsIgnoringDetailsAndMessage(ExceptionNotificationMap... warnings) {
        return create(notNullValue(), Arrays.asList(warnings), emptyList(), true, true);
    }

    public static ExpectedResult warningsIgnoringDetails(ExceptionNotificationMap... warnings) {
        return create(notNullValue(), Arrays.asList(warnings), emptyList(), true, false);
    }

    private static ExpectedResult warnings(Matcher idMatcher, ExceptionNotificationMap... warnings) {
        return create(idMatcher, Arrays.asList(warnings), emptyList(), false, false);
    }

    public static ExpectedResult errors(ExceptionNotificationMap... errors) {
        return create(nullValue(), emptyList(), Arrays.asList(errors), false, false);
    }

    public static ExpectedResult errorsIgnoringDetails(ExceptionNotificationMap... errors) {
        return create(nullValue(), emptyList(), Arrays.asList(errors), true, false);
    }

    public static ExpectedResult create(
            Matcher idMatcher,
            Collection<ExceptionNotificationMap> warnings,
            Collection<ExceptionNotificationMap> errors) {
        return create(idMatcher,
                warnings,
                errors,
                false,
                false);
    }
    
    private static ExpectedResult create(
            Matcher idMatcher,
            Collection<ExceptionNotificationMap> warnings,
            Collection<ExceptionNotificationMap> errors,
            boolean ignoreDetails,
            boolean ignoreMessage) {
        Assert.assertFalse("Neither warnings nor errors provided", warnings.isEmpty() && errors.isEmpty());
        ExpectedResult map = new ExpectedResult();
        List<BeanEquals> warningsMatchers = createMatchers(warnings, ignoreDetails, ignoreMessage);
        List<BeanEquals> errorsMatchers = createMatchers(errors, ignoreDetails, ignoreMessage);

        map.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(ID, idMatcher)
                .putFieldMatcher(WARNINGS, emptyIterable());
        map.resultCompareStrategy.putFieldMatcher(WARNINGS, warningsMatchers.isEmpty()
                ? emptyIterable()
                : containsInAnyOrder(warningsMatchers.toArray(new BeanEquals[0])));
        map.resultCompareStrategy.putFieldMatcher(ERRORS, errorsMatchers.isEmpty()
                ? emptyIterable()
                : containsInAnyOrder(errorsMatchers.toArray(new BeanEquals[0])));

        map.withWarnings(warnings.toArray(new ExceptionNotificationMap[0]));
        map.withErrors(errors.toArray(new ExceptionNotificationMap[0]));
        return map;
    }

    private static List<BeanEquals> createMatchers(Collection<ExceptionNotificationMap> notifications,
                                                   boolean ignoreDetails,
                                                   boolean ignoreMessage) {
        List<BeanEquals> expectedMatchers = new ArrayList<>();

        //replace details matcher to ignore code in messages like '#82518511 Warning message detail here'
        for (ExceptionNotificationMap notification : notifications) {
            BeanCompareStrategy strategy = new BeanCompareStrategy();
            if (((ExceptionNotification) notification.getBean()).getDetails() == null) {
                strategy.putFieldMatcher(ExceptionNotificationMap.DETAILS, nullValue());
            }
            BeanEquals matcher = BeanEquals.beanEquals(notification.getBean()).accordingStrategy(strategy);

            //ignore message and details if ignore code set in properties
            if (DirectTestRunProperties.getInstance().getDirectApiIgnoreDetailForCodes() != null &&
                    DirectTestRunProperties.getInstance().getDirectApiIgnoreDetailForCodes()
                            .contains(String.valueOf(((ExceptionNotification) notification.getBean()).getCode()))) {
                matcher.byFields(ExceptionNotificationMap.CODE);
            } else if (ignoreDetails && ignoreMessage) {
                matcher.byFields(ExceptionNotificationMap.CODE);
            } else if (ignoreDetails) {
                matcher.byFields(ExceptionNotificationMap.CODE, ExceptionNotificationMap.MESSAGE);
            }
            expectedMatchers.add(matcher);
        }
        return expectedMatchers;
    }

    public static ExpectedResult errorWithCode(int expectedErrorCode) {
        ExceptionNotificationMap map = new ExceptionNotificationMap()
                .withCode(expectedErrorCode);

        Matcher<Object> errorMatcher = BeanEquals.beanEquals(map.getBean())
                .byFields(ExceptionNotificationMap.CODE);

        ExpectedResult result = new ExpectedResult();
        result.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(ID, nullValue())
                .putFieldMatcher(WARNINGS, emptyIterable())
                .putFieldMatcher(ERRORS, contains(errorMatcher));
        return result;
    }

    public static ExpectedResult warningWithCode(int expectedWarningCode) {
        ExceptionNotificationMap map = new ExceptionNotificationMap()
                .withCode(expectedWarningCode);

        Matcher<Object> warningMatcher = BeanEquals.beanEquals(map.getBean())
                .byFields(ExceptionNotificationMap.CODE);

        ExpectedResult result = new ExpectedResult();
        result.resultCompareStrategy = new BeanCompareStrategy()
                .putFieldMatcher(ID, notNullValue())
                .putFieldMatcher(WARNINGS, contains(warningMatcher))
                .putFieldMatcher(ERRORS, emptyIterable());
        return result;
    }

    public BeanCompareStrategy getActionResultCompareStrategy() {
        return resultCompareStrategy;
    }

    public Matcher<ActionResult> getActionResultMatcher() {
        return BeanEquals.beanEquals((ActionResult) this.getBean())
                .accordingStrategy(this.getActionResultCompareStrategy());
    }
}
