package ru.yandex.autotests.directapi.model.geteventslog;

import java.util.Date;

import ru.yandex.autotests.direct.utils.model.ModelBeanMap;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.direct.utils.textresource.TextResourceFormatter;
import ru.yandex.autotests.directapi.model.banners.phrases.BannerPhraseInfoMap;

/**
 * User: mariabye
 * Date: 01.11.13
 */
public class EventsLogItemMap<T> extends ModelBeanMap<EventsLogItemMap> {

    private static final String EVENTS_LOG_ITEM = "EventsLogItem";
    public static final String CAMPAIGN_ID = "campaignID";
    public static final String BANNER_ID = "bannerID";
    public static final String PHRASE_ID = "phraseID";
    public static final String ACCOUNT_ID = "accountID";
    public static final String TEXT_DESCRIPTION = "textDescription";
    public static final String EVENT_TYPE = "eventType";
    public static final String TIMESTAMP = "timestamp";
    public static final String EVENT_NAME = "eventName";
    public static final String ATTRIBUTES = "attributes";

    public EventsLogItemMap(String packageName) {
        super(packageName, EVENTS_LOG_ITEM);
    }


    public EventsLogItemMap(T eventsLogItem) {
        super();
        setBean(eventsLogItem);
    }

    //region Default events
    public static EventsLogItemMap expectedMoneyInAccountEvent(EventsLogItemMap event) {
        event.put(ACCOUNT_ID, event.get(ACCOUNT_ID));
        event.put(CAMPAIGN_ID, null);
        event.put(EVENT_NAME, TextResourceFormatter.resource(EventName.MONEY_IN_ACCOUNT).toString());
        event.put(TEXT_DESCRIPTION, "Shared account (agent)");
        event.put(TIMESTAMP, null);
        return event;
    }

    public static EventsLogItemMap expectedMoneyOutAccountEvent(EventsLogItemMap event) {
        event.put(ACCOUNT_ID, event.get(ACCOUNT_ID));
        event.put(CAMPAIGN_ID, null);
        event.put(EVENT_NAME, TextResourceFormatter.resource(EventName.MONEY_OUT_ACCOUNT).toString());
        event.put(TEXT_DESCRIPTION, "Shared account (agent)");
        event.put(TIMESTAMP, null);
        return event;
    }

    public static EventsLogItemMap expectedMoneyWarningAccountEvent(EventsLogItemMap event) {
        event.put(ACCOUNT_ID, event.get(ACCOUNT_ID));
        event.put(CAMPAIGN_ID, null);
        event.put(EVENT_NAME, TextResourceFormatter.resource(EventName.MONEY_WARNING_ACCOUNT).toString());
        event.put(TEXT_DESCRIPTION, "Shared account (agent)");
        event.put(TIMESTAMP, null);
        return event;
    }

    public EventsLogItemMap moneyOutEvent(int campaignID) {
        return moneyOutEvent(campaignID, "");
    }

    public EventsLogItemMap moneyOutEvent(int campaignID, String timestamp) {
        put(CAMPAIGN_ID, campaignID);
        put(EVENT_TYPE, EventType.MONEYOUT);
        put(EVENT_NAME, "");
        put(TIMESTAMP, timestamp);
        return this;
    }

    public EventsLogItemMap moneyOutAccountEvent(int accountID) {
        return moneyOutAccountEvent(accountID, "");
    }

    public EventsLogItemMap moneyOutAccountEvent(int accountID, String timestamp) {
        put(ACCOUNT_ID, accountID);
        put(EVENT_TYPE, EventType.MONEYOUTACCOUNT);
        put(EVENT_NAME, "");
        put(TIMESTAMP, timestamp);
        return this;
    }

    public EventsLogItemMap moneyInEvent(int campaignID, Float payed) {
        return moneyInEvent(campaignID, payed, null);
    }

    public EventsLogItemMap moneyInEvent(int campaignID, Float payed, String currency) {
        put(CAMPAIGN_ID, campaignID);
        put(EVENT_TYPE, EventType.MONEYIN);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withPayed(payed)
                .withCurrency(currency));
        return this;
    }

    public EventsLogItemMap moneyInAccountEvent(int accountID, Float payed) {
        return moneyInAccountEvent(accountID, payed, null);
    }

    public EventsLogItemMap moneyInAccountEvent(int accountID, Float payed, String currency) {
        put(ACCOUNT_ID, accountID);
        put(EVENT_TYPE, EventType.MONEYINACCOUNT);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withPayed(payed)
                .withCurrency(currency));
        return this;
    }

    public EventsLogItemMap moneyWarningEvent(int campaignID, Float rest) {
        return moneyWarningEvent(campaignID, rest, null);
    }

    public EventsLogItemMap moneyWarningEvent(int campaignID, Float rest, String currency) {
        put(CAMPAIGN_ID, campaignID);
        put(EVENT_TYPE, EventType.MONEYWARNING);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withRest(rest)
                .withCurrency(currency));
        return this;
    }

    public EventsLogItemMap moneyWarningAccountEvent(int accountID, Float rest) {
        return moneyWarningAccountEvent(accountID, rest, null);
    }

    public EventsLogItemMap moneyWarningAccountEvent(int accountID, Float rest, String currency) {
        put(ACCOUNT_ID, accountID);
        put(EVENT_TYPE, EventType.MONEYWARNINGACCOUNT);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withRest(rest)
                .withCurrency(currency));
        return this;
    }

    public EventsLogItemMap warnMinPriceEvent(int campaignID, Float minPrice) {
        return warnMinPriceEvent(campaignID, minPrice, null);
    }

    public EventsLogItemMap warnMinPriceEvent(int campaignID, Float minPrice, Currency currency) {
        put(CAMPAIGN_ID, campaignID);
        put(EVENT_TYPE, EventType.WARNMINPRICE);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withMinPrice(minPrice)
                .withCurrency(currency));
        return this;
    }

    public EventsLogItemMap campaignFinishedEvent(int campaignID, Date finishDate) {
        put(CAMPAIGN_ID, campaignID);
        put(EVENT_TYPE, EventType.CAMPAIGNFINISHED);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withFinishDate(finishDate));
        return this;
    }

    public EventsLogItemMap pausedByDayBudgetEvent(int campaignID, String stopTime) {
        put(CAMPAIGN_ID, campaignID);
        put(EVENT_TYPE, EventType.PAUSEDBYDAYBUDGET);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withStopTime(stopTime));
        return this;
    }

    public EventsLogItemMap pausedByDayBudgetAccountEvent(int accountID, String stopTime) {
        put(ACCOUNT_ID, accountID);
        put(EVENT_TYPE, EventType.PAUSEDBYDAYBUDGETACCOUNT);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withStopTime(stopTime));
        return this;
    }

    public EventsLogItemMap bannerModeratedEvent(
            Integer campaignID, Long bannerID, String moderationResult, String isEditedByModerator)
    {
        put(CAMPAIGN_ID, campaignID);
        put(BANNER_ID, bannerID);
        put(EVENT_TYPE, EventType.BANNERMODERATED);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withModerationResult(moderationResult)
                .withIsEditedByModerator(isEditedByModerator));
        return this;
    }

    public EventsLogItemMap warnPlaceEvent(BannerPhraseInfoMap phraseInfo, String oldPlace) {
        put(CAMPAIGN_ID, phraseInfo.get(CAMPAIGN_ID));
        put(BANNER_ID, phraseInfo.get(BANNER_ID));
        put(PHRASE_ID, phraseInfo.get(PHRASE_ID));
        put(EVENT_TYPE, EventType.WARNPLACE);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        withAttributes(new EventsLogItemAttributesMap(super.getPackageName())
                .withOldPlace(oldPlace));
        return this;
    }

    public EventsLogItemMap lowCtrEvent(BannerPhraseInfoMap phraseInfo) {
        put(CAMPAIGN_ID, phraseInfo.get(CAMPAIGN_ID));
        put(BANNER_ID, phraseInfo.get(BANNER_ID));
        put(PHRASE_ID, phraseInfo.get(PHRASE_ID));
        put(EVENT_TYPE, EventType.LOWCTR);
        put(EVENT_NAME, "");
        put(TIMESTAMP, "");
        return this;
    }

    public EventsLogItemMap withTextDescription(String textDescription) {
        put(TEXT_DESCRIPTION, textDescription);
        return this;
    }

    public EventsLogItemMap withAttributes(EventsLogItemAttributesMap eventsLogItemAttributesMap) {
        put(ATTRIBUTES, eventsLogItemAttributesMap.getBean());
        return this;
    }

    public EventsLogItemMap withTimestamp(String timestamp) {
        put(TIMESTAMP, timestamp);
        return this;
    }

    public EventsLogItemMap withEventName(String eventName) {
        put(EVENT_NAME, eventName);
        return this;
    }
    //endregion
}
