package ru.yandex.autotests.directapi.rules;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Strings;
import org.apache.commons.collections.keyvalue.DefaultKeyValue;
import org.junit.rules.TestWatcher;
import org.junit.runner.Description;

import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.config.WsdlVersion;
import ru.yandex.autotests.directapi.model.Logins;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.api5.reports.ReportProcessingMode;
import ru.yandex.autotests.directapi.model.subscription.SubscriptionShortMap;
import ru.yandex.autotests.directapi.steps.UserSteps;
import ru.yandex.autotests.directapi.steps.common.PingJavaApiSteps;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.qatools.allure.annotations.Step;

/**
 * Created by IntelliJ IDEA.
 * User: mariabye
 * Date: 24.02.12
 * Time: 13:20
 * To change this template use File | Settings | File Templates.
 */
public class ApiSteps extends TestWatcher {

    public static final String USE_OPERATOR_UNITS_TRUE = "true";
    public static final String USE_OPERATOR_UNITS_FALSE = "false";
    public static final String USE_OPERATOR_UNITS_AUTO = "auto";

    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static DirectTestRunProperties properties = DirectTestRunProperties.getInstance();

    protected ConnectionConfig connectionConfig;
    protected RequestHeader requestHeader;
    public Set<Integer> staticResourceHolderForCampaignIDs;
    public Map<Long, String> staticResourceHolderForCampaignV5IDs;
    public Map<Long, String> staticResourceHolderForExtCampIDs;
    public Map<Long, String> staticResourceHolderForAdExtensionIds;
    public Set<DefaultKeyValue> staticResourceHolderForImages;
    public Map<String, String> staticResourceHolderForImagesV5;
    public Set<DefaultKeyValue> staticResourceHolderForRetargeting;
    public Map<Long, String> staticResourceHolderForRetargetingLists;
    public Set<Integer> staticResourceHolderForStat;
    public Set<DefaultKeyValue> staticResourceHolderForForecast;
    public Set<DefaultKeyValue> staticResourceHolderForWordstat;
    public Set<SubscriptionShortMap> staticResourceHolderForSubscription;

    public void addGeneratedDataToResourceHolder() {
        staticResourceHolderForCampaignIDs.addAll(userSteps.campaignStepsV4().bin.getBin());
        staticResourceHolderForCampaignV5IDs.putAll(userSteps.campaignSteps().bin.getBin());
        staticResourceHolderForAdExtensionIds.putAll(userSteps.adExtensionsSteps().bin.getBin());
        staticResourceHolderForImages.addAll(userSteps.imagesSteps().bin.getBin());
        staticResourceHolderForImagesV5.putAll(userSteps.imagesStepsV5().bin.getBin());
        staticResourceHolderForRetargeting.addAll(userSteps.retargetingSteps().bin.getBin());
        staticResourceHolderForStat.addAll(userSteps.statisticsSteps().bin.getBin());
        staticResourceHolderForForecast.addAll(userSteps.forecastSteps().bin.getBin());
        staticResourceHolderForSubscription.addAll(userSteps.subscriptionSteps().bin.getBin());
        staticResourceHolderForRetargetingLists.putAll(userSteps.retargetingListsSteps().bin.getBin());
        staticResourceHolderForExtCampIDs.putAll(userSteps.campExtSteps().bin.getBin());
    }

    @Override
    protected void starting(Description description) {
        staticResourceHolderForCampaignIDs = new HashSet<>();
        staticResourceHolderForCampaignV5IDs = new HashMap<>();
        staticResourceHolderForRetargetingLists = new HashMap<>();
        staticResourceHolderForAdExtensionIds = new HashMap<>();
        staticResourceHolderForImages = new HashSet<>();
        staticResourceHolderForImagesV5 = new HashMap<>();
        staticResourceHolderForRetargeting = new HashSet<>();
        staticResourceHolderForStat = new HashSet<>();
        staticResourceHolderForForecast = new HashSet<>();
        staticResourceHolderForWordstat = new HashSet<>();
        staticResourceHolderForSubscription = new HashSet<>();
        staticResourceHolderForExtCampIDs = new HashMap<>();
    }

    /**
     * Here must be called all clean test methods
     * In the future would be possible to specify Class Annotation to call only needed ones
     *
     * @param description class description
     */
    @Override
    protected void finished(Description description) {
        log.info("Удаляем статические тестовые данные");
        RequestHeader requestHeaderClone = requestHeader.clone();
        try {
            addGeneratedDataToResourceHolder();

            userSteps.statisticsSteps().clearBin(staticResourceHolderForStat).call();
            userSteps.forecastSteps().clearBin(staticResourceHolderForForecast).call();
            userSteps.forecastSteps().clearBin(staticResourceHolderForWordstat).call();
            userSteps.imagesSteps().clearBin(staticResourceHolderForImages).call();
            as(Logins.SUPER_LOGIN);
            userSteps.campaignStepsV4().clearBin(staticResourceHolderForCampaignIDs).call();
            userSteps.campaignSteps().clearBin(staticResourceHolderForCampaignV5IDs).call();
            userSteps.campExtSteps().clearBin(staticResourceHolderForExtCampIDs).call();
            userSteps.adExtensionsSteps().clearBin(staticResourceHolderForAdExtensionIds).call();
            userSteps.retargetingSteps().clearBin(staticResourceHolderForRetargeting).call();
            userSteps.retargetingListsSteps().clearBin(staticResourceHolderForRetargetingLists).call();
            userSteps.subscriptionSteps().clearBin(staticResourceHolderForSubscription).call();
            userSteps.imagesStepsV5().clearBin(staticResourceHolderForImagesV5).call();
        } catch (Exception e) {
            log.info("Ошибка при очистке тестовых данных", e);
        } finally {
            requestHeader = requestHeaderClone;
            reloadUserSteps();
            log.info("Тестовые данные удалены");
        }
    }

    @Deprecated
    public void saveCampaignToResource(Integer... campaignID) {
        staticResourceHolderForCampaignIDs.addAll(Arrays.asList(campaignID));
    }

    public UserSteps userSteps;

    protected void reloadUserSteps() {
        userSteps = UserSteps.getInstance(connectionConfig, requestHeader);
    }

    public ApiSteps() {
        this.connectionConfig = new ConnectionConfig(
                properties.getDirectAPIHost(),
                parseProtocolType(properties.getDirectApiDefaultContentType())
                );
        this.requestHeader = new RequestHeader(properties.getDirectAPILogin());
        reloadUserSteps();
    }

    private static ProtocolType parseProtocolType(String directApiDefaultContentType) {
        if (Strings.isNullOrEmpty(directApiDefaultContentType)) {
            return ProtocolType.JSON;
        } else {
            return ProtocolType.valueOf(directApiDefaultContentType);
        }
    }

    public ApiSteps url(String endPoint) {
        this.connectionConfig.url(endPoint);
        reloadUserSteps();
        return this;
    }

    public ApiSteps protocol(ProtocolType protocolType) {
        this.connectionConfig.protocolType(protocolType);
        reloadUserSteps();
        return this;
    }

    /**
     * Выставляет версию API и соответствующий ей пакет бинов
     *
     * @param version - версия API (4 | 104)
     * @return ApiSteps - степы API
     */
    @Step("Смена версии на {0}")
    public ApiSteps version(int version) {
        this.connectionConfig.version(version);
        this.connectionConfig.apiPortType(WsdlVersion.stubType.get(version));
        reloadUserSteps();
        return this;
    }

    public ApiSteps wsdl(Class apiPortType) {
        this.connectionConfig.apiPortType(apiPortType);
        reloadUserSteps();
        return this;
    }

    @Step("Смена логина пользователя. Login: {0}")
    public ApiSteps as(String login) {
        this.requestHeader = new RequestHeader(login);
        reloadUserSteps();
        return this;
    }

    @Step("Смена логина пользователя. Login: {0}, FakeLogin: {1}")
    public ApiSteps as(String login, String fakeLogin) {
        this.requestHeader = new RequestHeader(login);
        this.requestHeader.setFakeLogin(fakeLogin);
        reloadUserSteps();
        return this;
    }

    @Step("Смена локали запросов. Locale: {0}")
    public ApiSteps locale(String locale) {
        this.requestHeader.setLocale(locale);
        reloadUserSteps();
        return this;
    }

    @Step("Смена applicationID запросов. ApplicationID: {0}")
    public ApiSteps applicationID(String applicationID) {
        this.requestHeader.setApplicationID(applicationID);
        reloadUserSteps();
        return this;
    }

    @Step("Смена токена запросов. Token: {0}")
    public ApiSteps token(String token) {
        this.requestHeader.setToken(token);
        reloadUserSteps();
        return this;
    }

    public ApiSteps as(User account) {
        this.requestHeader = new RequestHeader(account);
        reloadUserSteps();
        return this;
    }

    public String type() {
        return connectionConfig.getPackage();
    }

    public String login() {
        return requestHeader.getLogin();
    }

    public String locale() {
        return requestHeader.getLocale();
    }

    @Step("Смена вечного токена запросов. persistent_token: {0}")
    public ApiSteps persistentToken(String persistentToken) {
        this.requestHeader.setPersistentToken(persistentToken);
        reloadUserSteps();
        return this;
    }

    @Step("Смена флага списания баллов с оператора. use_operator_units: {0}")
    public ApiSteps useOperatorUnits(String useOperatorUnits) {
        this.requestHeader.setUseOperatorUnits(useOperatorUnits);
        reloadUserSteps();
        return this;
    }

    @Step("Смена типа токена для API5. token_type: {0}")
    public ApiSteps tokenType(String tokenType) {
        this.requestHeader.setTokenType(tokenType);
        reloadUserSteps();
        return this;
    }

    @Step("Смена флага получения данных в микрофишках. return_money_in_micros: {0}")
    public ApiSteps returnMoneyInMicros(Boolean returnMoneyInMicros) {
        this.requestHeader.setReturnMoneyInMicros(returnMoneyInMicros);
        reloadUserSteps();
        return this;
    }

    @Step("Установка субклиента, от которого будет делаться запрос. Client-Login: {0}")
    public ApiSteps clientLogin(String clientLogin) {
        this.requestHeader.setClientLogin(clientLogin);
        reloadUserSteps();
        return this;
    }

    public PingJavaApiSteps pingJavaApiSteps() {
        return PingJavaApiSteps.getInstance(connectionConfig, requestHeader);
    }

    @Step("Смена флага получения строки заголовка в отчете. skip_report_header: {0}")
    public ApiSteps skipReportHeader(Boolean skipReportHeader) {
        this.requestHeader.setSkipReportHeader(skipReportHeader);
        reloadUserSteps();
        return this;
    }

    @Step("Смена флага получения строки с названиями полей в отчете. skip_column_header: {0}")
    public ApiSteps skipColumnHeader(Boolean skipColumnHeader) {
        this.requestHeader.setSkipColumnHeader(skipColumnHeader);
        reloadUserSteps();
        return this;
    }

    @Step("Смена флага получения последней строки с количеством строк в отчете. skip_report_summary: {0}")
    public ApiSteps skipReportSummary(Boolean skipReportSummary) {
        this.requestHeader.setSkipReportSummary(skipReportSummary);
        reloadUserSteps();
        return this;
    }

    @Step("Смена флага режим формирования отчета: online, offline или auto. processing_mode: {0}")
    public ApiSteps processingMode(ReportProcessingMode processingMode) {
        this.requestHeader.setProcessingMode(processingMode);
        reloadUserSteps();
        return this;
    }
    @Step("Форсирование использования перловой реализации сервиса. prefer_perl_implementation: {0}")
    public ApiSteps usePerlImplementation(Boolean usePerlImplementation) {
        this.requestHeader.setPreferPerlImplementation(usePerlImplementation);
        reloadUserSteps();
        return this;
    }

    public ApiSteps setPersistentTokenType() {
        return tokenType("persistent");
    }

}
