package ru.yandex.autotests.directapi.steps;

import java.util.Arrays;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.xmlrpc.XmlRpcException;
import org.hamcrest.Matcher;
import org.joda.time.DateTime;

import ru.yandex.autotests.balance.lib.environment.Environment;
import ru.yandex.autotests.balance.lib.integration.TestBalanceUtils;
import ru.yandex.autotests.balance.lib.xmlrpc.UnitTestLib;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.balanceclient.NotifyOrder2JSONRequest;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.CampaignFakeInfo;
import ru.yandex.autotests.directapi.darkside.exceptions.DarkSideException;
import ru.yandex.autotests.directapi.darkside.steps.DarkSideSteps;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.model.PaymentData;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.balance.ContractInfo;
import ru.yandex.autotests.directapi.model.balance.Invoice;
import ru.yandex.autotests.directapi.model.balance.OrderInfo;
import ru.yandex.autotests.directapi.model.balance.PaySys;
import ru.yandex.autotests.directapi.test.ClientIdLock;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Attachment;
import ru.yandex.qatools.allure.annotations.Step;

import static ch.lambdaj.Lambda.having;
import static ch.lambdaj.Lambda.on;
import static ch.lambdaj.Lambda.select;
import static ch.lambdaj.Lambda.sort;
import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.core.IsNull.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 10.10.13
 */
public class BalanceSteps {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static DirectTestRunProperties properties = DirectTestRunProperties.getInstance();
    private static final int SERVICE_ID = 7;
    private static final int FIRM_ID = 1;
    private UnitTestLib unitTestLib;
    private User operator;

    private static BalanceSteps _instance;
    private boolean ignoreDuplicatePersons;

    private BalanceSteps() {
        try {
            Environment balanceEnvironment = Environment.valueOf(properties.getBalanceStageType());
            unitTestLib = new UnitTestLib(balanceEnvironment);
            TestBalanceUtils.setEnvironment(balanceEnvironment);
        } catch (IllegalArgumentException e) {
            throw new DirectAPIException("Unknown balance environment: " + properties.getBalanceStageType(), e);
        }
    }

    public BalanceSteps withIgnoreDuplicatePersons(boolean ignore) {
        this.ignoreDuplicatePersons = ignore;
        return this;
    }

    public static BalanceSteps getInstance() {
        if (_instance == null) {
            _instance = new BalanceSteps();
        }
        return _instance;
    }

    @Attachment(value = "[Balance response]", type = "text/plain")
    public String attachResponse(String message) {
        return message;
    }

    @Step("Получение информации из Баланса для клиента {0}")
    public PaymentData getPaymentData(String login) {
        int clientID = Integer.parseInt(TestBalanceUtils.getClientIdByLogin(login));
        Object[] personDataArray = new Object[0];
        try {
            personDataArray = (Object[]) unitTestLib.getClientPersons(clientID);
            attachResponse(JsonUtils.toString(personDataArray));
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка получения данных по клиенту в балансе", e);
        }

        PaymentData paymentData = new PaymentData();
        if (personDataArray.length > 0) {
            List<Object> notArchivedPersons =
                    Stream.of(personDataArray)
                            .filter(x -> (((HashMap<String, String>) x).get("HIDDEN") == null))
                            .collect(Collectors.toList());
            if (!ignoreDuplicatePersons) {
                assumeThat("из Баланса получена информация по 1 плательщику", notArchivedPersons.size(), equalTo(1));
            }
            HashMap<String, String> personData = (HashMap<String, String>) notArchivedPersons.get(0);
            Integer personID = Integer.valueOf(personData.get("ID"));
            String personType = String.valueOf(personData.get("TYPE"));
            int paySysID = PaySys.fromValue(personType);

            paymentData.setPaySysID(paySysID);
            paymentData.setPersonID(personID);
        }
        return paymentData;
    }

    @Step("Проверяем/досоздаем плательщика - физ. лица для клиента {0}")
    public void verifyHasPhPerson(long clientID) {
        ClientIdLock lock = new ClientIdLock("verifyHasPhPerson" + clientID);
        lock.acquire(2);
        try {
            Object[] personDataArray = (Object[]) unitTestLib.getClientPersons((int) clientID);
            boolean createNew = Stream.of(personDataArray)
                    .map(o -> (Map<String, String>) o)
                    .filter(p -> p.get("HIDDEN") == null)
                    .noneMatch(p -> p.get("TYPE").equals("ph"));
            if (createNew) {
                createPersonPh((int) clientID);
            }
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка получения данных по клиенту в балансе", e);
        } finally {
            lock.release();
        }
    }

    public void createPerson(String login) {
        int clientID = Integer.parseInt(TestBalanceUtils.getClientIdByLogin(login));
        try {
            unitTestLib.createPersonPhDefault(clientID);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка создания плательщика в балансе для " + login);
        }
    }

    public void createHryvnasPerson(int clientID) {
        try {
            unitTestLib.createPerson("pu", clientID);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка создания плательщика в балансе для " + clientID);
        }
    }

    public void createByPhPerson(int clientID) {
        try {
            unitTestLib.createPerson("byp", clientID);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка создания плательщика в балансе для " + clientID);
        }
    }

    public OrderInfo getOrderInfo(Integer campaignID) {
        try {
            Object[] results =
                    (Object[]) unitTestLib.getOrdersInfo(Arrays.asList(7), Arrays.asList(campaignID));

            assumeThat("из Баланса вернулась информация по заказу " + campaignID, results.length, greaterThan(0));
            log.info("Параметры заказа в балансе");
            log.info(JsonUtils.toString(results));
            OrderInfo orderInfo = JsonUtils.getObject(JsonUtils.toString(results[0]), OrderInfo.class);
            return orderInfo;
        } catch (XmlRpcException e) {
            throw new DirectAPIException("Ошибка получения данных по заказам", e);
        }
    }

    public void orderAmountShouldBe(int campaignID, Matcher<Float> amount) {
        try {
            OrderInfo orderInfo = getOrderInfo(campaignID);
            TestSteps.assertThat("сумма на заказе - верна", Float.valueOf(orderInfo.getSum()), amount);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка получения данных по счету в балансе", e);
        }
    }

    /**
     * new BalanceSteps().operator("myoperator").createInvoice(1111)
     *
     * @param requestID
     * @return
     */
    @Step("Выставляем счет в баланс: requestID = {0}")
    public int createInvoice(String requestID) {
        assumeThat("задан оператор для работы с методами баланса", operator, not(nullValue()));
        try {
            int invoiceID = 0;
            if (operator.getContractID() != null) {
                invoiceID = unitTestLib.createInvoice(
                        operator.getPassportUID(),
                        requestID,
                        operator.getPaySysID(),
                        operator.getPersonID(),
                        Integer.valueOf(operator.getContractID()),
                        false);// если выставлятсья в кредит (true) - то будут разные счета по суб-клиентам
            } else {
                invoiceID = unitTestLib.createInvoice(
                        operator.getPassportUID(),
                        requestID,
                        operator.getPaySysID(),
                        operator.getPersonID());
            }
            log.info("Выставлен счет. InvoiceID = " + invoiceID);
            return invoiceID;
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка выставления счета в балансе", e);
        }
    }

    @Step("Включаем счет: invoiceId = {0}")
    public void turnOnInvoice(int invoiceId) {
        try {
            unitTestLib.turnOnInvoice(invoiceId);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка включения счета " + invoiceId, e);
        }
    }

    public void oebsPayment(int invoiceId) {
        try {
            unitTestLib.oebsPayment(invoiceId, null, null);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка включения счета (oebsPayment) " + invoiceId, e);
        }
    }

    public void requestAmountShouldBe(String requestID, Matcher<Float> amount) {
        assumeThat("задан оператор для работы с методами баланса", operator, not(nullValue()));
        int invoiceId = createInvoice(requestID);
        int operatorUID = Integer.parseInt(operator.getPassportUID());
        HashMap<String, String> result = null;
        try {
            result = (HashMap<String, String>) unitTestLib.getInvoice(operatorUID, invoiceId);
            log.info("Параметры счета в балансе");
            log.info(JsonUtils.toString(result));
            Invoice invoice = JsonUtils.getObject(JsonUtils.toString(result), Invoice.class);
            TestSteps.assertThat("сумма счета в балансе верна", invoice.getTotalSum(), amount);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка получения данных по счету в балансе", e);
        }
    }


    public Float getClientOverdraft(String login) {

        Double result = TestBalanceUtils.getClientOverdraft(
                TestBalanceUtils.getClientIdByLogin(login), BalanceSteps.SERVICE_ID,
                BalanceSteps.FIRM_ID);
        return result.floatValue();
    }

    public Float getAvailableOverdraft(String login, int orderID) {
        Object response;
        try {
            response =
                    unitTestLib.getMobilePaysysByPassport(
                            BalanceSteps.SERVICE_ID, String.valueOf(orderID), User.get(login).getPassportUID());
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка получения информации об овердрафте из Баланса", e);
        }
        HashMap<String, Object> result = (HashMap<String, Object>) response;
        HashMap<String, String> overdraftInfo = (HashMap<String, String>) result.get("overdraft");
        return Float.parseFloat(overdraftInfo.get("available_sum"));
    }

    public void createPersonPh(int clientID) {
        try {
            unitTestLib.createPersonPhDefault(clientID);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка создания плательщика в балансе", e);
        }
    }

    public void setClientOverdraft(String login) {
        TestBalanceUtils.turnOnAllInvoicesByLogin(login);
        TestBalanceUtils.setClientOverdraft(
                TestBalanceUtils.getClientIdByLogin(login), BalanceSteps.SERVICE_ID,
                BalanceSteps.FIRM_ID, 99999);

    }

    public void setClientOverdraft(String clientID, int value) {
        TestBalanceUtils.setClientOverdraft(clientID, BalanceSteps.SERVICE_ID, BalanceSteps.FIRM_ID, value);
    }

    public void setClientOverdraftByLogin(String login, int value) {
        TestBalanceUtils.turnOnAllInvoicesByLogin(login);
        TestBalanceUtils.setClientOverdraft(
                TestBalanceUtils.getClientIdByLogin(login),
                BalanceSteps.SERVICE_ID,
                BalanceSteps.FIRM_ID,
                value
        );
    }

    public void setClientOverdraftByLogin(String login, int value, Currency currency) {
        TestBalanceUtils.turnOnAllInvoicesByLogin(login);
        TestBalanceUtils.setClientOverdraft(
                TestBalanceUtils.getClientIdByLogin(login),
                BalanceSteps.SERVICE_ID,
                BalanceSteps.FIRM_ID,
                value,
                currency.value()
        );
    }

    @Step("Получение информации о всех контрактах у пользователя ClientID = {0}")
    public List<ContractInfo> getAllContracts(String clientID) {
        List results =
                unitTestLib.getClientContracts(Integer.parseInt(clientID), 0);
        attachResponse(JsonUtils.toString(results));
        List<ContractInfo> contracts =
                Arrays.asList((ContractInfo[])
                        JsonUtils.getObject(JsonUtils.toString(results), new ContractInfo[0].getClass()));
        return contracts;
    }

    public String getDefaultContract(String clientID) {
        return getDefaultContract(clientID, 1);
    }

    public String getInternalContractID(String clientID, String externalContractID) {
        List<ContractInfo> contracts = getAllContracts(clientID);
        List<ContractInfo> creditContracts =
                select(contracts, having(on(ContractInfo.class).getExternalID(), equalTo(externalContractID)));
        if (creditContracts.size() == 0) {
            throw new DirectAPIException("Отсутствует нужный контракт у агентства");
        }
        return creditContracts.get(0).getId();
    }

    public String getDefaultContract(String clientID, int contractNumber) {
        // если в этом методе возникает ошибка, надо починить (создать контракт), записать юзернейм
        // в BalanceContractsSmokeTest и попросить Баланс создавать такой контракт сразу после переналивки базы
        // (см. комментарий в BalanceContractsSmokeTest)

        List<ContractInfo> contracts = getAllContracts(clientID);
        assumeThat("у агентства есть контракты", contracts, not(empty()));

        List<ContractInfo> creditContracts =
                select(contracts, allOf(
                        having(on(ContractInfo.class).getCreditLimits(), notNullValue()),
                        having(on(ContractInfo.class).getPaymentType(), equalTo("3")),
                        having(on(ContractInfo.class).getIsActive(), equalTo("1"))));

        assumeThat("у агентства есть контракты с постоплатой", creditContracts, not(empty()));

        int currentContractNumber = 1;
        for (ContractInfo contract : creditContracts) {
            if (contract.getCreditLimits()[0].getLimit() == 100000000f) {
                if (currentContractNumber == contractNumber) {
                    return contract.getExternalID();
                } else {
                    currentContractNumber++;
                }
            }
        }
        throw new DirectAPIException("Отсутствует нужный контракт у агентства");
    }

    public String getContractWithSmallLimit(String clientID) {
        List<ContractInfo> contracts = getAllContracts(clientID);
        List<ContractInfo> creditContracts =
                select(contracts, allOf(
                        having(on(ContractInfo.class).getCreditLimits(), notNullValue()),
                        having(on(ContractInfo.class).getPaymentType(), equalTo("3")),
                        having(on(ContractInfo.class).getIsActive(), equalTo("1"))));
        for (ContractInfo contract : creditContracts) {
            if (contract.getCreditLimits()[0].getLimit() == 1f) {
                return contract.getExternalID();
            }
        }
        throw new DirectAPIException("Отсутствует нужный контракт у агентства");
    }

    public String getContractPayBefore(String clientID) {
        List<ContractInfo> contracts = getAllContracts(clientID);
        List<ContractInfo> creditContracts =
                select(contracts, allOf(
                        having(on(ContractInfo.class).getPaymentType(), equalTo("2")),
                        having(on(ContractInfo.class).getIsActive(), equalTo("1"))));
        if (creditContracts.size() == 0) {
            throw new DirectAPIException("Отсутствует нужный контракт у агентства");
        }
        return creditContracts.get(0).getExternalID();
    }

    public String getOldContract(String clientID) {
        List results =
                unitTestLib.getClientContracts(Integer.parseInt(clientID), 0, new GregorianCalendar(2010, 1, 1));
        attachResponse(JsonUtils.toString(results));
        List<ContractInfo> contracts =
                Arrays.asList((ContractInfo[])
                        JsonUtils.getObject(JsonUtils.toString(results), new ContractInfo[0].getClass()));
        List<ContractInfo> creditContracts = contracts.stream()
                .filter(contract -> contract.getFinishDt() != null &&
                        DateTime.parse(contract.getFinishDt()).isBeforeNow())
                .collect(Collectors.toList());
        if (creditContracts.size() == 0) {
            throw new DirectAPIException("Отсутствует нужный контракт у агентства");
        }
        return creditContracts.get(0).getExternalID();
    }

    @Step("Получение информации о контрактах у пользователя ClientID = {0}")
    public ContractInfo[] getContractInfo(String clientID, Integer personID) {
        if (clientID == null || personID == null) {
            throw new DirectAPIException(
                    "Не задан один из параметров: clientID = " + clientID + ", personID = " + personID);
        }
        List results = unitTestLib.getClientContracts(Integer.parseInt(clientID), personID);
        attachResponse(JsonUtils.toString(results));
        List<ContractInfo> contracts =
                Arrays.asList((ContractInfo[])
                        JsonUtils.getObject(JsonUtils.toString(results), new ContractInfo[0].getClass()));
        contracts = sort(contracts, on(ContractInfo.class).getId());
        List<ContractInfo> creditContracts = sort(
                select(contracts, having(on(ContractInfo.class).getCreditLimits(), notNullValue())),
                on(ContractInfo.class).getId());

        return creditContracts.toArray(new ContractInfo[]{});
    }

    public void repaymentInvoicesByContract(String internalContractID) {
        TestBalanceUtils.createRepaymentInvoicesByContract(internalContractID);
    }

    public void payAllInvoicesByContract(String internalContractID) {
        TestBalanceUtils.payAllInvoicesByContract(internalContractID);
    }

    public double getCurrentCurrencyRate(Currency currency) {
        try {
            return unitTestLib.getCurrencyRateDouble(currency.toString());
        } catch (Exception e) {
            throw new DirectAPIException("Не удалось получить из баланса текущий курс для валюты: " + currency);
        }
    }

    public double getCurrencyRate(Currency currency, DateTime dateTime) {
        try {
            return unitTestLib.getCurrencyRateDouble(currency.toString(), dateTime.toDate());
        } catch (Exception e) {
            throw new DirectAPIException(
                    "Не удалось получить из баланса курс для валюты: " + currency + " за дату " + dateTime);
        }
    }

    public void synchronizeWithBalance(int orderID) {
        OrderInfo orderInfo = getOrderInfo(orderID);
        String consumeQty = orderInfo.getSum();
        String consumeMoneyQty = orderInfo.getConsumeMoneyQty();
        String tid = Long.toString(DateTime.now().getMillis() * 10);
        DarkSideSteps darkSideSteps = new DarkSideSteps();
        CampaignFakeInfo campaignFakeInfo = darkSideSteps.getCampaignFakeSteps().fakeGetCampaignParams(orderID);
        String currency = campaignFakeInfo.getCurrency();
        if (currency == null) {
            currency = "";
        }
        try {
            darkSideSteps.getBalanceClientNotifyOrderJsonSteps().notifyOrderNoErrors(new NotifyOrder2JSONRequest()
                    .withServiceId(NotifyOrder2JSONRequest.DIRECT_SERVICE_ID)
                    .withServiceOrderId((long) orderID)
                    .withTid(tid)
                    .withConsumeQty(consumeQty)
                    .withConsumeMoneyQty(consumeMoneyQty)
                    .withProductCurrency(currency)
                    // не совсем честно тут выставлять consumeQty, т.к. должна быть сумма всех зачислений под ОС,
                    // но это лучше чем 0, и работает в случае, когда на кампаниях под ОС нет зачислений
                    .withTotalConsumeQty(campaignFakeInfo.getType().equals("wallet") ? consumeQty : "0")

            );
        } catch (DarkSideException e) {
            log.info("Ошибка при посылке NotifyOrder2:");
            e.printStackTrace();
        }
    }

    public void synchronizeWithBalance(Long orderID) {
        synchronizeWithBalance(orderID.intValue());
    }

    public void synchronizeCompletionQty(int orderID) {
        Float completionQty = getOrderInfo(orderID).getSpent();
        DarkSideSteps darkSideSteps = new DarkSideSteps();
        darkSideSteps.getCampaignFakeSteps().setSumSpent(orderID, completionQty);
    }

    public void synchronizeCompletionQty(Long orderID) {
        synchronizeCompletionQty(orderID.intValue());
    }

    public void doCampaigns(int campaignID, double sum) {
        try {
            int productId = new DarkSideSteps().getCampaignFakeSteps().fakeGetCampaignParams(campaignID).getProductID();
            // черная магия-копипаста из UnitTestLib doDirectDiscountForClient
            // в реальности, вероятно, для баяна и квазивалютных продуктов (новые KZT, возможно BYN) должно быть false
            boolean isMoney = productId != 1475;
            unitTestLib.doCampaigns(SERVICE_ID, String.valueOf(campaignID), sum, isMoney);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка создания откруток в балансе", e);
        }
    }

    public void doCampaigns(Long campaignID, double sum) {
        doCampaigns(campaignID.intValue(), sum);
    }

    public boolean checkOverdueInvoices(String login) {
        String clientID = TestBalanceUtils.getClientIdByLogin(login);
        try {
            HashMap response = (HashMap) ((Object[]) unitTestLib
                    .executeSQL(String.format("select count(1) from t_invoice where client_id = %s and " +
                                    "type = \'overdraft\' and payment_term_dt < sysdate and receipt_sum < consume_sum",
                            clientID)))[0];
            if ((response.values().size() == 1) && (response.containsKey("COUNT(1)"))) {
                Integer value = (Integer) response.get("COUNT(1)");
                return !value.equals(0);
            } else {
                throw new DirectAPIException("Неожиданный формат ответа из Баланса");
            }
        } catch (XmlRpcException e) {
            throw new DirectAPIException("Ошибка запроса информации о просроченных овердрафтных счетах из Баланса", e);
        }
    }

    @Step("Создание клиента в Балансе")
    public Integer createClientId(Integer clientId, Currency currency, Integer regionId) {
        try {
            return unitTestLib.createClient(clientId, false, null, regionId, currency.toString(), null, null);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка создания ClientID в Балансе");
        }
    }

    @Step("Привязка ClientID к uid'у в Балансе")
    public void associateClientIdToLogin(Integer clientId, String uid) {
        boolean result;
        try {
            result = unitTestLib.associateLogin(clientId, uid);
        } catch (XmlRpcException e) {
            throw new DirectAPIException("Ошибка привязки ClientID к uid'у в Балансе");
        }
        assertThat("привязка ClientID к uid'у завершилась успехом", result, equalTo(true));
    }

    public BalanceSteps operator(User user) {
        this.operator = user;
        return this;
    }

    public String getNotificationUrl() {
        try {
            return unitTestLib.getXmlrpcProxyUrlDirect();
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка получения урла для нотификаций из Баланса");
        }
    }

    public void setNotificationUrl(String url) {
        try {
            unitTestLib.setXmlrpcProxyUrlDirect(url);
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка установки урла для нотификаций из Баланса");
        }
    }
}
