package ru.yandex.autotests.directapi.steps;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import com.yandex.direct.api.v5.adimages.AdImageActionResult;
import com.yandex.direct.api.v5.bids.BidActionResult;
import com.yandex.direct.api.v5.general.ActionResult;
import com.yandex.direct.api.v5.general.SetBidsActionResult;
import com.yandex.direct.api.v5.reports.ReportDefinition;
import org.apache.commons.beanutils.BeanMap;
import org.apache.commons.lang3.StringUtils;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;

import ru.yandex.autotests.direct.db.steps.DirectJooqDbSteps;
import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.direct.utils.converter.BeanMapToBeanConverter;
import ru.yandex.autotests.direct.utils.matchers.BeanEquals;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorMatcher;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonError;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonErrorMatcher;
import ru.yandex.autotests.directapi.apiclient.errors.Api5XmlError;
import ru.yandex.autotests.directapi.apiclient.errors.Api5XmlErrorMatcher;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.errors.AxisErrorMatcher;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.apiclient.version4.BaseApiV4Client;
import ru.yandex.autotests.directapi.apiclient.version4.JsonClient;
import ru.yandex.autotests.directapi.apiclient.version4.SoapClient;
import ru.yandex.autotests.directapi.apiclient.version5.BaseApiV5Client;
import ru.yandex.autotests.directapi.apiclient.version5.InvalidJSONClientV5;
import ru.yandex.autotests.directapi.apiclient.version5.JSONClientV5;
import ru.yandex.autotests.directapi.apiclient.version5.JSONRequestV5;
import ru.yandex.autotests.directapi.apiclient.version5.MethodInvocationResult;
import ru.yandex.autotests.directapi.apiclient.version5.ReportsClient;
import ru.yandex.autotests.directapi.apiclient.version5.SOAPClientV5;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.apiclient.version5.XMLRequestV5;
import ru.yandex.autotests.directapi.darkside.steps.DarkSideSteps;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.matchers.axiserror.ApiResponse;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.adimages.AdImageExpectedResult;
import ru.yandex.autotests.directapi.model.api5.bids.BidExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlApi5Error;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.JavaOrPerlSetBidsExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.SetBidsExpectedResult;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportProcessingMode;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;

import static ch.lambdaj.Lambda.convert;
import static ch.lambdaj.Lambda.having;
import static ch.lambdaj.Lambda.on;
import static ch.lambdaj.Lambda.select;
import static com.jayway.restassured.module.jsv.JsonSchemaValidator.matchesJsonSchemaInClasspath;
import static java.util.Arrays.asList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.either;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.fail;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * User: mariabye
 * Date: 25.06.13
 */
public abstract class BaseApiSteps {
    protected ConnectionConfig connectionConfig;
    protected RequestHeader requestHeader;
    protected DarkSideSteps darkSideSteps;
    protected static DirectTestRunProperties properties = DirectTestRunProperties.getInstance();

    protected BaseApiSteps(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        this.connectionConfig = connectionConfig;
        this.requestHeader = requestHeader;
    }

    public void setConnectionConfig(ConnectionConfig connectionConfig) {
        this.connectionConfig = connectionConfig;
    }

    public void setRequestHeader(RequestHeader requestHeader) {
        this.requestHeader = requestHeader;
    }

    public JsonClient jsonClient() {
        return new JsonClient(connectionConfig, requestHeader);
    }

    public SoapClient soapClient() {
        return new SoapClient(connectionConfig, requestHeader);
    }

    public SOAPClientV5 soapClientV5() {
        return new SOAPClientV5(connectionConfig, requestHeader);
    }

    public JSONClientV5 jsonClientV5() {
        return new JSONClientV5(connectionConfig, requestHeader);
    }

    public InvalidJSONClientV5 invalidJSONClientV5() {
        return new InvalidJSONClientV5(connectionConfig, requestHeader);
    }

    public ReportsClient reportsClientXml() {
        return new ReportsClient(connectionConfig, requestHeader, new XMLRequestV5(ReportDefinition.class));
    }

    public ReportsClient reportsClientJson() {
        return new ReportsClient(connectionConfig, requestHeader, new JSONRequestV5());
    }

    public DarkSideSteps getDarkSideSteps() {
        if (darkSideSteps == null) {
            darkSideSteps = new DarkSideSteps(properties);
        }
        return darkSideSteps;
    }

    public DirectJooqDbSteps getDirectJooqDbSteps() {
        return getDarkSideSteps().getDirectJooqDbSteps();
    }

    public BaseApiV4Client defaultClient() {
        ProtocolType protocolType = connectionConfig.getProtocolType();
        switch (protocolType) {
            case JSON:
                return new JsonClient(connectionConfig, requestHeader);
            case SOAP:
                return new SoapClient(connectionConfig, requestHeader);
        }
        throw new DirectAPIException("Установлен неизвестный тип клиента для выполнения запросов V4 - " + protocolType);
    }

    public BaseApiV5Client defaultClientV5() {
        ProtocolType protocolType = connectionConfig.getProtocolType();
        switch (protocolType) {
            case JSON:
                return new JSONClientV5(connectionConfig, requestHeader);
            case SOAP:
                return new SOAPClientV5(connectionConfig, requestHeader);
            case INVALID_JSON:
                return new InvalidJSONClientV5(connectionConfig, requestHeader);
        }
        throw new DirectAPIException("Установлен неизвестный тип клиента для выполнения запросов V5 - " + protocolType);
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetErrorOn(
            String assertMessage,
            String methodName,
            Object params,
            AxisError expectedError)
    {
        try {
            if (params != null && BeanMap.class.isAssignableFrom(params.getClass())) {
                defaultClient().invokeMethod(methodName, ((BeanMap) params).getBean());
            } else {
                defaultClient().invokeMethod(methodName, params);
            }
            assertThat("получена ошибка", null, equalTo(expectedError));
        } catch (AxisError axisFault) {
            assertThat("совпал текст ошибки",
                    axisFault,
                    AxisErrorMatcher.equalToIgnoreLocale(expectedError,
                            properties.getDirectApiIgnoreDetailForCodes()));
        }
    }

    public void shouldGetErrorOn(
            Method methodName,
            Object params,
            AxisError expectedError)
    {
        shouldGetErrorOn("", methodName.toString(), params, expectedError);
    }

    public <T extends BeanMap> void shouldGetErrorOn(
            String assertMessage,
            Method methodName,
            T params,
            AxisError expectedError)
    {
        if (params != null) {
            shouldGetErrorOn(assertMessage, methodName.toString(), params.getBean(), expectedError);
        } else {
            shouldGetErrorOn(assertMessage, methodName.toString(), null, expectedError);
        }
    }

    public <T extends BeanMap> void shouldGetErrorOn(
            String assertMessage,
            Method methodName,
            T[] params,
            AxisError expectedError)
    {
        if (params != null) {
            shouldGetErrorOn(assertMessage, methodName.toString(),
                    convert(params, new BeanMapToBeanConverter<>()).toArray(), expectedError);
        } else {
            shouldGetErrorOn(assertMessage, methodName.toString(), null, expectedError);
        }
    }

    public <T extends BeanMap> void shouldGetErrorOn(
            Method methodName,
            T params,
            AxisError expectedError)
    {
        shouldGetErrorOn("", methodName, params, expectedError);
    }

    public <T extends BeanMap> void shouldGetErrorOn(
            Method methodName,
            T[] params,
            AxisError expectedError)
    {
        shouldGetErrorOn("", methodName, params, expectedError);
    }

    @Step("Валидация по схеме json ответа метода {0}")
    public void validateJsonResponseBySchema(Method method, Object args) {
        String schemaFileName = "schemas/" + method + ".json";
        String response = jsonClient().sendRequest(method, args);
        assertThat("json-response прошел валидацию по схеме",
                response, matchesJsonSchemaInClasspath(schemaFileName));
    }

    @Step(value = "Вызываем метод {0} и ожидаем получить ошибку в ответе.")
    public <T> T shouldGetMultipleErrorOn(
            Method methodName,
            Object params,
            String jsonPathToError,
            AxisError... errors)
    {
        Object response = jsonClient().invokeMethod(methodName, params);
        responseShouldContainErrors(response, jsonPathToError, errors);
        return (T) response;
    }

    @Step(value = "Вызываем метод {0} и ожидаем получить ошибку в ответе.")
    public <T> T shouldGetMultipleErrorOn(
            String methodName,
            Object params,
            AxisError... errors)
    {
        Object response = jsonClient().invokeMethod(methodName, params);
        responseShouldContainErrors(response, errors);
        return (T) response;
    }

    public <T> T shouldGetMultipleErrorOn(
            Method method,
            Object params,
            AxisError... errors)
    {
        return shouldGetMultipleErrorOn(method.toString(), params, errors);
    }

    @Step("Проверка ответа на наличие ошибок")
    public void responseShouldContainErrors(Object response, AxisError... axisError) {
        assumeThat("в response содержатся ошибки", JsonUtils.toString(response),
                not(ApiResponse.hasNoError()));
        assertThat("полученые ошибки верны", JsonUtils.toString(response), ApiResponse.hasErrors(axisError));
    }

    @Step(value = "Проверим, что метод возвращает ошибку с заданным кодом. {0}")
    public <T extends BeanMap> void shouldGetErrorWithCodeOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, T args,
            Integer expectedErrorCode)
    {
        try {
            defaultClientV5().invokeMethod(serviceName, login, action, args != null ? args.getBean() : null);
            assertThat("получена ошибка", null, equalTo(expectedErrorCode));
        } catch (Api5Error | Api5JsonError error) {
            Integer actualErrorCode = error.getErrorCode();
            assertThat("совпал код", actualErrorCode, equalTo(expectedErrorCode));
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку, не сравнивая details. {0}")
    public <T extends BeanMap> void shouldGetErrorOnIgnoringDetails(
            String assertMessage, ServiceNames serviceName, String login, Action action, T args,
            Api5Error expectedError)
    {
        shouldGetErrorOn(assertMessage, serviceName, login, action, args, expectedError, true);
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public <T extends BeanMap> void shouldGetErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, T args,
            Api5Error expectedError)
    {
        shouldGetErrorOn(assertMessage, serviceName, login, action, args, expectedError, false);
    }

    private <T extends BeanMap> void shouldGetErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, T args,
            Api5Error expectedError,
            boolean ignoreDetails)
    {
        try {
            if (args != null) {
                defaultClientV5().invokeMethod(serviceName, login, action, args.getBean());
            } else {
                defaultClientV5().invokeMethod(serviceName, login, action, null);
            }
            assertThat("получена ошибка", null, equalTo(expectedError));
        } catch (Api5Error soapError) {
            if (expectedError == null) {
                fail("Ожидался успех, а была получена ошибка " + soapError);
            } else {
                if (ignoreDetails) {
                    assertThat("совпал текст ошибки (SOAP)",
                            soapError,
                            Api5ErrorMatcher.equalToIgnoreLocaleAndDetails(expectedError,
                                    properties.getDirectApiIgnoreDetailForCodes()));
                } else {
                    assertThat("совпал текст ошибки (SOAP)",
                            soapError,
                            Api5ErrorMatcher.equalToIgnoreLocale(expectedError,
                                    properties.getDirectApiIgnoreDetailForCodes()));
                }
            }
        } catch (Api5JsonError jsonError) {
            if (expectedError == null) {
                fail("Ожидался успех, а была получена ошибка " + jsonError);
            } else {
                if (ignoreDetails) {
                    assertThat("совпал code и message ошибки (JSON)",
                            jsonError,
                            Api5JsonErrorMatcher.equalToIgnoreLocaleAndDetails(expectedError.toJsonError(),
                                    properties.getDirectApiIgnoreDetailForCodes()));
                } else {
                    assertThat("совпал текст ошибки (JSON)",
                            jsonError,
                            Api5JsonErrorMatcher.equalToIgnoreLocale(expectedError.toJsonError(),
                                    properties.getDirectApiIgnoreDetailForCodes()));
                }
            }
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public <T extends BeanMap> void shouldGetErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, T args,
            JavaOrPerlApi5Error expectedError)
    {
        try {
            MethodInvocationResult invocationResult;
            if (args != null) {
                invocationResult = defaultClientV5().invokeMethodEx(serviceName, login, action, args.getBean());
            } else {
                invocationResult = defaultClientV5().invokeMethodEx(serviceName, login, action, null);
            }
            assertThat("получена ошибка",
                    null,
                    equalTo(invocationResult.isJavaResponse() ? expectedError.getJavaError()
                            : expectedError.getPerlError()));
        } catch (Api5Error soapError) {
            Api5Error expectedApi5Error = soapError.isJavaResponse()
                    ? expectedError.getJavaError()
                    : expectedError.getPerlError();
            if (expectedApi5Error == null) {
                fail("Ожидался успех, а была получена ошибка " + soapError);
            } else {
                assertThat("совпал текст ошибки (SOAP)",
                        soapError,
                        Api5ErrorMatcher.equalToIgnoreLocale(
                                expectedApi5Error,
                                properties.getDirectApiIgnoreDetailForCodes()));
            }
        } catch (Api5JsonError jsonError) {
            Api5Error expectedApi5Error = jsonError.isJavaResponse()
                    ? expectedError.getJavaError()
                    : expectedError.getPerlError();
            if (expectedApi5Error == null) {
                fail("Ожидался успех, а была получена ошибка " + jsonError);
            } else {
                assertThat("совпал текст ошибки (JSON)",
                        jsonError,
                        Api5JsonErrorMatcher.equalToIgnoreLocale(
                                expectedApi5Error.toJsonError(),
                                properties.getDirectApiIgnoreDetailForCodes()));
            }
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public <T extends BeanMap> void shouldGetErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, T args,
            Api5Error expectedError, Api5Error alternativeError)
    {
        try {
            if (args != null) {
                defaultClientV5().invokeMethod(serviceName, login, action, args.getBean());
            } else {
                defaultClientV5().invokeMethod(serviceName, login, action, null);
            }
            assertThat("получена ошибка", null, equalTo(expectedError));
        } catch (Api5Error soapError) {
            if (expectedError == null) {
                fail("Ожидался успех, а была получена ошибка " + soapError);
            } else {
                assertThat("совпал текст ошибки (SOAP)",
                        soapError,
                        either(Api5ErrorMatcher
                                .equalToIgnoreLocale(expectedError, properties.getDirectApiIgnoreDetailForCodes()))
                                .or(Api5ErrorMatcher.equalToIgnoreLocale(alternativeError,
                                        properties.getDirectApiIgnoreDetailForCodes())));
            }
        } catch (Api5JsonError jsonError) {
            if (expectedError == null) {
                fail("Ожидался успех, а была получена ошибка " + jsonError);
            } else {
                assertThat("совпал текст ошибки (JSON)",
                        jsonError,
                        either(Api5JsonErrorMatcher.equalToIgnoreLocale(expectedError.toJsonError(),
                                properties.getDirectApiIgnoreDetailForCodes()))
                                .or(Api5JsonErrorMatcher.equalToIgnoreLocale(alternativeError.toJsonError(),
                                        properties.getDirectApiIgnoreDetailForCodes())));
            }
        }
    }

    public <T extends BeanMap> void shouldGetErrorOn(
            ServiceNames serviceName, String login, Action action, T args, Api5Error expectedError)
    {
        shouldGetErrorOn("", serviceName, login, action, args, expectedError);
    }

    public <T extends BeanMap> void shouldGetErrorOn(
            ServiceNames serviceName, String login, Action action, T args, JavaOrPerlApi5Error expectedError)
    {
        shouldGetErrorOn("", serviceName, login, action, args, expectedError);
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetJSONErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Api5Error expectedError)
    {
        try {
            if (args != null) {
                jsonClientV5().invokeMethod(serviceName, login, action, args);
            } else {
                jsonClientV5().invokeMethod(serviceName, login, action, null);
            }
            assertThat("получена ошибка", null, equalTo(expectedError));
        } catch (Api5JsonError jsonError) {
            assertThat("совпал текст ошибки (JSON)",
                    jsonError,
                    Api5JsonErrorMatcher.equalToIgnoreLocale(
                            expectedError.toJsonError(),
                            properties.getDirectApiIgnoreDetailForCodes()));
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetJSONErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            JavaOrPerlApi5Error expectedError)
    {
        try {
            MethodInvocationResult<Object> invocationResult;
            if (args != null) {
                invocationResult = jsonClientV5().invokeMethodEx(serviceName, login, action, args);
            } else {
                invocationResult = jsonClientV5().invokeMethodEx(serviceName, login, action, null);
            }
            assertThat(
                    "получена ошибка",
                    null,
                    equalTo(invocationResult.isJavaResponse() ? expectedError.getJavaError()
                            : expectedError.getPerlError()));
        } catch (Api5JsonError jsonError) {
            Api5Error expectedApi5Error = jsonError.isJavaResponse()
                    ? expectedError.getJavaError()
                    : expectedError.getPerlError();
            assertThat("совпал текст ошибки (JSON)",
                    jsonError,
                    Api5JsonErrorMatcher.equalToIgnoreLocale(
                            expectedApi5Error.toJsonError(),
                            properties.getDirectApiIgnoreDetailForCodes()));
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetJSONErrorOn(
            String assertMessage, ServiceNames serviceName, String login, Action action, Object args,
            Api5Error expectedError, Api5Error alternativeError)
    {
        try {
            if (args != null) {
                jsonClientV5().invokeMethod(serviceName, login, action, args);
            } else {
                jsonClientV5().invokeMethod(serviceName, login, action, null);
            }
            assertThat("получена ошибка", null, either(equalTo(expectedError)).or(equalTo(alternativeError)));
        } catch (Api5JsonError jsonError) {
            assertThat("совпал текст ошибки (JSON)",
                    jsonError,
                    either(Api5JsonErrorMatcher.equalToIgnoreLocale(expectedError.toJsonError(),
                            properties.getDirectApiIgnoreDetailForCodes()))
                            .or(Api5JsonErrorMatcher.equalToIgnoreLocale(alternativeError.toJsonError(),
                                    properties.getDirectApiIgnoreDetailForCodes())));
        }
    }

    public void shouldGetJSONErrorOn(
            ServiceNames serviceName, String login, Action action, Object args, Api5Error expectedError)
    {
        shouldGetJSONErrorOn("", serviceName, login, action, args, expectedError);
    }

    public void shouldGetJSONErrorOn(
            ServiceNames serviceName, String login, Action action, Object args, JavaOrPerlApi5Error expectedError)
    {
        shouldGetJSONErrorOn("", serviceName, login, action, args, expectedError);
    }

    public void shouldGetJSONErrorOn(
            ServiceNames serviceName, String login, Action action, Object args, Api5Error expectedError,
            Api5Error alternativeError)
    {
        shouldGetJSONErrorOn("", serviceName, login, action, args, expectedError, alternativeError);
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetReportsXmlErrorOn(String login, ReportDefinitionMap args, Api5Error expectedError,
            ReportProcessingMode processingMode, Integer timeout)
    {
        try {
            reportsClientXml().invokeMethod(ServiceNames.REPORTS, login, null, args.getBean(), processingMode, timeout);
            TestSteps.assertThat("получена ошибка", null, equalTo(expectedError));
        } catch (Api5XmlError xmlError) {
            TestSteps.assertThat("совпал текст ошибки (XML)",
                    xmlError,
                    Api5XmlErrorMatcher.equalToIgnoreLocale(
                            expectedError.toXmlError(),
                            properties.getDirectApiIgnoreDetailForCodes()));
        }
    }

    public void shouldGetReportsXmlErrorOn(String login, ReportDefinitionMap args, Api5Error expectedError) {
        shouldGetReportsXmlErrorOn(login, args, expectedError, null, null);
    }


    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetReportsXmlErrorOn(String login, Object args, Api5Error expectedError) {
        try {
            reportsClientXml().invokeMethod(ServiceNames.REPORTS, login, null, args);
            TestSteps.assertThat("получена ошибка", null, equalTo(expectedError));
        } catch (Api5XmlError xmlError) {
            TestSteps.assertThat("совпал текст ошибки (XML)",
                    xmlError,
                    Api5XmlErrorMatcher.equalToIgnoreLocale(
                            expectedError.toXmlError(),
                            properties.getDirectApiIgnoreDetailForCodes()));
        }
    }

    @Step(value = "Проверим, что метод возвращает ошибку. {0}")
    public void shouldGetReportsJsonErrorOn(String login, ReportDefinitionMap args, Api5Error expectedError) {
        try {
            reportsClientJson().invokeMethod(ServiceNames.REPORTS, login, null, args.getBean());
            TestSteps.assertThat("получена ошибка", null, equalTo(expectedError));
        } catch (Api5JsonError jsonError) {
            TestSteps.assertThat("совпал текст ошибки (JSON)",
                    jsonError,
                    Api5JsonErrorMatcher.equalToIgnoreLocale(
                            expectedError.toJsonError(),
                            properties.getDirectApiIgnoreDetailForCodes()));
        }
    }

    @Step("Проверка ответа на наличие ошибок")
    public void responseShouldContainErrors(Object response, String jsonPathToError, AxisError... axisError) {
        assumeThat("в response содержатся ошибки", JsonUtils.toString(response),
                not(ApiResponse.hasNoError(jsonPathToError)));

        assertThat("полученые ошибки верны", JsonUtils.toString(response),
                ApiResponse.hasErrors(jsonPathToError, axisError));
    }

    protected <T, ResultType> ArrayList<ResultType> extractActionResults(T response, Action action) {
        assumeThat("получен ответ сервиса", response, notNullValue());
        String structureName = StringUtils.capitalize(action.toString()) + "Results";
        String methodToCall = "get" + structureName;
        java.lang.reflect.Method resultsGetter;
        try {
            resultsGetter = response.getClass().getMethod(methodToCall);
        } catch (NoSuchMethodException noMethodEx) {
            throw new DirectAPIException("Метод API5 вернул ответ " + response.getClass().getName() + ", " +
                    "в котором отсутствует структура " + structureName, noMethodEx);
        }
        ArrayList<ResultType> results;
        try {
            results = (ArrayList<ResultType>) resultsGetter.invoke(response);
        } catch (IllegalAccessException accessExc) {
            throw new DirectAPIException(
                    "Ошибка доступа при вызове метода " + methodToCall + "в классе " + response.getClass(), accessExc);
        } catch (InvocationTargetException invTargetExc) {
            throw new DirectAPIException("Ошибка reflection при вызове метода " + methodToCall, invTargetExc);
        }
        return results;
    }

    @Step("Проверяем ID/Warnings/Errors в ответе сервиса")
    public Object shouldGetResultOn(
            ServiceNames serviceName, String login, Action action, Object args, ExpectedResult... expectedResults)
    {
        Object response =
                defaultClientV5().invokeMethod(serviceName, login, action, args);
        ArrayList<ActionResult> actualResult = extractActionResults(response, action);
        checkActionResults(actualResult, expectedResults);
        return response;
    }

    public <T extends BeanMap> Object shouldGetResultOn(
            ServiceNames serviceName, String login, Action action, T args, ExpectedResult... expectedResults)
    {
        return shouldGetResultOn(serviceName, login, action, args == null ? null : args.getBean(), expectedResults);
    }

    @Step("Проверяем ID/Warnings/Errors в ответе сервиса")
    public <T extends BeanMap> Object shouldGetResultOn(
            ServiceNames serviceName, String login, Action action, T args,
            List<Matcher<? super ActionResult>> expectedResultMatchers)
    {
        Object response =
                defaultClientV5().invokeMethod(serviceName, login, action, (args == null ? null : args.getBean()));
        List<ActionResult> actualResult = extractActionResults(response, action);

        assertThat("содержание результатов соответствует ожидаемому",
                actualResult, contains(expectedResultMatchers));
        return response;
    }

    public <T extends BeanMap> Object shouldGetResultOn(
            ServiceNames serviceName, String login, Action action, T args, JavaOrPerlExpectedResult... expectedResults)
    {
        @SuppressWarnings("unchecked")
        MethodInvocationResult response =
                defaultClientV5().invokeMethodEx(serviceName, login, action, (args == null ? null : args.getBean()));
        ArrayList<ActionResult> actualResult = extractActionResults(response.getResponseObject(), action);
        checkActionResults(actualResult, response.isJavaResponse(), expectedResults);
        return response.getResponseObject();
    }

    @Step("Проверяем ID/Warnings/Errors в ответе сервиса")
    protected <T> Object shouldGetResultOnJson(
            ServiceNames serviceName, String login, Action action, T args, ExpectedResult... expectedResults)
    {
        Object response = jsonClientV5().invokeMethod(serviceName, login, action, args);
        ArrayList<ActionResult> actualResult = extractActionResults(response, action);
        checkActionResults(actualResult, expectedResults);
        return response;
    }

    protected void checkActionResults(List<ActionResult> actualResults, ExpectedResult... expectedResults) {
        Matcher<Iterable<? extends ActionResult>> resultMatcher =
                getActionResultListMatcher(asList(expectedResults));
        assertThat("содержание результатов соответствует ожидаемому", actualResults, resultMatcher);
    }

    protected static void checkActionResultsMatchEither(List<ActionResult> actualResults,
            List<ExpectedResult> expectedResults, List<ExpectedResult> alternativeExpectedResults)
    {
        Matcher<Iterable<? extends ActionResult>> matcher = getActionResultListMatcher(expectedResults);
        Matcher<Iterable<? extends ActionResult>> alternativeMatcher =
                getActionResultListMatcher(alternativeExpectedResults);
        assertThat("содержание результатов соответствует ожидаемому", actualResults,
                either(matcher).or(alternativeMatcher));
    }

    private static Matcher<Iterable<? extends ActionResult>> getActionResultListMatcher(
            List<ExpectedResult> expectedResults)
    {
        List<Matcher<? super ActionResult>> expectedMatchers = new ArrayList<>();

        for (ExpectedResult expResult : expectedResults) {
            expectedMatchers.add(BeanEquals.beanEquals((ActionResult) expResult.getBean())
                    .accordingStrategy(expResult.getActionResultCompareStrategy())
            );
        }

        return contains(expectedMatchers);
    }

    protected void checkActionResults(
            List<ActionResult> actualResults, boolean isJavaResponse, JavaOrPerlExpectedResult[] expectedResults)
    {
        List<Matcher<? super ActionResult>> expectedMatchers = new ArrayList<>();

        for (JavaOrPerlExpectedResult javaOrPerlExpectedResult : expectedResults) {
            ExpectedResult expectedResult = isJavaResponse
                    ? javaOrPerlExpectedResult.getJavaResult()
                    : javaOrPerlExpectedResult.getPerlResult();
            expectedMatchers.add(BeanEquals.beanEquals((ActionResult) expectedResult.getBean())
                    .accordingStrategy(expectedResult.getActionResultCompareStrategy()));
        }

        Matcher<Iterable<? extends ActionResult>> resultMatcher = contains(expectedMatchers);
        assertThat("содержание результатов соответствует ожидаемому", actualResults, resultMatcher);
    }

    private void checkBidActionResults(
            String message, List<BidActionResult> actualResults, BidExpectedResult... expectedResults)
    {
        List<Matcher<? super BidActionResult>> expectedMatchers = new ArrayList<>();

        for (BidExpectedResult expResult : expectedResults) {
            expectedMatchers.add(BeanEquals.beanEquals((BidActionResult) expResult.getBean())
                    .accordingStrategy(expResult.getActionResultCompareStrategy())
            );
        }

        Matcher<Iterable<? extends BidActionResult>> resultMatcher = contains(expectedMatchers);
        assertThat(message, actualResults, resultMatcher);
    }

    @Step("Проверяем ID/Warnings/Errors в ответе сервиса Bid")
    public <T extends BeanMap> void shouldGetBidResultOn(
            String assertMessage, String login, ServiceNames serviceName, Action action, T args,
            BidExpectedResult... expectedResults)
    {
        Object response = defaultClientV5().invokeMethod(serviceName, login, action, args.getBean());
        ArrayList<BidActionResult> actualResult = extractActionResults(response, action);
        checkBidActionResults(assertMessage, actualResult, expectedResults);
    }

    @Step("Проверяем ID/Warnings/Errors в ответе сервиса")
    protected <T> Object shouldGetBidResultOnJson(
            ServiceNames serviceName, String login, Action action, T args, BidExpectedResult... expectedResults)
    {
        Object response = jsonClientV5().invokeMethod(serviceName, login, action, args);
        ArrayList<BidActionResult> actualResult = extractActionResults(response, action);
        checkBidActionResults("", actualResult, expectedResults);
        return response;
    }

    public <T extends BeanMap> void shouldGetBidResultOn(
            String login, ServiceNames serviceName, Action action, T args, BidExpectedResult... expectedResults)
    {
        shouldGetBidResultOn("", login, serviceName, action, args, expectedResults);
    }

    private void checkSetBidsActionResults(
            String message, List<SetBidsActionResult> actualResults, SetBidsExpectedResult... expectedResults)
    {
        List<Matcher<? super SetBidsActionResult>> expectedMatchers = new ArrayList<>();

        for (SetBidsExpectedResult expResult : expectedResults) {
            expectedMatchers.add(BeanEquals.beanEquals((SetBidsActionResult) expResult.getBean())
                    .accordingStrategy(expResult.getActionResultCompareStrategy())
            );
        }

        Matcher<Iterable<? extends SetBidsActionResult>> resultMatcher = contains(expectedMatchers);
        assertThat(message, actualResults, resultMatcher);
    }

    protected void checkSetBidsActionResults(
            List<SetBidsActionResult> actualResults, boolean isJavaResponse,
            JavaOrPerlSetBidsExpectedResult[] expectedResults)
    {

        List<Matcher<? super SetBidsActionResult>> expectedMatchers = new ArrayList<>();

        for (JavaOrPerlSetBidsExpectedResult javaOrPerlExpectedResult : expectedResults) {
            SetBidsExpectedResult expResult = isJavaResponse
                    ? javaOrPerlExpectedResult.getJavaResult()
                    : javaOrPerlExpectedResult.getPerlResult();

            expectedMatchers.add(BeanEquals.beanEquals((SetBidsActionResult) expResult.getBean())
                    .accordingStrategy(expResult.getActionResultCompareStrategy())
            );
        }

        Matcher<Iterable<? extends SetBidsActionResult>> resultMatcher = contains(expectedMatchers);
        assertThat("содержание результатов соответствует ожидаемому", actualResults, resultMatcher);
    }

    @Step("Проверяем ID/Warnings/Errors в ответе setBids")
    public <T extends BeanMap> Object shouldGetSetBidsResultOn(
            String assertMessage, String login, ServiceNames serviceName, Action action, T args,
            SetBidsExpectedResult... expectedResults)
    {
        Object response = defaultClientV5().invokeMethod(serviceName, login, action, args.getBean());
        ArrayList<SetBidsActionResult> actualResult = extractActionResults(response, action);
        checkSetBidsActionResults(assertMessage, actualResult, expectedResults);
        return response;
    }

    @Step("Проверяем ID/Warnings/Errors в ответе setBids")
    public <T extends BeanMap> Object shouldGetSetBidsResultOn(String login, ServiceNames serviceName, Action action,
            T args, JavaOrPerlSetBidsExpectedResult... expectedResults)
    {
        MethodInvocationResult response =
                defaultClientV5().invokeMethodEx(serviceName, login, action, args.getBean());
        ArrayList<SetBidsActionResult> actualResult = extractActionResults(response.getResponseObject(), action);
        checkSetBidsActionResults(actualResult, response.isJavaResponse(), expectedResults);
        return response;
    }

    @Step("Проверяем ID/Warnings/Errors в ответе сервиса")
    protected <T> Object shouldGetSetBidsResultOnJson(
            ServiceNames serviceName, String login, Action action, T args, SetBidsExpectedResult... expectedResults)
    {
        Object response = jsonClientV5().invokeMethod(serviceName, login, action, args);
        ArrayList<SetBidsActionResult> actualResult = extractActionResults(response, action);
        checkSetBidsActionResults("", actualResult, expectedResults);
        return response;
    }

    public <T extends BeanMap> void shouldGetSetBidsResultOn(
            String login, ServiceNames serviceName, Action action, T args, SetBidsExpectedResult... expectedResults)
    {
        shouldGetSetBidsResultOn("", login, serviceName, action, args, expectedResults);
    }

    private void checkAdImageActionResults(List<AdImageActionResult> actualResults,
            AdImageExpectedResult... expectedResults)
    {
        List<Matcher<? super AdImageActionResult>> expectedMatchers = new ArrayList<>();

        for (AdImageExpectedResult expResult : expectedResults) {
            expectedMatchers.add(BeanEquals.beanEquals((AdImageActionResult) expResult.getBean())
                    .accordingStrategy(expResult.getActionResultCompareStrategy())
            );
        }

        Matcher<Iterable<? extends AdImageActionResult>> resultMatcher = contains(expectedMatchers);
        assertThat("не получен ожидаемый ответ", actualResults, resultMatcher);
    }

    @Step("Проверяем AdImageHash/Warnings/Errors в ответе сервиса AdImages")
    public <T extends BeanMap> Object shouldGetAdImageResultOn(
            String assertMessage, String login, ServiceNames serviceName, Action action, T args,
            AdImageExpectedResult... expectedResults)
    {
        Object response = defaultClientV5().invokeMethod(serviceName, login, action, args.getBean());
        ArrayList<AdImageActionResult> actualResult = extractActionResults(response, action);
        checkAdImageActionResults(actualResult, expectedResults);
        return response;
    }

    @Step("Проверяем AdImageHash/Warnings/Errors в ответе сервиса")
    protected <T> Object shouldGetAdImageResultOnJson(
            ServiceNames serviceName, String login, Action action, T args, AdImageExpectedResult... expectedResults)
    {
        Object response = jsonClientV5().invokeMethod(serviceName, login, action, args);
        ArrayList<AdImageActionResult> actualResult = extractActionResults(response, action);
        checkAdImageActionResults(actualResult, expectedResults);
        return response;
    }

    public <T extends BeanMap> Object shouldGetAdImageResultOn(
            String login, ServiceNames serviceName, Action action, T args, AdImageExpectedResult... expectedResults)
    {
        return shouldGetAdImageResultOn("", login, serviceName, action, args, expectedResults);
    }

    public void checkResponseErrors(List<ActionResult> results) {
        List<ActionResult> errors = select(results, having(on(ActionResult.class).getErrors(), is(not(empty()))));
        MatcherAssert.assertThat("Есть ошибки при вызове метода", errors, is(empty()));
    }
}
