package ru.yandex.autotests.directapi.steps.banners;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Callable;

import ch.lambdaj.function.convert.PropertyExtractor;
import org.apache.commons.lang.ArrayUtils;
import org.hamcrest.Matcher;
import org.hamcrest.collection.IsArrayWithSize;

import ru.yandex.autotests.direct.utils.ReflectionUtils;
import ru.yandex.autotests.direct.utils.beans.BeanWrapper;
import ru.yandex.autotests.direct.utils.converter.BeanMapToBeanConverter;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.matchers.banners.BannerMatchers;
import ru.yandex.autotests.directapi.model.banners.Action;
import ru.yandex.autotests.directapi.model.banners.BannerInfoMap;
import ru.yandex.autotests.directapi.model.banners.CampaignBidsInfoMap;
import ru.yandex.autotests.directapi.model.banners.GetBannersInfoMap;
import ru.yandex.autotests.directapi.model.banners.KeywordRequestMap;
import ru.yandex.autotests.directapi.model.banners.phrases.BannerPhrasesFilterRequestInfoMap;
import ru.yandex.autotests.directapi.steps.BaseApiSteps;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;

import static ch.lambdaj.Lambda.convert;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNull.notNullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created with IntelliJ IDEA.
 * User: mariabye
 * Date: 24.06.13
 * Time: 18:29
 * To change this template use File | Settings | File Templates.
 */
public class BannerSteps extends BaseApiSteps {
    private LogSteps log = LogSteps.getLogger(this.getClass());

    public static int BANNER_COUNT_LIMIT = 1000;
    public static final int PHRASE_LENGTH_LIMIT = 4096;
    public static final Long BANNER_OLD_WITH_RUBRIC_ID = 4963L;
    public static final int MAX_TITLE_LENGTH = 33;
    public static final int MAX_BANNERS_COUNT_IN_GROUP = 50;
    public static final int AD_GROUP_NAME_LENGTH = 255;
    public static final int KEYWORD_IDS_LIMIT = 10000;

    private static BannerSteps _instance;

    private BannerSteps(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        super(connectionConfig, requestHeader);
    }

    public static BannerSteps getInstance(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        if (_instance == null) {
            _instance = new BannerSteps(connectionConfig, requestHeader);
        } else {
            _instance.setConnectionConfig(connectionConfig);
            _instance.setRequestHeader(requestHeader);
        }
        return _instance;

    }


    //region CreateOrUpdateBanners
    @Step(value = "[CreateOrUpdateBanners]")
    public Long[] createOrUpdateBanners(BannerInfoMap... bannerInfoModels) {
        List<Object> bannerInfos = convert(bannerInfoModels, new BeanMapToBeanConverter());
        long[] bannerIDs = defaultClient().invokeMethod(Method.CREATE_OR_UPDATE_BANNERS, bannerInfos.toArray());
        log.info("Объявления: bannerIDs = " + JsonUtils.toString(bannerIDs));
        return ArrayUtils.toObject(bannerIDs);
    }

    @Step(value = "Создать новое объявление")
    public Long createBanner(BannerInfoMap bannerInfoModel) {
        Long[] bannerIDS = createOrUpdateBanners(bannerInfoModel.withBannerID(0L));
        assumeThat("вернулось одно объявление", bannerIDS.length, equalTo(1));
        return bannerIDS[0];
    }

    @Step(value = "Создать новое объявление {0}")
    public Long createBanner(String details, BannerInfoMap bannerInfoModel) {
        Long[] bannerIDS = createOrUpdateBanners(bannerInfoModel.withBannerID(0L));
        assumeThat("вернулось одно объявление", bannerIDS.length, equalTo(1));
        return bannerIDS[0];
    }

    public void expectErrorOnBannerCreateOrUpdate(BannerInfoMap bannerInfoModel, AxisError axisError) {
        expectErrorOnBannerCreateOrUpdate(new BannerInfoMap[]{bannerInfoModel}, axisError);
    }

    public void expectErrorOnBannerCreateOrUpdate(BannerInfoMap[] bannerInfoModels, AxisError axisError) {
        shouldGetErrorOn(
                Method.CREATE_OR_UPDATE_BANNERS,
                bannerInfoModels,
                axisError
        );
    }

    @Step(value = "Сохранить объявление")
    public Long updateBanner(BannerInfoMap bannerInfoModel) {
        return createOrUpdateBanners(bannerInfoModel)[0];
    }

    @Step(value = "Сохранить объявление {0}")
    public Long updateBanner(String details, BannerInfoMap bannerInfoModel) {
        return createOrUpdateBanners(bannerInfoModel)[0];
    }

    public Long createDefaultBanner(int campaignID) {
        return createBanner(
                new BannerInfoMap(defaultClient().getPackageName())
                        .defaultBanner()
                        .withCampaignID(campaignID)
        );
    }

    public Long createDefaultBanner(Long campaignID) {
        return createDefaultBanner(campaignID.intValue());
    }

    public Long createDefaultBannerWithNoLinks(int campaignID) {
        return createBanner(
                new BannerInfoMap(defaultClient().getPackageName())
                        .defaultBannerWithNoLinks()
                        .withCampaignID(campaignID));
    }

    /**
     * Создает bannerCount объявлений в кампании
     *
     * @param campaignId  идентификатор кампании
     * @param bannerCount количество объявлений, создаваемое в каждой кампании
     * @return массив идентификаторов созданных объявлений
     * @throws DirectAPIException ошибка создания объявления
     */
    public Long[] createManyBanners(int campaignId, int bannerCount) {
        List<BannerInfoMap> bannerInfoModels = new ArrayList<>();
        for (int i = 0; i < bannerCount; i++) {
            bannerInfoModels.add(
                    new BannerInfoMap(defaultClient().getPackageName())
                            .defaultBanner()
                            .withCampaignID(campaignId)
            );
        }
        return createOrUpdateBanners(bannerInfoModels.toArray(new BannerInfoMap[0]));
    }

    @Step("Создать группу баннеров по умолчанию")
    public Long[] createDefaultGroup(int campaignID) {
        return createGroupWithManyBanners(campaignID, 2);
    }

    public Long[] createGroupWithManyBanners(int campaignID, int bannerCount) {
        BannerInfoMap[] banners = new BannerInfoMap[bannerCount];
        Arrays.fill(banners,
                new BannerInfoMap(defaultClient().getPackageName())
                        .defaultBanner()
                        .withCampaignID(campaignID)
                        .withBannerID(0L)
                        .withContacts(null)
                        .withAdGroupID(null));
        return createGroupWithBanners(banners);
    }

    public <T> Long[] createGroupWithBanners(BannerInfoMap[] banners) {
        int bannerCount = banners.length;
        Long[] bannerIDS = new Long[bannerCount];
        Long bannerID1 = createBanner(banners[0]);
        bannerIDS[0] = bannerID1;
        Long adGroupID = (Long) new BannerInfoMap((T) getBanner(bannerID1)).get(BannerInfoMap.AD_GROUP_ID);
        for (int i = 1; i < bannerCount; i++) {
            bannerIDS[i] = createBanner(banners[i].withAdGroupID(adGroupID));
        }
        return bannerIDS;
    }

    public Long createDefaultBannerWithOneTextPhrase(int campaignID) {
        return createDefaultBannerWithTextPhrases(campaignID, 1);
    }

    public Long createDefaultBannerWithTextPhrases(int campaignID, int phraseCount) {
        return createBanner(
                new BannerInfoMap(defaultClient().getPackageName())
                        .defaultBanner()
                        .withCampaignID(campaignID)
                        .withDefaultPhrases(phraseCount));

    }
    //endregion

    //region GetBanners

    /**
     * Запрос списка объявлений по параметрам
     *
     * @param getBannersInfo - модель для GetBannersInfo
     * @param <T>            - класс BannerInfo из нужного пакета
     * @return
     */
    @Step(value = "[GetBanners]")
    public <T> T[] getBanners(GetBannersInfoMap getBannersInfo) {
        //log.info("Получить список объявлений");
        //log.info(BeanWrapper.wrap(getBannersInfo.getBean()));
        return (T[]) defaultClient().invokeMethod(Method.GET_BANNERS, getBannersInfo.getBean());
    }

    @Step(value = "Получить список объявлений: bannerIDs = {0}")
    public <T> T[] getBanners(Long[] bannerIDs) {
        return (T[]) getBanners(
                new GetBannersInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerIDs));
    }


    @Step(value = "Получить список объявлений: campaignID = {0}")
    public <T> T[] getBanners(int campaignID) {
        return (T[]) getBanners(
                new GetBannersInfoMap(defaultClient().getPackageName())
                        .withCampaignIDS(campaignID));
    }

    @Step(value = "Получить объявление: bannerID = {0}")
    public <T> T getBanner(Long bannerID) {
        Object[] banners = getBanners(
                new GetBannersInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerID));
        assumeThat("получено одно объявление", banners, allOf(notNullValue(), IsArrayWithSize.arrayWithSize(1)));
        return (T) banners[0];
    }

    @Step(value = "Получить объявление: bannerID = {0}")
    public <T> T getBanner(Long bannerID, Currency currency) {
        Object[] banners = getBanners(
                new GetBannersInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerID).withCurrency(currency));
        assumeThat("получено одно объявление", banners, allOf(notNullValue(), IsArrayWithSize.arrayWithSize(1)));
        return (T) banners[0];
    }

    @Step(value = "Получить объявление: bannerID = {0},  withGetPhrases = {1}")
    public <T> T getBanner(Long bannerID, String phraseFilter) {
        Object[] banners = getBanners(
                new GetBannersInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerID)
                        .withGetPhrases(phraseFilter));
        assumeThat("получено одно объявление", banners, allOf(notNullValue(), IsArrayWithSize.arrayWithSize(1)));
        return (T) banners[0];
    }

    @Step(value = "Получить adGroupID по bannerID = {0}")
    public Long getAdGroupIDByBannerID(Long bannerID) {
        Object banner = getBanner(bannerID);
        return (Long) new BannerInfoMap<>(banner).get(BannerInfoMap.AD_GROUP_ID);
    }

    /*@Step(value = "Выполнение проверки для баннера: bannerID:{0}, match {1}")
    public <T> void bannerShould(Long bannerID, Matcher<BeanWrapper<T>> match) {
        T banner = getBanner(bannerID);
        assertThat(BeanWrapper.wrap(banner), match);
    }*/

    @Step(value = "Выполнение проверки для баннера: bannerID:{0} должен иметь авто-имя для группы")
    public <T> void bannerShouldHasAutoGeneratedAdGroupName(Long bannerID) {
        T banner = getBanner(bannerID);
        Matcher<BeanWrapper<T>> matcher = BannerMatchers.has(BannerInfoMap.AD_GROUP_NAME, equalTo(
                String.format(BannerInfoMap.AD_GROUP_NAME_TEMPLATE,
                        new BannerInfoMap(banner).get(BannerInfoMap.AD_GROUP_ID))
        ));
        assertThat("у объявления указано имя его группы, назначенное по умолчанию", BeanWrapper.wrap(banner), matcher);
    }
    //endregion


    //region GetBannerPhrases(Filter)
    @Step("[GetBannerPhrases]: bannerIDS = {0}")
    public <T> T[] getBannerPhrases(Long... bannerIDS) {
        return (T[]) defaultClient().invokeMethod(Method.GET_BANNER_PHRASES, bannerIDS);
    }

    @Step("[GetBannerPhrasesFilter]")
    public <T> T[] getBannerPhrasesFilter(BannerPhrasesFilterRequestInfoMap bannerPhrasesFilterRequestInfo) {
        return (T[]) defaultClient().invokeMethod(Method.GET_BANNER_PHRASES_FILTER,
                bannerPhrasesFilterRequestInfo.getBean());
    }

    @Step("Получить список фраз объявлений с учетом фильтра: bannerIDs = {0}")
    public <T> T[] getBannerPhrasesFilter(Long[] bannerIDs) {
        return (T[]) getBannerPhrasesFilter(
                new BannerPhrasesFilterRequestInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerIDs));
    }

    @Step("Получить список фраз объявления с учетом фильтра: bannerIDs = {0}")
    public <T> T[] getBannerPhrasesFilter(Long bannerID) {
        return (T[]) getBannerPhrasesFilter(
                new BannerPhrasesFilterRequestInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerID));
    }

    @Step("Получить фразу объявления: bannerIDs = {0}")
    public <T> T getBannerPhrase(Long bannerID) {
        T[] phrases = defaultClient().invokeMethod(Method.GET_BANNER_PHRASES, new Long[]{bannerID});
        assumeThat("получена одна фраза", phrases, allOf(notNullValue(), IsArrayWithSize.arrayWithSize(1)));
        return phrases[0];
    }

    //endregion
    //region Archive/UnArchive/Stop/Resume/Delete/Moderate
    @Step("[DeleteBanners]")
    public void deleteBanners(CampaignBidsInfoMap campaignBidsInfo) {
        defaultClient().invokeMethod(Method.DELETE_BANNERS, campaignBidsInfo.getBean());
        //log.info("Объявления удалены");
    }

    @Step("Удалить объявление: bannerID = {0}")
    public void deleteBanner(Long bannerID) {
        deleteBanners(new CampaignBidsInfoMap(defaultClient().getPackageName()).withBannerIDS(bannerID));
    }

    @Step("Удалить объявления: campaignID = {0}, bannerIDs = {1}")
    public void deleteBanners(int campaignID, Long... bannerIDs) {
        deleteBanners(
                new CampaignBidsInfoMap(defaultClient().getPackageName())
                        .withCampaignID(campaignID)
                        .withBannerIDS(bannerIDs));
    }

    @Step("[ModerateBanners]")
    public int moderateBanners(CampaignBidsInfoMap campaignBidsInfo) {
        return defaultClient().invokeMethod(Method.MODERATE_BANNERS, campaignBidsInfo.getBean());
    }

    @Step("Отправить на модерацию объявление: bannerID = {0}")
    public int moderateBanner(Long bannerID) {
        return moderateBanners(
                new CampaignBidsInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerID));
    }

    @Step("Отправить на модерацию объявления")
    public int moderateBanners(Long... bannerID) {
        return moderateBanners(
                new CampaignBidsInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerID));
    }

    @Step("[StopBanners]")
    public void stopBanners(CampaignBidsInfoMap campaignBidsInfo) {
        //log.info("Остановка объявлений");
        defaultClient().invokeMethod(Method.STOP_BANNERS, campaignBidsInfo.getBean());
        //log.info("Объявления остановлены");
    }

    @Step("Остановить объявление: bannerID = {0}")
    public void stopBanner(Long bannerID) {
        stopBanners(
                new CampaignBidsInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerID));
    }

    @Step("Остановить объявления: campaignID = {0}, bannerIDs = {1}")
    public void stopBanners(int campaignID, Long... bannerIDs) {
        stopBanners(new CampaignBidsInfoMap(defaultClient().getPackageName())
                .withCampaignID(campaignID)
                .withBannerIDS(bannerIDs));
    }

    @Step("[ResumeBanners]")
    public void resumeBanners(CampaignBidsInfoMap campaignBidsInfo) {
        defaultClient().invokeMethod(Method.RESUME_BANNERS, campaignBidsInfo.getBean());
    }

    @Step("Запустить объявление: bannerID = {0}")
    public void resumeBanner(Long bannerID) {
        resumeBanners(new CampaignBidsInfoMap(defaultClient().getPackageName()).withBannerIDS(bannerID));
    }

    @Step("[ArchiveBanners]")
    public void archiveBanners(CampaignBidsInfoMap campaignBidsInfo) {
        //log.info("Архивирование объявлений");
        defaultClient().invokeMethod(Method.ARCHIVE_BANNERS, campaignBidsInfo.getBean());
        //log.info("Объявления заархивированы");
    }

    @Step("[ArchiveBanners]")
    public int archiveBannersWithResult(CampaignBidsInfoMap campaignBidsInfo) {
        //log.info("Архивирование объявлений");
        int result = defaultClient().invokeMethod(Method.ARCHIVE_BANNERS, campaignBidsInfo.getBean());
        //log.info("Объявления заархивированы");
        return result;
    }

    @Step("Заархивировать объявления: campaignID = {0}, bannerIDs = {1}")
    public void archiveBanners(int campaignID, Long... bannerIDs) {
        archiveBanners(
                new CampaignBidsInfoMap(defaultClient().getPackageName())
                        .withCampaignID(campaignID)
                        .withBannerIDS(bannerIDs));
    }

    @Step("Заархивировать объявления: bannerIDs = {0}")
    public void archiveBanners(Long[] bannerIDs) {
        archiveBanners(
                new CampaignBidsInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerIDs));
    }

    @Step("Заархивировать объявления: bannerIDs = {0}")
    public int archiveBannersWithResult(Long[] bannerIDs) {
        int result = archiveBannersWithResult(
                new CampaignBidsInfoMap(defaultClient().getPackageName())
                        .withBannerIDS(bannerIDs));
        return result;
    }

    @Step("Заархивировать объявление: bannerID = {0}")
    public void archiveBanner(Long bannerID) {
        archiveBanners(new CampaignBidsInfoMap(defaultClient().getPackageName()).withBannerIDS(bannerID));
    }

    @Step("[UnArchiveBanners]")
    public void unArchiveBanners(CampaignBidsInfoMap campaignBidsInfo) {
        defaultClient().invokeMethod(Method.UNARCHIVE_BANNERS, campaignBidsInfo.getBean());
    }

    @Step("Разархивировать объявление: bannerID = {0}")
    public void unArchiveBanner(Long bannerID) {
        unArchiveBanners(new CampaignBidsInfoMap(defaultClient().getPackageName()).withBannerIDS(bannerID));
    }

    public void shouldGetErrorOnModerateBanners(AxisError axisError, CampaignBidsInfoMap campaignBidsInfoMap) {
        shouldGetErrorOn(Method.MODERATE_BANNERS, campaignBidsInfoMap, axisError);
    }
    //endregion

    @Step("Запрос списка рубрик")
    public <T> T[] getRubrics() {
        //log.info("Запрос списка рубрик");
        return (T[]) defaultClient().invokeMethod(Method.GET_RUBRICS, null);
    }

    public <T> Callable<Boolean> bannerDomainIs(final Long bannerID, final String domain) {
        return new Callable<Boolean>() {
            public Boolean call() throws Exception {
                return domain.equals(
                        new BannerInfoMap(getBanner(bannerID)).get(BannerInfoMap.DOMAIN));
            }
        };
    }

    public Callable<Boolean> statusChanged(final Long bannerId, final String statusField, final String value) {
        return new Callable<Boolean>() {
            public Boolean call() throws Exception {
                return ReflectionUtils.invokeGetter(getBanner(bannerId), capitalize(statusField)).equals(value);
            }
        };
    }

    public Callable<Boolean> phrasePriceChanged(final Long bannerId, final int phrasePosition,
            final float expectedPrice)
    {
        return new Callable<Boolean>() {
            public Boolean call() throws Exception {
                return ReflectionUtils.invokeGetter(getBannerPhrases(bannerId)[phrasePosition], "Price")
                        .equals(expectedPrice);
            }
        };
    }

    public Long[] getBannerIDs(int campaignID) {
        Object[] banners = getBanners(campaignID);
        List<Long> bannerIDs = convert(banners, new PropertyExtractor<Object, Long>(BannerInfoMap.BANNER_ID));
        return bannerIDs.toArray(new Long[0]);
    }

    //region Keyword
    @Step("[Keyword]")
    public <T> T keyword(KeywordRequestMap keywordRequest) {
        return (T) defaultClient().invokeMethod(Method.KEYWORD, keywordRequest.getBean());
    }

    @Step("Запустить фразы: login = {0}, keywordIDS = {1}")
    public <T> T resumeKeywords(String login, long... keywordIDS) {
        return (T) keyword(new KeywordRequestMap(defaultClient().getPackageName())
                .withAction(Action.RESUME)
                .withLogin(login)
                .withKeywordIDS(keywordIDS));
    }

    @Step("Остановить фразы: login = {0}, keywordIDS = {1}")
    public <T> T suspendKeywords(String login, long... keywordIDS) {
        return (T) keyword(new KeywordRequestMap(defaultClient().getPackageName())
                .withAction(Action.SUSPEND)
                .withLogin(login)
                .withKeywordIDS(keywordIDS));
    }

    @Step("Получить фразы: login = {0}, keywordsIDS = {1}")
    public <T> T getKeywords(String login, long... keywordIDS) {
        return (T) keyword(new KeywordRequestMap(defaultClient().getPackageName())
                .withAction(Action.GET)
                .withLogin(login)
                .withKeywordIDS(keywordIDS));
    }
    //endregion
}
