package ru.yandex.autotests.directapi.steps.banners;

import java.net.URI;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

import org.apache.commons.beanutils.BeanMap;
import org.apache.commons.collections.keyvalue.DefaultKeyValue;
import org.apache.commons.lang.ArrayUtils;
import org.apache.http.client.utils.URIBuilder;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.hamcrest.Matchers;

import ru.yandex.autotests.direct.utils.converter.BeanToBeanMapConverter;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.methods.Action;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.enums.ImageType;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.matchers.axiserror.ApiResponse;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.model.images.AdImageAssociationMap;
import ru.yandex.autotests.directapi.model.images.AdImageAssociationRequestMap;
import ru.yandex.autotests.directapi.model.images.AdImageAssociationSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.images.AdImageMap;
import ru.yandex.autotests.directapi.model.images.AdImageRawMap;
import ru.yandex.autotests.directapi.model.images.AdImageRequestMap;
import ru.yandex.autotests.directapi.model.images.AdImageSelectionCriteriaMap;
import ru.yandex.autotests.directapi.model.images.AdImageURLMap;
import ru.yandex.autotests.directapi.model.images.AdImageUploadMap;
import ru.yandex.autotests.directapi.model.images.Status;
import ru.yandex.autotests.directapi.rules.Bin;
import ru.yandex.autotests.directapi.rules.Binable;
import ru.yandex.autotests.directapi.steps.BaseApiSteps;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.directapi.utils.ImageBase64Utils;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;

import static ch.lambdaj.Lambda.convert;
import static ch.lambdaj.Lambda.extractProperty;
import static ch.lambdaj.Lambda.forEach;
import static org.hamcrest.Matchers.lessThan;
import static org.hamcrest.core.IsEqual.equalTo;

/**
 * Инструкция по заливке картинок в MDS-S3:
 * https://wiki.yandex-team.ru/direct/development/howto/test-data-in-mds-s3/#poshagovajainstrukcijapopoluchenijudostupaizalivkefajjlavxranilishhe
 */
public class ImagesSteps extends BaseApiSteps implements Binable<DefaultKeyValue> {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static Logger log4j = LogManager.getLogger(ImagesSteps.class);

    private static ImagesSteps _instance;

    public Bin<DefaultKeyValue> bin = new Bin<>(this);

    private ImagesSteps(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        super(connectionConfig, requestHeader);
    }

    public static ImagesSteps getInstance(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        if (_instance == null) {
            _instance = new ImagesSteps(connectionConfig, requestHeader);
        } else {
            _instance.setConnectionConfig(connectionConfig);
            _instance.setRequestHeader(requestHeader);
        }
        return _instance;
    }

    public static Map<Integer, String> nameInvalidSmallImages = new HashMap<Integer, String>() {{
        put(1, "jpg 149x150");
        put(2, "jpg 150x149");
        put(3, "jpg 149x149");
        put(4, "jpg 150x201");
        put(5, "jpg 201x150");
        put(6, "bmp 150x150");
        put(7, "jpg 200x150");
        put(8, "gif 180x150");
        put(9, "png 150x150");
        put(10, "jpg 150x150");
        put(11, "jpg 150x200");
    }};


    public static Map<Integer, String> urlInvalidSmallImages = new HashMap<Integer, String>() {{
        put(1,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/old/invalidimg1.gif");
        put(2,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/old/invalidimg2.jpg");
        put(3,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/old/invalidimg3.jpg");
        put(4,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/old/invalidimg4.jpg");
        put(5,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/old/invalidimg5.jpg");
        put(6,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/small150x150.bmp");
        put(7,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/small200x150.jpg");
        put(8,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/small180x150.gif");
        put(9,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/small150x150.png");
        put(10,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/small150x150.jpg");
        put(11,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/small150x200.jpg");
    }};


    //BigImage

    public static Map<Integer, String> nameValidRegularImages = new HashMap<Integer, String>() {{
        put(1, "jpg 450X600");
        put(2, "jpg 600X450");
        put(3, "jpg 450X450");
        put(4, "jpg 1200x900");
        put(5, "450x600 jpg");
        put(6, "jpg 600x800");
        put(7, "png");
        put(8, "jpg");
        put(9, "jpg 3750X5000");
        put(10, "jpg 5000X3750");
        put(11, "png 5000X5000");
        put(12, "gif 640X640");
    }};

    public static Map<Integer, String> validRegularImages = new HashMap<Integer, String>() {{
        put(1,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_450x600.jpg");
        put(2,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_600x450.jpg");
        put(3,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_450x450.jpg");
        put(4,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_1200x900.jpg");
        put(5,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regularjpg.jpg");
        put(6,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular600x800.jpg");
        put(7,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_png.png");
        put(8,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_jpg.jpg");
        put(9,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_3750x5000.jpg");
        put(10,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_5000x3750.jpg");
        put(11,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_5000x5000.png");
        put(12,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/640x640.gif");
    }};

    public static Map<Integer, String> nameValidWideImages = new HashMap<Integer, String>() {{
        put(1, "jpg 1080x607");
        put(2, "jpg 1080x608");
        put(3, "jpg 5000X2813");
    }};

    public static Map<Integer, String> validWideImages = new HashMap<Integer, String>() {{
        put(1,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/wide_1080x607.jpg");
        put(2,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/wide_1080x608.jpg");
        put(3,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/wide_5000x2813.jpg");
    }};

    public static Map<Integer, String> nameInvalidBigImages = new HashMap<Integer, String>() {{
        put(1, "jpg 1079X607");
        put(2, "jpg 2814X5001");
        put(3, "jpg 5001X2814");
        put(4, "jpg 5001X5001");
        put(5, "jpg 6000X6000");
        put(6, "bmp 500X500");
    }};

    public static Map<Integer, String> invalidBigImages = new HashMap<Integer, String>() {{
        put(1,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/undefined_1079x607.jpg");
        put(2,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/undefined_2814x5001.jpg");
        put(3,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/undefined_5001x2814.jpg");
        put(4,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/undefined_5001x5001.jpg");
        put(5,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/undefined_6000x6000.jpg");
        put(6,
                "https://direct-qa-pub.s3.yandex.net/fcf9ad19f7/images/regular_bmp.bmp");
    }};

    public Callable clearBin(final Set<DefaultKeyValue> binData) {
        return new Callable() {
            public Object call() throws Exception {
                if (binData != null) {
                    Object[] entries = binData.toArray();
                    for (Object entry : entries) {
                        try {
                            DefaultKeyValue image = (DefaultKeyValue) entry;
                            Object adImageAssociationResponse = getAdImageAssociation(
                                    new AdImageAssociationSelectionCriteriaMap(defaultClient().getPackageName())
                                            .withLogins((String) image.getKey())
                                            .withAdImageHashes((String) image.getValue())
                            );
                            Object[] adImageAssociations = (Object[]) new BeanMap(adImageAssociationResponse)
                                    .get(AdImageAssociationRequestMap.AD_IMAGE_ASSOCIATIONS);

                            List<AdImageAssociationMap> associationMaps =
                                    convert(adImageAssociations,
                                            new BeanToBeanMapConverter<>
                                                    (AdImageAssociationMap.class));
                            if (associationMaps != null && associationMaps.size() > 0) {
                                forEach(associationMaps).put(AdImageAssociationMap.AD_IMAGE_HASH, null);
                            }

                            setAdImageAssociation(associationMaps.toArray(new AdImageAssociationMap[0]));
                            adImageDelete(
                                    new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                                            .withLogins((String) image.getKey())
                                            .withAdImageHashes((String) image.getValue()));
                            binData.remove(entry);
                        } catch (AxisError axisError) {
                            log4j.error("TRASHER: ошибка удаления картинки. ", axisError);
                        }
                    }
                }
                return null;
            }
        };
    }


    public static final int GET_LIMIT_LOGINS_MAX_SIZE = 1000;
    public static final int RAW_IMAGE_POOL_LENGTH = 10;
    public static final int AD_IMAGE_HASHES_MAX_VALUE = 10000;
    public static final int AD_IMAGE_TASKS_MAX_VALUE = 10000;
    public static final int CHECK_UPLOAD_STATUS_MAX_LIMIT = 10000;
    public static final int AD_IMAGE_GET_LOGIN_MAX_COUNT = 10;
    public static final int AD_IMAGE_GET_LIMIT_MAX_NUMBER = 10000;
    public static final int IMAGE_RAW_DATA_LIMIT = 15000000;

    public static String getDefaultImageUrl(int imgNumber, ImageType imageType) {
        // внешний доступ к картинкам через Zora
        if (imageType == ImageType.REGULAR) {
            // внешний доступ к картинкам через Zora
            if (imgNumber > validRegularImages.size()) {
                throw new DirectAPIException("Отсутствует URL для корректной картинки с номером " + imgNumber);
            }
            return validRegularImages.get(imgNumber);
        } else if (imageType == ImageType.WIDE) {
            // внешний доступ к картинкам через Zora
            if (imgNumber > validWideImages.size()) {
                throw new DirectAPIException("Отсутствует URL для корректной картинки с номером " + imgNumber);
            }
            return validWideImages.get(imgNumber);
        } else if (imageType == ImageType.SMALL) {
            throw new DirectAPIException("картинки типа SMALL не поддерживаются");
        }
        return null;
    }

    public static String getInvalidImageUrl(int imgNumber, ImageType imageType) {
        if (imageType == ImageType.SMALL) {
            // внешний доступ к картинкам через Zora
            if (imgNumber > urlInvalidSmallImages.size()) {
                throw new DirectAPIException("Отсутствует URL для некорректной картинки с номером " + imgNumber);
            }
            return urlInvalidSmallImages.get(imgNumber);
             /*return elliptics()
                .indefinitely()
                .path("direct/images/invalid/rate")
                .name(String.format("/img%s.jpg", imgNumber)).get().url();*/
        } else if ((imageType == ImageType.WIDE) || (imageType == ImageType.REGULAR)) {
            // внешний доступ к картинкам через Zora
            if (imgNumber > invalidBigImages.size()) {
                throw new DirectAPIException("Отсутствует URL для некорректной картинки с номером " + imgNumber);
            }
            return invalidBigImages.get(imgNumber);

        }
        return null;
    }

    //region  AdImageAssociation: Get/Set
    // http://wiki.yandex-team.ru/VasilijjBrjadov/API/AdImages#adimageassociation
    @Step("[AdImageAssociation]")
    public <AdImageAssociationResponse> AdImageAssociationResponse adImageAssociation(
            AdImageAssociationRequestMap adImageAssociationRequestMap)
    {
        return defaultClient().invokeMethod(Method.AD_IMAGE_ASSOCIATION, adImageAssociationRequestMap.getBean());
    }


    @Step("[AdImageAssociation.Get]")
    public <AdImageAssociationResponse> AdImageAssociationResponse getAdImageAssociation(
            AdImageAssociationSelectionCriteriaMap selectionCriteria)
    {
        return adImageAssociation(
                new AdImageAssociationRequestMap(defaultClient().getPackageName())
                        .withAction(Action.GET)
                        .withAdImageAssociationSelectionCriteria(selectionCriteria)
        );
    }

    @Step("[AdImageAssociation.Set]")
    public <AdImageAssociationResponse> AdImageAssociationResponse setAdImageAssociation(
            AdImageAssociationMap... adImageAssociations)
    {
        return adImageAssociation(
                new AdImageAssociationRequestMap(defaultClient().getPackageName())
                        .withAction(Action.SET)
                        .withAdImageAssociations(adImageAssociations)
        );
    }

    @Step("[AdImageAssociation.Set]. AdID = {0}, AdImageHash = {1}")
    public <AdImageAssociationResponse> AdImageAssociationResponse setAdImageAssociation(Long adID,
            String adImageHash)
    {
        AdImageAssociationMap associationMap =
                new AdImageAssociationMap(defaultClient().getPackageName())
                        .withAdID(adID)
                        .withAdImageHash(adImageHash);
        return setAdImageAssociation(associationMap);
    }

    @Step("[AdImageAssociation]. Привязать картинку {1} к объявлению {0}")
    public void assignImageToBannerIfNot(Long adID, String adImageHash) {
        Object image = getAdImage(adImageHash);
        AdImageMap adImageMap = new AdImageMap(image);
        if (!adImageMap.get(AdImageMap.ASSIGNED).equals(Value.YES)) {
            setAdImageAssociation(adID, adImageHash);
        }
    }

    //endregion

    //region AdImage: http://wiki.yandex-team.ru/VasilijjBrjadov/API/AdImages#adimage
    @Step("[AdImage]")
    public <AdImageResponse> AdImageResponse adImage(AdImageRequestMap adImageRequestMap) {
        return jsonClient().invokeMethod(Method.AD_IMAGE, adImageRequestMap.getBean());
    }

    //region GET
    @Step("[AdImage.Get]")
    public <AdImageResponse> AdImageResponse adImageGet(AdImageSelectionCriteriaMap adImageSelectionCriteriaMap) {
        return adImage(
                new AdImageRequestMap(defaultClient().getPackageName())
                        .withAction(Action.GET)
                        .withAdImageSelectionCriteria(adImageSelectionCriteriaMap));
    }

    public <AdImage> AdImage[] getAdImages(AdImageSelectionCriteriaMap adImageSelectionCriteriaMap) {
        Object response = adImageGet(adImageSelectionCriteriaMap);
        TestSteps.assumeThat("ответ не содержит ошибок",
                new BeanMap(response).get("actionsResult"), equalTo(null));
        return (AdImage[]) new BeanMap(response).get("adImages");
    }

    public <AdImage> AdImage getAdImage(String adImageHash, String login) {
        Object[] images = getAdImages(
                new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                        .withAdImageHashes(adImageHash)
                        .withLogins(login));
        TestSteps.assumeThat("в ответе одна картинка", images.length, lessThan(2));
        if (images.length == 1) {
            return (AdImage) images[0];
        } else {
            return null;
        }
    }

    public <AdImage> AdImage getAdImage(String adImageHash) {
        return getAdImage(adImageHash, null);
    }

    public String[] getAccountImageHashes(String login) {
        Object[] images = getAdImages(new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                .withLogins(login)
        );
        return extractAdImageHashesProperty(images);
    }

    public static String[] extractAdImageHashesProperty(Object[] images) {
        List<String> hashes = extractProperty(images, AdImageMap.AD_IMAGE_HASH);
        return hashes.toArray(new String[0]);
    }

    public static int[] extractAdImageUploadTaskIDS(Object[] adImageUploads) {
        List<Integer> tasks = extractProperty(adImageUploads, AdImageUploadMap.AD_IMAGE_UPLOAD_TASK_ID);
        return ArrayUtils.toPrimitive(tasks.toArray(new Integer[0]));
    }


    //endregion

    //region DELETE

    @Step("[AdImage.Delete]")
    public <AdImageResponse> AdImageResponse adImageDelete(AdImageSelectionCriteriaMap adImageSelectionCriteriaMap) {
        return adImage(
                new AdImageRequestMap(defaultClient().getPackageName())
                        .withAction(Action.DELETE)
                        .withAdImageSelectionCriteria(adImageSelectionCriteriaMap));
    }

    public <AdImageResponse> AdImageResponse adImageDelete(String... adImageHash) {
        return adImageDelete(
                new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                        .withAdImageHashes(adImageHash));
    }

    @Step("[AdImage.Delete]: Удалить все картинки для {0}")
    public void adImageDeleteForLogin(String login) {
        String[] hashes = getAccountImageHashes(login);
        if (hashes != null && hashes.length > 0) {
            Object adImageAssociationsResponse = getAdImageAssociation(
                    new AdImageAssociationSelectionCriteriaMap(defaultClient().getPackageName()).withLogins(login));
            Object[] adImageAssociations =
                    (Object[]) new BeanMap(adImageAssociationsResponse).get("adImageAssociations");

            for (Object association : adImageAssociations) {
                new AdImageAssociationMap(association).withAdImageHash(null);
            }
            List<AdImageAssociationMap> adImageAssociationMaps =
                    convert(adImageAssociations,
                            new BeanToBeanMapConverter<AdImageAssociationMap>
                                    (AdImageAssociationMap.class));
            setAdImageAssociation(adImageAssociationMaps.toArray(new AdImageAssociationMap[0]));
            adImageDelete(new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                    .withAdImageHashes(hashes).withLogins(login));
        }
    }
    //endregion

    //region CHECK UPLOAD STATUS
    @Step("[AdImage.CheckUploadStatus]")
    public <AdImageResponse> AdImageResponse adImageCheckUploadStatus(
            AdImageSelectionCriteriaMap adImageSelectionCriteriaMap)
    {
        return adImage(
                new AdImageRequestMap(defaultClient().getPackageName())
                        .withAction(Action.CHECK_UPLOAD_STATUS)
                        .withAdImageSelectionCriteria(adImageSelectionCriteriaMap));
    }

    public <AdImageUpload> AdImageUpload[] getAdImageUploads(AdImageSelectionCriteriaMap adImageSelectionCriteriaMap) {
        Object response = adImageCheckUploadStatus(adImageSelectionCriteriaMap);

        return (AdImageUpload[]) new BeanMap(response).get("adImageUploads");
    }

    public int[] getTaskIds(String login) {
        Object[] adImageUploads = getAdImageUploads(
                new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                        .withLogins(login)
        );
        return extractAdImageUploadTaskIDS(adImageUploads);
    }


    public <AdImageUpload> AdImageUpload getAdImageUpload(String login, int taskID) {
        Object[] result = getAdImageUploads(
                new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                        .withLogins(login)
                        .withAdImageUploadTaskIDS(taskID));
        TestSteps.assumeThat("верное число задач загрузки", result.length, equalTo(1));
        return (AdImageUpload) result[0];
    }

    public <AdImageUpload> AdImageUpload getAdImageUpload(int taskID) {
        return getAdImageUpload(null, taskID);
    }

    public <AdImageUpload> AdImageUpload getAdImageUpload(String hash) {
        Object[] result = getAdImageUploads(
                new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                        .withAdImageHashes(hash));
        TestSteps.assumeThat("верное число задач загрузки", result.length, equalTo(1));
        return (AdImageUpload) result[0];
    }

    public <AdImageUpload> AdImageUpload[] waitForImageHandleResult(int... taskIDs) {
        return waitForImageHandleResult(null, taskIDs);
    }

    public <AdImageUpload> AdImageUpload[] waitForImageHandleResult(String login, int... taskIDs) {
        if (taskIDs.length == 0) {
            throw new DirectAPIException("Empty taskIDs");
        }
        Object[] images = new Object[taskIDs.length];
        for (int i = 0; i < taskIDs.length; i++) {
            ConditionFactories.IMAGE.until(imageHandled(login, taskIDs[i]));
            images[i] = getAdImageUpload(login, taskIDs[i]);
        }
        return (AdImageUpload[]) images;
    }

    /**
     * An expectation for checking the status of image uploading
     *
     * @param uploadTaskID task for image handling
     * @return true если отчет forecastID готов, иначе (даже если его не существует) false
     */
    public <T> Callable<Boolean> imageHandled(final String login, final int uploadTaskID) {
        return new Callable<Boolean>() {
            public Boolean call() throws Exception {
                String status = (String)
                        new AdImageUploadMap((T) getAdImageUpload(login, uploadTaskID)).get(AdImageUploadMap.STATUS);
                return (status.equals(Status.DONE) || status.equals(Status.ERROR));
            }
        };
    }

    public Callable<Boolean> imageHandled(final int uploadTaskID) {
        return imageHandled(null, uploadTaskID);
    }
    //endregion

    //region UPLOAD (Url)
    @Step("[AdImage.Upload]")
    public <AdImageResponse> AdImageResponse adImageUpload(AdImageURLMap... adImageURLMap) {
        return adImage(
                new AdImageRequestMap(jsonClient().getPackageName())
                        .withAction(Action.UPLOAD)
                        .withAdImageURL(adImageURLMap));
    }

    /**
     * @param adImageURLMaps
     * @return AdImageUploadTaskIDs
     */
    public int[] putImageInUploadQueue(AdImageURLMap... adImageURLMaps) {
        Object response = adImageUpload(adImageURLMaps);
        TestSteps.assumeThat("отсутствуют ошибки в ответе", JsonUtils.toString(response), ApiResponse.hasNoError());
        return extractAdImageUploadTaskIDS(
                (Object[]) new BeanMap(response).get("actionsResult"));
    }

    public int putImageInUploadQueue(String name, String url, String login) {
        int[] taskIDs = putImageInUploadQueue(
                new AdImageURLMap(defaultClient().getPackageName())
                        .withName(name)
                        .withURL(url)
                        .withLogin(login)
        );
        TestSteps.assumeThat("число задач загрузки = 1", taskIDs.length, Matchers.equalTo(1));
        return taskIDs[0];
    }

    /**
     * @param name
     * @param url
     * @param login
     * @return AdImage
     */
    public <AdImage> AdImage uploadImage(String name, String url, String login) {
        Object[] adImageUploads = uploadImageAndWaitResultStatus(
                new AdImageURLMap(jsonClient().getPackageName())
                        .withName(name)
                        .withLogin(login)
                        .withURL(url)
        );
        return getAdImage((String) new AdImageUploadMap(adImageUploads[0]).get(AdImageUploadMap.AD_IMAGE_HASH));
    }

    public <AdImage> AdImage uploadImage(String url) {
        return uploadImage("моя картинка", url, null);
    }

    public <AdImage> AdImage uploadImage(String name, String url) {
        return uploadImage(name, url, null);
    }

    public <AdImageUpload> AdImageUpload[] uploadImageAndWaitResultStatus(AdImageURLMap... adImageURLMaps) {
        int[] taskIDs = putImageInUploadQueue(adImageURLMaps);
        Object[] adImageUploads = waitForImageHandleResult(taskIDs);
        for (Object adImageUpload : adImageUploads) {
            String adImageHash = (String) new BeanMap(adImageUpload).get(AdImageUploadMap.AD_IMAGE_HASH);
            String login = (String) new BeanMap(adImageUpload).get(AdImageUploadMap.LOGIN);
            //Если картинка успешно загрузилась
            if (adImageHash != null) {
                bin.throwToBin(new DefaultKeyValue(login, adImageHash));
            }
        }
        return (AdImageUpload[]) adImageUploads;
    }
    //endregion

    //region UploadRawData

    @Step("[AdImage.UploadRawData]")
    public <AdImageResponse> AdImageResponse adImageUploadRawData(AdImageRawMap... adImageRawMaps) {
        AdImageResponse response = adImage(
                new AdImageRequestMap(defaultClient().getPackageName())
                        .withAction(Action.UPLOAD_RAW_DATA)
                        .withAdImageRawData(adImageRawMaps));

        Object[] adImages = (Object[]) new BeanMap(response).get("actionsResult");
        for (Object adImage : adImages) {
            String adImageHash = (String) new BeanMap(adImage).get(AdImageMap.AD_IMAGE_HASH);
            String login = (String) new BeanMap(adImage).get(AdImageMap.LOGIN);
            if (login == null) {
                login = (String) adImageRawMaps[0].get(AdImageRawMap.LOGIN);
            }
            bin.throwToBin(new DefaultKeyValue(login, adImageHash));
        }

        return response;
    }

    @Step("Подготовка к тесту: очистка и загрузка картинок raw (тип {1})")
    public String[] configLoginImages(String login, ImageType imageType, int... imageNumbers) {
        return configLoginImages(login, imageType, true, imageNumbers);
    }


    @Step("Подготовка к тесту: очистка таблицы ({2}), загрузка картинок raw (тип {1})")
    public String[] configLoginImages(String login, ImageType imageType, Boolean clearImages, int... imageNumbers) {
        if (clearImages) {
            adImageDeleteForLogin(login);
        }
        AdImageRawMap[] adImageRawMaps = new AdImageRawMap[0];
        for (int i = 0; i < imageNumbers.length; i++) {
            if (imageType == ImageType.REGULAR) {
                AdImageRawMap rawMap = new AdImageRawMap(defaultClient().getPackageName())
                        .withLogin(login)
                        .withRawData(new String(ImageBase64Utils.getBase64EncodedFileData(
                                ImagesSteps.getDefaultImageUrl(imageNumbers[i], ImageType.REGULAR))))
                        .withName(ImagesSteps.nameValidRegularImages.get(imageNumbers[i]));
                adImageRawMaps = (AdImageRawMap[]) ArrayUtils.add(adImageRawMaps, rawMap);
            } else if (imageType == ImageType.WIDE) {
                AdImageRawMap rawMap = new AdImageRawMap(defaultClient().getPackageName())
                        .withLogin(login)
                        .withRawData(new String(ImageBase64Utils.getBase64EncodedFileData(
                                ImagesSteps.getDefaultImageUrl(imageNumbers[i], ImageType.WIDE))))
                        .withName(ImagesSteps.nameValidWideImages.get(imageNumbers[i]));
                adImageRawMaps = (AdImageRawMap[]) ArrayUtils.add(adImageRawMaps, rawMap);
            } else if (imageType == ImageType.SMALL) {
                throw new DirectAPIException("картинки типа SMALL не поддерживаются");
            }
        }
        Object response = adImage(
                new AdImageRequestMap(defaultClient().getPackageName())
                        .uploadRawData(adImageRawMaps));

        Object[] adImages = (Object[]) new BeanMap(response).get("actionsResult");
        for (Object adImage : adImages) {
            String adImageHash = (String) new BeanMap(adImage).get(AdImageMap.AD_IMAGE_HASH);
            bin.throwToBin(new DefaultKeyValue(login, adImageHash));
        }

        return extractAdImageHashesProperty(adImages);
    }


    @Step("Подготовка к тесту: загрузка картинок url")
    public int[] configLoginTasks(String login, int[] validImageNumbers, int[] invalidImageNumbers) {
        adImageDeleteForLogin(login);
        AdImageURLMap[] urls = new AdImageURLMap[0];
        for (int i = 0; i < validImageNumbers.length; i++) {
            AdImageURLMap urlMap = new AdImageURLMap(defaultClient().getPackageName())
                    .withLogin(login)
                    .withURL(ImagesSteps.getDefaultImageUrl(validImageNumbers[i], ImageType.REGULAR))
                    .withName(ImagesSteps.nameValidRegularImages.get(validImageNumbers[i]));
            urls = (AdImageURLMap[]) ArrayUtils.add(urls, urlMap);
        }
        for (int i = 0; i < invalidImageNumbers.length; i++) {
            AdImageURLMap urlMap = new AdImageURLMap(defaultClient().getPackageName())
                    .withLogin(login)
                    .withURL(ImagesSteps.getInvalidImageUrl(invalidImageNumbers[i], ImageType.REGULAR))
                    .withName(ImagesSteps.nameInvalidSmallImages.get(invalidImageNumbers[i]));
            urls = (AdImageURLMap[]) ArrayUtils.add(urls, urlMap);
        }
        Object[] adImageUploads = uploadImageAndWaitResultStatus(urls);
        return extractAdImageUploadTaskIDS(adImageUploads);
    }

    /**
     * @param name
     * @param url
     * @param login
     * @param <AdImageResponse>
     * @return AdImageResponse
     */
    @Step("[AdImage.UploadRawData]. Name = {0}, URL = {1}")
    public <AdImageResponse> AdImageResponse adImageUploadRawData(String name, String url, String login) {
        log.info("Грузим картинку с адреса: " + "[[a href=\"" + url + "\"]]");
        return adImageUploadRawData(new AdImageRawMap(defaultClient().getPackageName())
                .withName(name)
                .withLogin(login)
                .withRawData(new String(ImageBase64Utils.getBase64EncodedFileData(url))));
    }

    /**
     * @param url
     * @param <T>
     * @return AdImage
     */
    public <T> T uploadRawImage(String url) {
        return uploadRawImage("моя картинка", url, null);
    }

    /**
     * @param name
     * @param url
     * @param <AdImage>
     * @return AdImage
     */
    public <AdImage> AdImage uploadRawImage(String name, String url) {
        return uploadRawImage(name, url, null);
    }


    public <AdImage> AdImage uploadRawImage(String name, String url, String login) {
        Object adImageResponse = adImageUploadRawData(name, url, login);
        TestSteps.assumeThat("отсутствуют ошибки в ответе",
                JsonUtils.toString(adImageResponse), ApiResponse.hasNoError());
        Object[] actionResult = (Object[]) new BeanMap(adImageResponse).get("actionsResult");
        String adImageHash = (String) new BeanMap(actionResult[0]).get(AdImageUploadMap.AD_IMAGE_HASH);
        return getAdImage(adImageHash, login);
    }

    public String[] loadImages(int[] imgIndexes) {
        String[] hashes = new String[imgIndexes.length];
        for (int i = 0; i < imgIndexes.length; i++) {
            Object image = uploadRawImage(
                    "picture number " + imgIndexes[i],
                    getDefaultImageUrl(imgIndexes[i], ImageType.SMALL),
                    null
            );
            hashes[i] = (String) new BeanMap(image).get(AdImageUploadMap.AD_IMAGE_HASH);
        }
        return hashes;
    }

    //endregion

    //region GetLimits
    @Step("[AdImage.GetLimits]")
    public <AdImageResponse> AdImageResponse adImageGetLimits(AdImageSelectionCriteriaMap adImageSelectionCriteriaMap) {
        return adImage(
                new AdImageRequestMap(defaultClient().getPackageName())
                        .getLimits(adImageSelectionCriteriaMap));
    }


    public <AdImageResponse> AdImageResponse adImageGetLimits(String... logins) {
        return adImageGetLimits(
                new AdImageSelectionCriteriaMap(defaultClient().getPackageName())
                        .withLogins(logins)
        );
    }


    public <T> T getLimit(String login) {
        return (T) getLimits(login)[0];
    }

    public <T> T[] getLimits(String... logins) {
        Object adImageResponse = adImageGetLimits(logins);
        Object[] adImageLimits = (Object[]) new BeanMap(adImageResponse).get("adImageLimits");
        return (T[]) adImageLimits;
    }

    public String getDirectAdImageHref(String adImageHash) {
        return getDirectAdImageHref(adImageHash, null, null);
    }

    public String getDirectAdImageHref(String adImageHash, Long mdsGroupId, String namespace) {
        String path;
        if (mdsGroupId == null) {
            path = String.format("/images/%s", adImageHash);
        } else {
            path = String.format("/images/%s/%d/%s", namespace, mdsGroupId, adImageHash);
        }

        try {
            URI uri = new URIBuilder(connectionConfig.getEndPoint(ProtocolType.SOAP).replace("api.", ""))
                    .setPath(path)
                    .setPort(-1)
                    .setScheme("https")
                    .build();

            return uri.toString();
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка формирования урла для картинки", e);
        }
    }


    //endregion

    //endregion
}
