package ru.yandex.autotests.directapi.steps.clients;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

import javax.annotation.Nullable;

import com.hazelcast.core.HazelcastInstance;
import org.apache.commons.beanutils.BeanMap;
import org.apache.commons.lang3.StringUtils;
import org.hamcrest.Matcher;
import org.hamcrest.collection.IsArrayWithSize;

import ru.yandex.autotests.direct.utils.ReflectionUtils;
import ru.yandex.autotests.direct.utils.beans.BeanWrapper;
import ru.yandex.autotests.direct.utils.converter.BeanMapToBeanConverter;
import ru.yandex.autotests.direct.utils.money.Currency;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45mng.CreateNewSubclientResponse;
import ru.yandex.autotests.directapi.model.User;
import ru.yandex.autotests.directapi.model.clients.ClientFilterMap;
import ru.yandex.autotests.directapi.model.clients.ClientInfoMap;
import ru.yandex.autotests.directapi.model.clients.ClientInfoRequestMap;
import ru.yandex.autotests.directapi.model.clients.ClientRightMap;
import ru.yandex.autotests.directapi.model.clients.CreateNewSubclientRequestMap;
import ru.yandex.autotests.directapi.model.clients.GetSubClientsRequestMap;
import ru.yandex.autotests.directapi.model.clients.Role;
import ru.yandex.autotests.directapi.model.clients.SearchClientsRequestMap;
import ru.yandex.autotests.directapi.model.common.Value;
import ru.yandex.autotests.directapi.steps.BaseApiSteps;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.qatools.allure.annotations.Step;

import static ch.lambdaj.Lambda.convert;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNot.not;
import static ru.yandex.autotests.direct.db.models.jooq.ppc.Tables.USERS;
import static ru.yandex.autotests.directapi.model.Logins.DEFAULT_MANAGER_IDM_GROUP_ID;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;

/**
 * Created with IntelliJ IDEA.
 * User: mariabye
 * Date: 25.06.13
 * Time: 17:16
 * To change this template use File | Settings | File Templates.
 */
public class ClientSteps extends BaseApiSteps {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private static HazelcastInstance hz;
    private static ClientSteps _instance;

    public static final int SEARCH_CLIENTS_RESPONSE_MAX_ITEMS = 1000;

    private ClientSteps(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        super(connectionConfig, requestHeader);
    }

    public static ClientSteps getInstance(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        if (_instance == null) {
            _instance = new ClientSteps(connectionConfig, requestHeader);
        } else {
            _instance.setConnectionConfig(connectionConfig);
            _instance.setRequestHeader(requestHeader);
        }
        return _instance;
    }

    @Step("[GetClientInfo]: login = {0}")
    public <T> T getClientInfo(String login) {
        T[] clients = getClientInfo(new String[]{login});
        assumeThat("найден пользователь с логином " + login, clients, not(IsArrayWithSize.emptyArray()));
        return clients[0];
    }

    @Step("[GetClientInfo]")
    public <T> T[] getClientInfo(String... logins) {
        Object[] users = (T[]) defaultClient().invokeMethod(Method.GET_CLIENT_INFO, logins);
        return (T[]) users;
    }

    public Callable<Boolean> accountEnabled(final String login) {
        return new Callable<Boolean>() {
            public Boolean call() throws Exception {
                String shareAccountEnabled =
                        (String) new ClientInfoMap(
                                getClientInfo(login))
                                .get(ClientInfoMap.SHARED_ACCOUNT_ENABLED);
                return shareAccountEnabled.equals(Value.ENABLED);
            }
        };
    }


    public void waitForUnifiedAccountEnabled(String login) {
        ConditionFactories.CLIENT_UNIFIED_ACCOUNT.until(accountEnabled(login), equalTo(true));
    }

    public <T> void clientShouldHasUnifiedAccount(String login, Matcher<String> matcher) {
        String shareAccountEnabled =
                (String) new ClientInfoMap((T) getClientInfo(login)).get(ClientInfoMap.SHARED_ACCOUNT_ENABLED);
        TestSteps.assertThat("значение поля " + ClientInfoMap.SHARED_ACCOUNT_ENABLED + "- верное",
                shareAccountEnabled, matcher);
    }

    public String getClientRole(String login) {
        return (String) new BeanMap(getClientInfo(login)).get(ClientInfoMap.ROLE);
    }
    //endregion

    //region CreateNewSubclient
    @Step("[CreateNewSubclient]")
    public <T> T createNewSubClient(CreateNewSubclientRequestMap createNewSubclientRequestMap) {
        Object user = defaultClient().invokeMethod(
                Method.CREATE_NEW_SUBCLIENT,
                createNewSubclientRequestMap
                        .getBean());
        return (T) user;
    }


    @Step("[CreateNewSubclient]:  создать суб-клиента с login {0}<...> для агентства {1}")
    public <T> T createNewAgencySubClient(String base, String agencyLogin) {
        CreateNewSubclientRequestMap requestMap = new CreateNewSubclientRequestMap(connectionConfig.getPackage())
                .withRandomLogin(base)
                .withAgencyLogin(agencyLogin);
        return createNewSubClient(requestMap);
    }

    @Step("[CreateNewSubclient]:  создать валютного {2} суб-клиента с login {0}<...> для агентства {1}")
    public <T> T createNewAgencySubClient(String base, String agencyLogin, Currency currency) {
        CreateNewSubclientRequestMap requestMap = new CreateNewSubclientRequestMap(connectionConfig.getPackage())
                .withRandomLogin(base)
                .withAgencyLogin(agencyLogin)
                .withCurrency(currency);
        return createNewSubClient(requestMap);
    }

    public <T> T createNewServicedSubClient(String base, String mngrLogin, Currency currency) {
        if (currency != null) {
            return createNewServicedSubClient(base, mngrLogin, currency.toString());
        } else {
            return createNewServicedSubClient(base, mngrLogin, (String) null);
        }
    }

    public <CreateNewSubclientResponse> CreateNewSubclientResponse createNewServicedSubClient(
            String base, String mngrLogin)
    {
        return createNewServicedSubClient(base, mngrLogin, (String) null);
    }

    @Step("[CreateNewSubclient]:  создать сервисируемого валютного {2} клиента с login {0}<...> для менеджера {1}")
    public <T> T createNewServicedSubClient(String base, String mngrLogin, String currency) {
        CreateNewSubclientRequestMap requestMap = new CreateNewSubclientRequestMap(connectionConfig.getPackage())
                .withRandomLogin(base)
                .withServicedClient(Value.YES)
                .withCurrency(currency);
        T result = createNewSubClient(requestMap);

        String login = (String) new BeanMap(requestMap.getBean()).get(CreateNewSubclientRequestMap.LOGIN);
        setIdmPrimaryManagerForClient(login, mngrLogin);

        return result;
    }

    @Step("[setIdmPrimaryManagerForClient]: привязываем менеджера по схеме с IDM")
    public void setIdmPrimaryManagerForClient(String login, String mngrLogin) {
        Long clientId = getDirectJooqDbSteps().shardingSteps().getClientIdByLogin(login);
        // Даём дефолтной группе менеджеров доступ к созданному клиенту
        getDirectJooqDbSteps().useShardForClientId(clientId)
                .clientsSteps().addIdmGroupAccess(clientId, DEFAULT_MANAGER_IDM_GROUP_ID);

        // Устанавливаем для созданного клиента "главного менеджера"
        long managerUid = getDirectJooqDbSteps().useShardForLogin(mngrLogin)
                .usersSteps().getUser(mngrLogin).get(USERS.UID);
        getDirectJooqDbSteps().useShardForClientId(clientId)
                .clientsSteps().setIdmPrimaryManager((long) clientId, managerUid);
        return;
    }

    @Step("[CreateServicedClient]:  создать сервисируемого клиента с login {0}<...> для менеджера {1}")
    public <T> T createServicedClient(String base, String mngrLogin) {
        return createNewServicedSubClient(base, mngrLogin, (String) null);
    }

    @Step("[CreateServicedClient]:  создать сервисируемого клиента с login {0}<...> для менеджера {1} с валютой {2}")
    public <T> T createServicedClient(String base, String mngrLogin, String currency) {
        return createNewServicedSubClient(base, mngrLogin, currency);
    }

    //endregion

    //region GetClientsList
    @Step("[GetClientsList]")
    public <ClientInfo> ClientInfo[] getClientsList(ClientInfoRequestMap clientInfoRequestMap) {
        Object params = (clientInfoRequestMap != null) ? clientInfoRequestMap.getBean() : null;
        Object clients = defaultClient().invokeMethod(Method.GET_CLIENTS_LIST, params);
        return (ClientInfo[]) clients;
    }

    @Step("[GetClientsList]")
    public <ClientInfo> ClientInfo[] getClientsList() {
        return getClientsList((ClientInfoRequestMap) null);
    }

    @Step("[GetClientsList]")
    public <ClientInfo> ClientInfo[] getClientsList(String statusArch) {
        return getClientsList(new ClientInfoRequestMap(soapClient().getPackageName())
                .withFilter(new ClientFilterMap(soapClient().getPackageName()).withStatusArch(statusArch)));
    }

    @Step("[GetClientsList]")
    public <ClientInfo> ClientInfo[] getClientsListNotArch() {
        return getClientsList(Value.NO);
    }
    //endregion

    //region GetSubClients
    @Step("[GetSubClients]")
    public <ShortClientInfo> ShortClientInfo[] getSubClients(GetSubClientsRequestMap getSubClientsRequestMap) {
        Object clients = jsonClient().invokeMethod(Method.GET_SUB_CLIENTS, getSubClientsRequestMap.getBean());
        return (ShortClientInfo[]) clients;
    }

    @Step("[GetSubClients]: получить суб-клиентов агентства {0}")
    public <ShortClientInfo> ShortClientInfo[] getSubClients(String login) {
        return getSubClients(new GetSubClientsRequestMap(jsonClient().getPackageName()).withLogin(login));
    }

    @Step("[GetSubClients]")
    public <ShortClientInfo> ShortClientInfo[] getSubClients(String login, String statusArch) {
        return getSubClients(
                new GetSubClientsRequestMap(soapClient().getPackageName())
                        .withLogin(login)
                        .withFilter(new ClientFilterMap(soapClient().getPackageName()).withStatusArch(statusArch)));
    }

    @Step("[GetSubClients]")
    public <T> List<T> getSubClientsNotArch(String login) {
        List<T> onlyClients = new ArrayList();
        T[] allSubClients = getSubClients(login, Value.NO);
        String role;
        for (T client : allSubClients) {
            role = (String) ReflectionUtils.invokeGetter(client, StringUtils.capitalize(ClientInfoMap.ROLE));
            if (role.equals(Role.CHIEF_REP_SUB_CLIENT) || role.equals(Role.UNLIMITED_REP_CLIENT) ||
                    role.equals(Role.CHIEF_REP_CLIENT))
            {
                onlyClients.add(client);
            }

        }
        return onlyClients;
    }
    //endregion

    //region GetClientsUnits
    @Step("[GetClientsUnits]: login = {0}")
    public <ClientsUnitInfo> ClientsUnitInfo[] getClientsUnits(String... login) {
        return jsonClient().invokeMethod(Method.GET_CLIENTS_UNITS, login);
    }

    @Step("[GetClientsUnits]: login = {0}")
    public <ClientsUnitInfo> ClientsUnitInfo getClientUnits(String login) {
        Object[] clientsUnits = getClientsUnits(login);
        assumeThat("ответ содержит данные одного пользователя", clientsUnits.length, equalTo(1));
        return (ClientsUnitInfo) clientsUnits[0];
    }
    //endregion


    @Step(value = "Выполнение проверки для клиента: login:{0}, match {1}")
    public <T> void clientShould(String login, Matcher<BeanWrapper<T>> match) {
        T client = getClientInfo(login);
        TestSteps.assertThat("параметры клиента - верны", BeanWrapper.wrap(client), match);
    }

    //region UpdateClientInfo
    @Step("[UpdateClientInfo]")
    public void updateClientInfo(ClientInfoMap... clientInfoMaps) {
        jsonClient().invokeMethod(Method.UPDATE_CLIENT_INFO,
                convert(clientInfoMaps, new BeanMapToBeanConverter()).toArray(new Object[0]));
    }

    //region UpdateClientInfo
    @Step("Изменение прав клиента")
    public <T> void updateClientRights(String login, ClientRightMap... clientRightMaps) {
        ClientInfoMap clientInfo = new ClientInfoMap((T) getClientInfo(login));
        String phone = (String) clientInfo.get(ClientInfoMap.PHONE);
        //По дефолту в паспорте клиенты создаются без телефона
        //API требует указяния номера телефона
        if (phone == null || phone == "") {
            phone = "+7921750073";
        }
        updateClientInfo(clientInfo.withClientRights(clientRightMaps).withPhone(phone));
    }
    //endregion

    @Step("[SearchClients]")
    public <SearchClientsResponse> SearchClientsResponse searchClients(
            SearchClientsRequestMap searchClientsRequestMap)
    {
        Object response = jsonClient().invokeMethod(Method.SEARCH_CLIENTS, searchClientsRequestMap.getBean());
        return (SearchClientsResponse) response;
    }

    public Callable<Float> clientOverdraftIs(final String login) {
        return () -> (Float) new BeanMap(getClientInfo(login)).get(ClientInfoMap.OVERDRAFT_SUM_AVAILABLE_IN_CURRENCY);
    }

    public void reshardUserToExpectedShard(String login) {
        User user = User.get(login);
        assumeThat("пользователь найден в монге", user, notNullValue());
        getDarkSideSteps().getClientFakeSteps().reshardUser(login, user.getExpectedShard());
    }
}
