package ru.yandex.autotests.directapi.steps.forecast;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hamcrest.Description;
import org.hamcrest.TypeSafeMatcher;

import ru.yandex.autotests.directapi.common.api45mng.WordstatItem;
import ru.yandex.autotests.directapi.common.api45mng.WordstatReportInfo;

import static org.hamcrest.core.IsNull.notNullValue;
import static org.junit.Assume.assumeThat;

public class ApproximateWordstatReportMatcher extends TypeSafeMatcher<WordstatReportInfo> {

    private final WordstatReportInfo expectedBean;
    private final int showsPercent;
    private final int itemsPercent;
    private final boolean compareSearchedAlso;

    public ApproximateWordstatReportMatcher(
            WordstatReportInfo expectedBean, int showsPercent, int itemsPercent, boolean compareSearchedAlso) {
        assumeThat("Expected object cannot be null", expectedBean, notNullValue());
        this.expectedBean = expectedBean;
        this.showsPercent = showsPercent;
        this.itemsPercent = itemsPercent;
        this.compareSearchedAlso = compareSearchedAlso;
    }

    @Override
    protected boolean matchesSafely(WordstatReportInfo actual) {
        WordstatReportDiffer differ = new WordstatReportDiffer(
                expectedBean, actual, showsPercent, itemsPercent, compareSearchedAlso);
        return differ.isEquals();
    }

    @Override
    protected void describeMismatchSafely(WordstatReportInfo actual, Description mismatchDescription) {
        WordstatReportDiffer differ = new WordstatReportDiffer(
                expectedBean, actual, showsPercent, itemsPercent, compareSearchedAlso);
        differ.describeMismatchTo(mismatchDescription);
    }

    @Override
    public void describeTo(Description description) {
        description.appendValue(expectedBean);
    }

    private static class WordstatReportDiffer {
        private final WordstatReportInfo left;
        private final WordstatReportInfo right;
        private final WordstatItemsDiffer searchedWithDiffer;
        private final WordstatItemsDiffer searchedAlsoDiffer;
        private final boolean compareSearchedAlso;
        private final boolean geoApproximatelyEquals;
        private final boolean phrasesAreEquals;
        private final List<Integer> leftMissedGeos = new ArrayList<>();
        private final List<Integer> rightMissedGeos = new ArrayList<>();
        private int approximatelyEqualsGeosCount = 0;

        public WordstatReportDiffer(WordstatReportInfo left, WordstatReportInfo right,
                                    int showsPercent, int itemsPercent, boolean compareSearchedAlso) {
            this.left = left;
            this.right = right;
            this.searchedWithDiffer = new WordstatItemsDiffer(
                    left.getSearchedWith(), right.getSearchedWith(), showsPercent, itemsPercent);
            this.searchedAlsoDiffer = compareSearchedAlso ?
                    new WordstatItemsDiffer(left.getSearchedAlso(), right.getSearchedAlso(), showsPercent, itemsPercent)
                    :
                    null;
            this.compareSearchedAlso = compareSearchedAlso;
            phrasesAreEquals = (left.getPhrase() == null && right.getPhrase() == null) ||
                    (left.getPhrase() != null && left.getPhrase().equals(right.getPhrase()));
            int[] leftGeos = left.getGeoID();
            if (leftGeos == null) {
                leftGeos = new int[0];
            }
            int[] rightGeos = right.getGeoID();
            if (rightGeos == null) {
                rightGeos = new int[0];
            }
            compareGeos(leftGeos, rightGeos, itemsPercent);
            geoApproximatelyEquals = ApproximateCompareHelper.areIntsApproximatelyEqualWithFlowPercents(
                    approximatelyEqualsGeosCount, Math.max(leftGeos.length, rightGeos.length), itemsPercent);
        }

        private void compareGeos(int[] leftGeos, int[] rightGeos, int itemsPercent) {
            Set<Integer> leftSet = new HashSet<>(leftGeos.length);
            for (int geoId: leftGeos) {
                leftSet.add(geoId);
            }
            Set<Integer> rightSet = new HashSet<>(rightGeos.length);
            for (int geoId: rightGeos) {
                rightSet.add(geoId);
            }
            for (int geoId: leftGeos) {
                if (!rightSet.contains(geoId)) {
                    leftMissedGeos.add(geoId);
                } else {
                    approximatelyEqualsGeosCount++;
                }
            }
            for (int geoId: rightGeos) {
                if (!leftSet.contains(geoId)) {
                    rightMissedGeos.add(geoId);
                }
            }
        }

        void describeMismatchTo(Description description) {
            if (isEquals()) {
                return;
            }
            description.appendText(String.format("found following difference:%n"));
            if (!phrasesAreEquals) {
                description.appendText(String.format("expected phrase: %s but found actual phrase: %s%n",
                        left.getPhrase(), right.getPhrase()));
            }
            if (!geoApproximatelyEquals) {
                if (!leftMissedGeos.isEmpty()) {
                    description.appendText(String.format("missed expected geoIds in actual: %s:%n", leftMissedGeos));
                }
                if (!rightMissedGeos.isEmpty()) {
                    description.appendText(String.format("new actual gioIds not found in expected: %s:%n",
                            rightMissedGeos));
                }
            }
            if (!searchedWithDiffer.isEquals()) {
                description.appendText(String.format("found difference in searchedWith:%n"));
                searchedWithDiffer.describeMismatchTo(description);
            }
            if (compareSearchedAlso && !searchedAlsoDiffer.isEquals()) {
                description.appendText(String.format("found difference in searchedAlso:%n"));
                searchedAlsoDiffer.describeMismatchTo(description);
            }
        }

        boolean isEquals() {
            return phrasesAreEquals && geoApproximatelyEquals && searchedWithDiffer.isEquals() &&
                    (!compareSearchedAlso || searchedAlsoDiffer.isEquals());
        }
    }

    private static class WordstatItemsDiffer {
        private final List<WordstatItem> missedLeft = new ArrayList<>();
        private final List<WordstatItem> missedRight = new ArrayList<>();
        private final List<WordstatItemDiff> approximatelyEqualsItems = new ArrayList<>();
        private final List<WordstatItemDiff> approximatelyNotEqualsItems = new ArrayList<>();
        private final boolean itemsCountApproximatelyMatch;

        WordstatItemsDiffer(WordstatItem[] left, WordstatItem[] right, int showsPercent, int itemsPercent) {
            if (left == null) {
                left = new WordstatItem[0];
            }
            if (right == null) {
                right = new WordstatItem[0];
            }
            compareNonNullItems(left, right, showsPercent);
            itemsCountApproximatelyMatch = ApproximateCompareHelper.areIntsApproximatelyEqualWithFlowPercents(
                    approximatelyEqualsItems.size(), Math.max(left.length, right.length), itemsPercent);
        }

        private void compareNonNullItems(
                WordstatItem[] left, WordstatItem[] right, int showsPercent) {
            Map<String, Integer> leftMap = new HashMap<>(left.length);
            for (WordstatItem item : left) {
                leftMap.put(item.getPhrase(), item.getShows());
            }

            Map<String, Integer> rightMap = new HashMap<>(right.length);
            for (WordstatItem item : right) {
                rightMap.put(item.getPhrase(), item.getShows());
            }

            for (WordstatItem item : left) {
                String phrase = item.getPhrase();
                int leftShows = item.getShows();
                Integer rightShows = rightMap.get(phrase);
                if (rightShows == null) {
                    missedLeft.add(item);
                } else {
                    WordstatItemDiff diff = new WordstatItemDiff(phrase, leftShows, rightShows);
                    if (ApproximateCompareHelper.areIntsApproximatelyEqualWithFlowPercents(
                            leftShows, rightShows, showsPercent)) {
                        approximatelyEqualsItems.add(diff);
                    } else {
                        approximatelyNotEqualsItems.add(diff);
                    }
                }
            }
            for (WordstatItem item : right) {
                if (!leftMap.containsKey(item.getPhrase())) {
                    missedRight.add(item);
                }
            }
        }

        boolean isEquals() {
            return approximatelyNotEqualsItems.isEmpty() && itemsCountApproximatelyMatch;
        }

        void describeMismatchTo(Description description) {
            if (isEquals()) {
                return;
            }
            StringBuilder sbld = new StringBuilder();
            if (!itemsCountApproximatelyMatch) {
                if (!missedLeft.isEmpty()) {
                    sbld.append(String.format("\tmissed expected items in actual count: %d:%n", missedLeft.size()));
                    for (WordstatItem item: missedLeft) {
                        sbld.append(String.format("\t\tphrase: %s, shows: %d%n", item.getPhrase(), item.getShows()));
                    }
                    sbld.append('\n');
                }
                if (!missedRight.isEmpty()) {
                    sbld.append(String.format("\tnew actual items not found in expected count: %d:%n", missedRight.size()));
                    for (WordstatItem item: missedRight) {
                        sbld.append(String.format("\t\tphrase: %s, shows: %d%n", item.getPhrase(), item.getShows()));
                    }
                    sbld.append('\n');
                }
            }
            if (!approximatelyNotEqualsItems.isEmpty()) {
                sbld.append(String.format("\tfound non approximately equals items count: %d:%n",
                        approximatelyNotEqualsItems.size()));
                for (WordstatItemDiff diff: approximatelyNotEqualsItems) {
                    sbld.append(String.format("\t\tphrase: %s, expected shows: %d, actual shows: %d%n",
                            diff.phrase, diff.leftShows, diff.rightShows));
                }
                sbld.append('\n');
            }
            description.appendText(sbld.toString());
        }
    }

    private static class WordstatItemDiff {
        String phrase;
        int leftShows;
        int rightShows;

        public WordstatItemDiff(String phrase, int leftShows, int rightShows) {
            this.phrase = phrase;
            this.leftShows = leftShows;
            this.rightShows = rightShows;
        }
    }

}
