package ru.yandex.autotests.directapi.steps.forecast;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;

import ch.lambdaj.function.convert.PropertyExtractor;
import org.apache.commons.beanutils.BeanMap;
import org.apache.commons.collections.keyvalue.DefaultKeyValue;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.errors.AxisError;
import ru.yandex.autotests.directapi.apiclient.methods.Method;
import ru.yandex.autotests.directapi.common.api45mng.WordstatReportInfo;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.exceptions.TimeoutException;
import ru.yandex.autotests.directapi.model.forecast.ForecastInfoMap;
import ru.yandex.autotests.directapi.model.forecast.NewForecastInfoExtendedMap;
import ru.yandex.autotests.directapi.model.forecast.NewForecastInfoMap;
import ru.yandex.autotests.directapi.model.forecast.NewWordstatReportInfoMap;
import ru.yandex.autotests.directapi.model.statistics.StatusReport;
import ru.yandex.autotests.directapi.rules.Bin;
import ru.yandex.autotests.directapi.rules.Binable;
import ru.yandex.autotests.directapi.steps.BaseApiSteps;
import ru.yandex.autotests.directapi.steps.ConditionFactories;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.autotests.irt.testutils.json.JsonUtils;
import ru.yandex.qatools.allure.annotations.Step;

import static ch.lambdaj.Lambda.convert;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.lessThanOrEqualTo;
import static ru.yandex.autotests.irt.testutils.beandiffer.BeanDifferMatcher.beanEquivalent;

/**
 * Created with IntelliJ IDEA.
 * User: ginger
 * Date: 25.11.13
 * Time: 16:36
 * To change this template use File | Settings | File Templates.
 */
public class ForecastSteps extends BaseApiSteps implements Binable<DefaultKeyValue> {
    private LogSteps log = LogSteps.getLogger(this.getClass());
    private Logger log4j = LogManager.getLogger(this.getClass());

    public static final String FORECAST = "FORECAST";
    public static final String WORDSTAT = "WORDSTAT";
    public static final int NEW_FORECAST_REPORT_PHRASES_LIMIT = 100;
    public static final int NEW_WORDSTAT_REPORT_PHRASES_LIMIT = 10;
    public static final int NEW_WORDSTAT_REPORT_PHRASES_LENGTH_LIMIT = 4096;
    public static final int WORDSTAT_REPORT_QUEUE_LIMIT = 5;
    public static final int WORDSTAT_REPORT_CREATION_COST = 10;
    public static final int NEW_WORDSTAT_SYNC_REPORT_LENGTH_LIMIT = 30000;
    public static final int NEW_WORDSTAT_SYNC_REPORT_PHRASES_LIMIT = 100;
    public static final int NEW_FORECAST_SYNC_REPORT_PHRASES_LIMIT = 100;

    // Колличество элементов в отчете сформированном через CreateNewWordstatReport
    // почему-то advq возвращает на 1 айтем больше запрощенного
    public static final int WORDSTAT_REPORT_RESPONCE_PHRASES_LIMIT = 301;

    // количество элементов в ответе GetWordstatSync
    // почему-то advq возвращает на 1 айтем больше запрощенного
    public static final int WORDSTAT_SYNC_REPORT_RESPONCE_PHRASES_LIMIT = 601;

    private static ForecastSteps _instance;

    public Bin<DefaultKeyValue> bin = new Bin<>(this);


    private ForecastSteps(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        super(connectionConfig, requestHeader);
    }

    public static ForecastSteps getInstance(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        if (_instance == null) {
            _instance = new ForecastSteps(connectionConfig, requestHeader);
        } else {
            _instance.setConnectionConfig(connectionConfig);
            _instance.setRequestHeader(requestHeader);
        }
        return _instance;

    }

    public Callable clearBin(final Set<DefaultKeyValue> binData) {
        return new Callable() {
            public Object call() throws Exception {
                if (binData != null) {
                    for (DefaultKeyValue keyValue : binData) {
                        try {
                            if (keyValue.getValue().equals(FORECAST)) {
                                deleteForecast((Integer) keyValue.getKey());
                            } else {
                                deleteWordstatReport((Integer) keyValue.getKey());
                            }

                        } catch (AxisError axisFault) {
                            log4j.error(axisFault);
                        }
                    }
                }
                return null;
            }
        };
    }


    //region Forecast
    @Step("[CreateNewForecast]")
    public int createNewForecast(NewForecastInfoMap newForecastInfo) {
        int forecastId = defaultClient().invokeMethod(Method.CREATE_NEW_FORECAST, newForecastInfo.getBean());
        bin.throwToBin(new DefaultKeyValue(forecastId, FORECAST));
        return forecastId;
    }

    @Step("[GetForecastList]")
    public <T> T[] getForecastList() {
        return (T[]) defaultClient().invokeMethod(Method.GET_FORECAST_LIST, null);
    }

    @Step("[DeleteForecastReport]: {0}")
    public int deleteForecast(int forecastID) {
        bin.removeFromBin(new DefaultKeyValue(forecastID, FORECAST));

        return (Integer) defaultClient().invokeMethod(Method.DELETE_FORECAST_REPORT, forecastID);
    }

    @Step("[GetForecastStatus]: {0}")
    public String getForecastStatus(int forecastID) {
        Object forecast = getForecastFromList(forecastID);
        if (forecast != null) {
            return (String) new BeanMap(forecast).get(ForecastInfoMap.STATUS_FORECAST);
        } else {
            return "Not exist";
        }
    }

    public Callable<Boolean> forecastIsReady(final int forecastID) {
        return new Callable<Boolean>() {
            public Boolean call() throws Exception {
                return getForecastStatus(forecastID).equals(StatusReport.DONE);
            }
        };
    }

    @Step("Ожидание готовности отчета {0}")
    public <T> T getForecast(int forecastId) {
        ConditionFactories.FORECAST.until(forecastIsReady(forecastId));
        return (T) defaultClient().invokeMethod(Method.GET_FORECAST, forecastId);
    }

    public <T> T getForecastFromList(int forecastID) {
        Object forecasts = getForecastList();
        if (forecasts.equals(null)) {
            return null;
        }
        for (int i = 0; i < Array.getLength(forecasts); i++) {
            Object forecast = Array.get(forecasts, i);
            int currentForecastId = (Integer) (new BeanMap(forecast).get(ForecastInfoMap.FORECAST_ID));
            if (currentForecastId == forecastID) {
                return (T) forecast;
            }
        }
        return null;
    }

    public <T> T getForecast(NewForecastInfoMap newForecastInfo) {
        int forecastId = 0;
        try {
            forecastId = createNewForecast(newForecastInfo);
            return getForecast(forecastId);
        } catch (AxisError axisFault) {
            throw new DirectAPIException("Ошибка API при запросе отчета прогноза", axisFault);
        }
    }

    public <T> T getForecastSync(NewForecastInfoExtendedMap newForecastInfo) {
        return (T) defaultClient().invokeMethod(Method.GET_FORECAST_SYNC, newForecastInfo.getBean());
    }

    public List<String> getForecastReportPhrases(NewForecastInfoExtendedMap newForecastInfo) {
        Object reportData = getForecastSync(newForecastInfo);

        return convert(new BeanMap(reportData).get(NewForecastInfoMap.PHRASES),
                new PropertyExtractor<Object, String>("phrase"));
    }

    public void deleteForecastReports() {
        try {
            List<Integer> reportIDs = convert(getForecastList(), new PropertyExtractor("ForecastID"));
            for (Integer reportID : reportIDs) {
                deleteForecast(reportID);
            }
        } catch (AxisError axisFault) {
            throw new DirectAPIException("Ошибка удаления отчетов клиента");
        }

    }

    public void compareForecastToSyncReport(
            NewForecastInfoExtendedMap newForecastInfo, int percent) {
        NewForecastInfoMap newForecastInfoMap = new NewForecastInfoMap(connectionConfig.getPackage());
        newForecastInfoMap.putAllWriteable(newForecastInfo);
        Object asyncReportInfo = getForecast(newForecastInfoMap);
        Object syncReportInfo = getForecastSync(newForecastInfo);
        TestSteps.assertThat("отчеты, полученные синхронным и асинхронным методом совпадают",
                asyncReportInfo, beanEquivalent(syncReportInfo)
                        .withVariation(new ApproximateNumbersMatchVariation(percent, 0, true)));
    }

    //endregion

    //region Wordstat
    public int createNewWordstat(NewWordstatReportInfoMap newWordstatInfo) {
        int wordstatID = defaultClient().invokeMethod(Method.CREATE_NEW_WORDSTAT_REPORT, newWordstatInfo.getBean());
        bin.throwToBin(new DefaultKeyValue(wordstatID, WORDSTAT));
        return wordstatID;
    }

    public <T> T[] getWordstatReportList() {
        return (T[]) defaultClient().invokeMethod(Method.GET_WORDSTAT_REPORT_LIST, null);
    }

    @Step("[DeleteWordstatReport]: {0}")
    public int deleteWordstatReport(int wordstatReportID) {
        bin.removeFromBin(new DefaultKeyValue(wordstatReportID, WORDSTAT));
        return (Integer) defaultClient().invokeMethod(Method.DELETE_WORDSTAT_REPORT, wordstatReportID);
    }

    public String getWordstatReportStatus(int wordstatReportID) {
        Object wordstatReport = getWordstatReportFromList(wordstatReportID);
        if (wordstatReport != null) {
            return (String) new BeanMap(wordstatReport).get("statusReport");
        } else {
            return "Not exist";
        }
    }

    public Callable<Boolean> wordstatReportIsReady(final int wordstatReportID) {
        return new Callable<Boolean>() {
            public Boolean call() throws Exception {
                return getWordstatReportStatus(wordstatReportID).equals(StatusReport.DONE);
            }
        };
    }

    @Step("Ожидание готовности отчета {0}")
    public <WordstatReportInfo> WordstatReportInfo[] getWordstatReport(int wordstatReportID) throws TimeoutException {
        try {
            ConditionFactories.FORECAST.until(wordstatReportIsReady(wordstatReportID));
        } catch (Exception e) {
            throw new TimeoutException("Ошибка при ожидании формирования отчета wordstat: " +
                    "возможно, проблемы со скриптом " +
                    "./protected/apiWordstatReportDaemon.pl и " +
                    "./scripts/gearman_worker.pl --module API::Wordstat " +
                    "(на бете может быть вообще не запущен, тогда надо запустить)", e);
        }
        return (WordstatReportInfo[]) defaultClient().invokeMethod(Method.GET_WORDSTAT_REPORT, wordstatReportID);
    }

    public <WordstatReportInfo> WordstatReportInfo getWordstatReportFromList(int wordstatReportID) {
        Object wordstatReports = getWordstatReportList();
        if (wordstatReports.equals(null)) {
            return null;
        }
        for (int i = 0; i < Array.getLength(wordstatReports); i++) {
            Object wordstatReport = Array.get(wordstatReports, i);
            int currentWordstatReportId = (Integer) (new BeanMap(wordstatReport).get("reportID"));
            if (currentWordstatReportId == wordstatReportID) {
                return (WordstatReportInfo) wordstatReport;
            }
        }
        return null;
    }

    public <WordstatReportInfo> WordstatReportInfo[] getWordstatReport(
            NewWordstatReportInfoMap newWordstatReportInfoMap)
            throws TimeoutException, AxisError
    {
        int wordstatReportID = 0;
        try {
            wordstatReportID = createNewWordstat(newWordstatReportInfoMap);
            return getWordstatReport(wordstatReportID);
        } catch (AxisError axisFault) {
            throw new DirectAPIException("Ошибка API при работе с отчетом wordstat", axisFault);
        }
    }

    public void deleteWordstatReports() {
        try {
            List<Integer> reportIDs = convert(getWordstatReportList(), new PropertyExtractor("ReportID"));
            for (Integer reportID : reportIDs) {
                deleteWordstatReport(reportID);
            }
        } catch (AxisError axisFault) {
            throw new DirectAPIException("Ошибка удаления отчетов клиента");
        }

    }

    public List<String> getWordstatReportSearchedWithPhrases(NewWordstatReportInfoMap reportInfo) {
        Object[] reportData = (Object[]) getWordstatReport(reportInfo);
        TestSteps.assumeThat("получено верное число отчетов",
                reportData.length, equalTo(reportInfo.getPhrases().length));
        Object reportSearchedWith = new BeanMap(reportData[0]).get("searchedWith");
        return convert(reportSearchedWith, new PropertyExtractor<Object, String>("phrase"));
    }

    public List<String> getWordstatReportPhrases(NewWordstatReportInfoMap reportInfo) {
        Object[] reportData = (Object[]) getWordstatReport(reportInfo);
        TestSteps.assumeThat("получено верное число отчетов",
                reportData.length, equalTo(reportInfo.getPhrases().length));
        return convert(reportData, new PropertyExtractor<Object, String>("phrase"));
    }

    public <T> T getWordstatSyncReport(NewWordstatReportInfoMap reportInfo) {
        return (T) defaultClient().invokeMethod(Method.GET_WORDSTAT_SYNC, reportInfo.getBean());
    }

    public void compareWordstatToSyncReport(
            NewWordstatReportInfoMap reportInfo,
            int showsPercents, int itemsPercent, boolean compareAlsoFields) {
        // лимиты по размерку отчетов у GetWordstatSync и CreateNewWordstatReport
        // разные, поэтому если один из ответов превышает минимум из двух лимитом
        // то обрезаем его, см. https://st.yandex-team.ru/DIRECT-76787
        int maxReportSize = Math.min(WORDSTAT_REPORT_RESPONCE_PHRASES_LIMIT, WORDSTAT_SYNC_REPORT_RESPONCE_PHRASES_LIMIT);
        final WordstatReportInfo[] asyncResponse = getWordstatReport(reportInfo);
        final WordstatReportInfo[] syncResponse = getWordstatSyncReport(reportInfo);
        final WordstatReportInfo asyncReportInfo = asyncResponse[0];
        final WordstatReportInfo syncReportInfo = syncResponse[0];
        TestSteps.assertThat("GetWordstatReport SearchedWith не длиннее лимита в " + WORDSTAT_REPORT_RESPONCE_PHRASES_LIMIT + "объектов",
                asyncReportInfo.getSearchedWith().length, lessThanOrEqualTo(WORDSTAT_REPORT_RESPONCE_PHRASES_LIMIT));
        TestSteps.assertThat("GetWordstatSyncReport SearchedWith не длиннее лимита в " + WORDSTAT_SYNC_REPORT_RESPONCE_PHRASES_LIMIT +" объектов",
                asyncReportInfo.getSearchedWith().length, lessThanOrEqualTo(WORDSTAT_SYNC_REPORT_RESPONCE_PHRASES_LIMIT));

        if (asyncReportInfo.getSearchedWith().length > maxReportSize) {
            asyncReportInfo.setSearchedWith(Arrays.copyOf(asyncReportInfo.getSearchedWith(), maxReportSize));
        }

        if (syncReportInfo.getSearchedWith().length > maxReportSize) {
            syncReportInfo.setSearchedWith(Arrays.copyOf(syncReportInfo.getSearchedWith(), maxReportSize));
        }

        TestSteps.assertThat("отчеты, полученные синхронным и асинхронным методом совпадают",
                asyncReportInfo, beanEquivalent(syncReportInfo)
                        .withVariation(new ApproximateNumbersMatchVariation(
                                showsPercents, itemsPercent, compareAlsoFields)));
    }

    //endregion
}
