package ru.yandex.autotests.directapi.steps.statistics;

import ru.yandex.autotests.direct.utils.config.DirectTestRunProperties;
import ru.yandex.autotests.directapi.apiclient.RequestHeader;
import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.errors.Api5Error;
import ru.yandex.autotests.directapi.apiclient.version5.ServiceNames;
import ru.yandex.autotests.directapi.model.api5.reports.ReportDefinitionMap;
import ru.yandex.autotests.directapi.model.api5.reports.ReportProcessingMode;
import ru.yandex.autotests.directapi.model.api5.reports.ReportResponse;
import ru.yandex.autotests.directapi.model.api5.reports.ReportsData;
import ru.yandex.autotests.directapi.steps.BaseApiSteps;
import ru.yandex.qatools.allure.annotations.Step;

/**
 * Created by pavryabov on 01.06.16.
 */
public class ReportsSteps extends BaseApiSteps {

    private static final Boolean FORCE_OFFLINE_MODE;
    private static final Integer DEFAULT_OFFLINE_REPORT_TIMEOUT;
    private static ReportsSteps _instance;

    static {
        DirectTestRunProperties properties = DirectTestRunProperties.getInstance();
        FORCE_OFFLINE_MODE = properties.getDirectApi5ReportsForceOfflineMode();
        DEFAULT_OFFLINE_REPORT_TIMEOUT = properties.getDirectApi5ReportsOfflineModeTimeout();
    }

    class ReportGenerationOptions {
        final ReportProcessingMode processingMode;
        final Integer timeout;

        ReportGenerationOptions(ReportProcessingMode processingMode, Integer timeout) {
            if (FORCE_OFFLINE_MODE && (processingMode == null || processingMode == ReportProcessingMode.AUTO)) {
                this.processingMode = ReportProcessingMode.OFFLINE;
                this.timeout = DEFAULT_OFFLINE_REPORT_TIMEOUT;
            } else {
                this.processingMode = processingMode;
                this.timeout =
                        ((processingMode == null || processingMode == ReportProcessingMode.AUTO) && timeout == null)
                                ? DEFAULT_OFFLINE_REPORT_TIMEOUT : timeout;
            }
        }
    }

    private ReportsSteps(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        super(connectionConfig, requestHeader);
    }

    public static ReportsSteps getInstance(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        if (_instance == null) {
            _instance = new ReportsSteps(connectionConfig, requestHeader);
        } else {
            _instance.setConnectionConfig(connectionConfig);
            _instance.setRequestHeader(requestHeader);
        }
        return _instance;
    }

    @Step("[Reports]")
    public ReportsData callReportsXml(ReportDefinitionMap parameters, String login, ReportProcessingMode processingMode,
            Integer timeout)
    {
        ReportResponse reportResponse = callReportsXmlGetReportResponse(parameters, login, processingMode, timeout);
        return reportResponse.getReportsData(parameters.getFieldNames());
    }

    @Step("[Reports]")
    public void callReportsXmlWithoutResponse(ReportDefinitionMap parameters, String login,
            ReportProcessingMode processingMode, Integer timeout)
    {
        callReportsXmlGetReportResponse(parameters, login, processingMode, timeout);
    }

    public ReportsData callReportsXml(ReportDefinitionMap parameters) {
        return callReportsXml(parameters, null, null, null);
    }

    public ReportsData callReportsXml(ReportDefinitionMap parameters, String login) {
        return callReportsXml(parameters, login, null, null);
    }

    public ReportsData callOnlineReportsXml(ReportDefinitionMap parameters) {
        return callReportsXml(parameters, null, ReportProcessingMode.ONLINE, null);
    }

    public ReportsData callOfflineReportsXml(ReportDefinitionMap parameters) {
        return callReportsXml(parameters, null, ReportProcessingMode.OFFLINE, DEFAULT_OFFLINE_REPORT_TIMEOUT);
    }

    public ReportsData callOfflineReportsXmlForPreviouslyGeneratedReport(ReportDefinitionMap parameters) {
        return callReportsXml(parameters, null, ReportProcessingMode.OFFLINE, null);
    }

    public void callOfflineReportsXmlDoNotWaitForReady(ReportDefinitionMap parameters) {
        callReportsXmlWithoutResponse(parameters, null, ReportProcessingMode.OFFLINE, null);
    }

    public ReportsData callReportsJson(ReportDefinitionMap parameters, String login) {
        return callReportsJson(parameters, login, null, null);
    }

    @Step("[Reports]")
    public ReportsData callReportsJson(ReportDefinitionMap parameters, String login,
            ReportProcessingMode processingMode, Integer timeout)
    {
        ReportGenerationOptions generationOptions = new ReportGenerationOptions(processingMode, timeout);
        return callReportsJsonGetReportResponse(parameters, login, processingMode, timeout)
                .getReportsData(parameters.getFieldNames());
    }

    public ReportsData callReportsJson(ReportDefinitionMap parameters) {
        return callReportsJson(parameters, null, null, null);
    }

    public ReportsData callReportsJson(ReportDefinitionMap parameters, ReportProcessingMode processingMode) {
        return callReportsJson(parameters, null, processingMode, null);
    }

    public void expectXmlErrorOnReports(ReportDefinitionMap parameters, Api5Error api5Error) {
        expectXmlErrorOnReports(parameters, null, api5Error);
    }

    public void expectXmlErrorOnReports(Object parameters, Api5Error api5Error) {
        expectXmlErrorOnReports(parameters, null, api5Error);
    }

    public void expectXmlErrorOnReports(ReportDefinitionMap parameters, Api5Error api5Error,
            ReportProcessingMode processingMode)
    {
        expectXmlErrorOnReports(parameters, null, api5Error, processingMode, null);
    }

    public void expectXmlErrorOnReports(ReportDefinitionMap parameters, String login, Api5Error api5Error) {
        expectXmlErrorOnReports(parameters, login, api5Error, null, null);
    }

    public void expectXmlErrorOnReports(ReportDefinitionMap parameters, String login, Api5Error api5Error,
            ReportProcessingMode processingMode, Integer timeout)
    {
        ReportGenerationOptions generationOptions = new ReportGenerationOptions(processingMode, timeout);
        shouldGetReportsXmlErrorOn(login, parameters, api5Error, generationOptions.processingMode,
                generationOptions.timeout);
    }

    public void expectXmlErrorOnReports(Object parameters, String login, Api5Error api5Error) {
        shouldGetReportsXmlErrorOn(login, parameters, api5Error);
    }

    public void expectJsonErrorOnReports(ReportDefinitionMap parameters, String login, Api5Error api5Error) {
        shouldGetReportsJsonErrorOn(login, parameters, api5Error);
    }

    public void expectXmlErrorOnOfflineReports(ReportDefinitionMap parameters, String login, Api5Error api5Error) {
        expectXmlErrorOnReports(parameters, login, api5Error, ReportProcessingMode.OFFLINE, null);
    }

    public void expectXmlErrorOnOfflineReports(ReportDefinitionMap parameters, Api5Error api5Error) {
        expectXmlErrorOnOfflineReports(parameters, null, api5Error);
    }

    public void expectJsonErrorOnReports(ReportDefinitionMap parameters, Api5Error api5Error) {
        expectJsonErrorOnReports(parameters, null, api5Error);
    }

    public ReportResponse callReportsJsonGetReportResponse(ReportDefinitionMap parameters,
            ReportProcessingMode processingMode)
    {
        return callReportsJsonGetReportResponse(parameters, null, processingMode, null);
    }

    private ReportResponse callReportsJsonGetReportResponse(ReportDefinitionMap parameters, String login,
            ReportProcessingMode processingMode, Integer timeout)
    {
        ReportGenerationOptions generationOptions = new ReportGenerationOptions(processingMode, timeout);
        return reportsClientJson()
                .invokeMethod(ServiceNames.REPORTS, login, null, parameters.getBean(), generationOptions.processingMode,
                        generationOptions.timeout);
    }

    private ReportResponse callReportsXmlGetReportResponse(ReportDefinitionMap parameters, String login,
            ReportProcessingMode processingMode, Integer timeout)
    {
        ReportGenerationOptions generationOptions = new ReportGenerationOptions(processingMode, timeout);
        return reportsClientXml()
                .invokeMethod(ServiceNames.REPORTS, login, null, parameters.getBean(), generationOptions.processingMode,
                        generationOptions.timeout);
    }

    public ReportResponse callReportsXmlGetReportResponse(ReportDefinitionMap parameters) {
        return callReportsXmlGetReportResponse(parameters, null, null, null);
    }

    public ReportResponse callReportsXmlGetReportResponse(ReportDefinitionMap parameters,
            ReportProcessingMode reportProcessingMode)
    {
        return callReportsXmlGetReportResponse(parameters, null, reportProcessingMode, null);
    }

    public ReportResponse callReportsXmlGetOfflineReportResponse(ReportDefinitionMap parameters)
    {
        return callReportsXmlGetReportResponse(parameters, null, ReportProcessingMode.OFFLINE, null);
    }

}
