package ru.yandex.autotests.httpclient.lite.core.config;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.HttpContext;

import javax.net.ssl.SSLException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.UnknownHostException;

/**
 * @author Roman Kuhta (kuhtich@yandex-team.ru)
 */
public class HttpClientFactory {
    private static HttpRequestRetryHandler getRetryHandler() {
        return new HttpRequestRetryHandler() {

            public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
                if (executionCount >= 5) {
                    // Do not retry if over max retry count
                    return false;
                }
                if (exception instanceof InterruptedIOException) {
                    // Timeout
                    return false;
                }

                if (exception instanceof UnknownHostException) {
                    // Unknown host
                    return false;
                }

                if (exception instanceof SSLException) {
                    // SSL handshake exception
                    return false;
                }

                HttpClientContext clientContext = HttpClientContext.adapt(context);

                HttpRequest request = clientContext.getRequest();

                return !(request instanceof HttpEntityEnclosingRequest);
            }

        };
    }

    public static CloseableHttpClient getHttpClient(HttpClientConnectionConfig config) {
        SSLConnectionSocketFactory sslConnectionSocketFactory = new SSLConnectionSocketFactory(
                AllowAllSSLContext.get(),
                SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);

        ConnectionSocketFactory plainConnectionSocketFactory = PlainConnectionSocketFactory.getSocketFactory();

        Registry<ConnectionSocketFactory> r = RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", plainConnectionSocketFactory)
                .register("https", sslConnectionSocketFactory)
                .build();

        HttpClientConnectionManager cm = new PoolingHttpClientConnectionManager(r);

        HttpHost proxy = null;
        if (config.getProxyAddress() != null && config.getProxyPort() != null) {
            proxy = new HttpHost(config.getProxyAddress(), config.getProxyPort());
        }

        HttpClientBuilder httpClientBuilder = HttpClients.custom()
                .setConnectionManager(cm)
                .disableRedirectHandling()
                .setProxy(proxy)
                .setRetryHandler(getRetryHandler());
        if (config.getTimeout() > 0) {
            RequestConfig requestConfig = RequestConfig.custom()
                    .setConnectTimeout(config.getTimeout() * 1000)
                    .setConnectionRequestTimeout(config.getTimeout() * 1000)
                    .setSocketTimeout(config.getTimeout() * 1000).build();
            httpClientBuilder.setDefaultRequestConfig(requestConfig);
        }
        return httpClientBuilder
                .build();
    }
}
