package builders

import javaposse.jobdsl.dsl.helpers.scm.SvnDepth
import util.BuildUtils
import util.Constants

abstract class JobBuilder {
    protected static String CREDENTIALS_ID = 'robot-irt-jenkins-ssh'

    abstract Closure build(Map config);

    Closure defaultBuild(Map config) {
        return {
            logRotator {
                numToKeep(10)
            }

            wrappers {
                timestamps()
                colorizeOutput()
                sshAgent(CREDENTIALS_ID)
                credentialsBinding {
                    string('VAULT_TOKEN', 'VAULT_OAUTH')
                }
            }

            triggers {
                scm('H/2 * * * *')
            }

            label('direct-jenkins-autotests-slave || direct-jenkins-slave')

            jdk('java8')

            if(!config.enabled) {
                disabled()
            }

            description(config.repo)

            scm {
                svn {
                    location(config.repo + '@$REVISION') {
                        credentials(Constants.SVN_CREDENTIALS)
                        depth(SvnDepth.INFINITY)
                        directory(config.repoName)
                    }
                }
            }

            publishers {
                extendedEmail {
                    recipientList(config.email)

                    triggers {
                        contentType('text/html')
                        firstFailure {
                            sendTo {
                                developers()
                                requester()
                                culprits()
                                recipientList()
                            }
                        }
                        stillFailing {
                            sendTo {
                                developers()
                                requester()
                                culprits()
                                recipientList()
                            }
                        }
                        fixed {
                            sendTo {
                                developers()
                                requester()
                                culprits()
                                recipientList()
                            }
                        }
                    }
                }
                if(!config.skipPostBuild) {
                    groovyPostBuild('manager.addShortText("${manager.build.buildVariables.get(\'DEPLOY_VERSION\')}: ${manager.build.buildVariables.get(\'MVN_GOAL\')}")', Behavior.DoNothing)
                }
            }
        }
    }

    Closure params(Map config) {
        return {
            parameters {
                stringParam('DEPLOY_VERSION', config.version, '')
                stringParam('REVISION', 'HEAD')
                choiceParam('MVN_GOAL', ['deploy', 'compile', 'install'])
            }
        }
    }

    Closure downstream(Map config) {
        Closure base = {}
        if (config.downstream) {
            config.downstream.each {
                def downstreamConf = it
                base = base >> {
                    steps {
                        conditionalSteps {
                            condition {
                                stringsMatch('$DEPLOY_VERSION', config.version, false)
                            }
                            steps {
                                downstreamParameterized {
                                    trigger(downstreamConf.key) {
                                        parameters {
                                            predefinedProp('MVN_GOAL', downstreamConf.value?.goal ?: 'compile')
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        base
    }
}
