package ru.yandex.autotests.mediaplan.client;

import com.google.gson.JsonSyntaxException;
import freemarker.cache.ClassTemplateLoader;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.DefaultObjectWrapper;
import freemarker.template.Template;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.json.JSONObject;
import ru.yandex.autotests.httpclient.lite.core.config.HttpClientConnectionConfig;
import ru.yandex.autotests.httpclient.lite.core.config.HttpClientFactory;
import ru.yandex.autotests.irt.testutils.allure.LogSteps;
import ru.yandex.autotests.mediaplan.exceptions.DirectException;
import ru.yandex.qatools.allure.annotations.Attachment;

import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Changes by ginger on 17.02.16.
 */
public class JSONClient extends BaseClient{
    private LogSteps log = LogSteps.getLogger(this.getClass());

    @Override
    public <T> T invokeMethod(ServiceNames serviceName, Action methodName, Object params) {
        Class resultType = getReturnClass(serviceName, methodName);
        String response = sendRequest(serviceName, methodName, params);
        T result = (T) JSONBuilder.gson.fromJson(response, resultType);
        return result;
    }
    public JSONClient(ConnectionConfig connectionConfig) {
        super(connectionConfig);
    }

    public String sendRequest(ServiceNames serviceName, Action methodName, Object params) {
        JSONRequest request = new JSONRequest();
        request.setMethod(methodName.toString());
        request.setParameters(params);

        String endpoint = connectionConfig.getEndPointForV5Mediaplan(serviceName.toString());
        //custom object for logging
        HttpMessage message = new HttpMessage();
        message.setUrl(endpoint);
        message.setRequest(request.toString());

        String response;
        try {
            HttpClientConnectionConfig clientConnectionConfig =
                    ConnectionConfig.getHttpClientConnectionConfig(endpoint);
            CloseableHttpClient httpClient = HttpClientFactory.getHttpClient(clientConnectionConfig);

            HttpPost httpRequest = new HttpPost(endpoint);
            StringEntity entity = new StringEntity(request.toString(),
                    ContentType.create("application/json", Charset.forName("UTF-8")));
            httpRequest.setEntity(entity);
            response = httpClient.execute(httpRequest, new ResponseStringHandler(message));

            if (response == null) {
                throw new DirectException("Получен пустой ответ от сервера");
            }

            try {
                message.setResponse(new JSONObject(response).toString(2));
            } catch (JsonSyntaxException e) {
                throw new DirectException("Получен некорреткный ответ от сервера", e);
            }
            traceHttpMessage(message);
            response = getResponseResult(response);
        } catch (DirectException knownException) {
            traceHttpMessage(message);
            throw knownException;
        } catch (Exception e) {
            traceHttpMessage(message);
            throw new DirectException("Неизвестная ошибка при выполнении JSON запроса", e);
        }
        return response;
    }

    @Attachment(value = "[JSON/CURL]: Request", type = "text/html")
    public String traceHttpMessage(HttpMessage message) {
        log.info(message.toString());
        log.info(message.toCurlStringForJson());
        Map root = new HashMap();
        root.put("usualFormatString", message.toString());
        root.put("curlString", message.toCurlStringForJson());
        root.put("buttonName", "JSON");

        String attachmentHTML;
        try {
            TemplateLoader file = new ClassTemplateLoader(this.getClass(), "/");
            Configuration cfg = new Configuration();
            cfg.setTemplateLoader(file);
            cfg.setObjectWrapper(new DefaultObjectWrapper());
            cfg.setDefaultEncoding("UTF-8");
            Template template = cfg.getTemplate("request2curlTemplate.ftl");
            Writer fileWriter = new StringWriter();
            try {
                template.process(root, fileWriter);
            } finally {
                fileWriter.close();
            }
            attachmentHTML = fileWriter.toString();
        } catch (Exception e) {
            throw new RuntimeException("Error", e);
        }

        return attachmentHTML;
    }

    private Class getReturnClass(ServiceNames serviceName, Action methodName) {
        java.lang.reflect.Method method = null;
        java.lang.reflect.Method[] methods = serviceName.getMethodsClass().getDeclaredMethods();
        for (java.lang.reflect.Method m : methods) {
            if (m.getName().toLowerCase().replaceAll("_", "").equals(methodName.toString().toLowerCase().replaceAll("_", ""))) {
                method = m;
            }
        }
        if (method == null) {
            return String.class;
        } else {
            return method.getReturnType();
        }
    }

    /**
     * Возвращает содержимое структуры result {} или error {}, полученное в ответе сервиса
     * @param response ответ сервиса
     * @return содержимое структуры result{} или error{}, полученное в ответе сервиса в формате json
     */
    public String getResponseResult(String response) {
        String newResponse = response.replace(", \"error\": null", ""); //https://st.yandex-team.ru/MEDIAPLAN-139
        newResponse = newResponse.replace(":null", ":\"null\"");
        Pattern p = Pattern.compile("(result|error).*?:(.*)}");
        Matcher m = p.matcher(newResponse);
        if (m.find()) {
            return m.group(2);
        }
        throw new DirectException("Не удалось извлечь result или error из ответа:\n" + newResponse);
    }
}//
